#!/bin/bash

# ******************************************************************************** #
# Copyright Kbox Technologies Co., Ltd. 2023-2023. All rights reserved.
# File Name: kbox11_deploy.sh
# Description: openEuler kbox11环境部署脚本。
# Usage: 在使用imageTailor工具制作openEuler镜像时，该文件需放置在如下目录：
#        /opt/imageTailor/custom/cfg_openEuler/usr_file/home/kbox11_packages/deploy_scripts/openEuler_android11_deploy/
# ******************************************************************************** #

#set -x
# 脚本解释器 强制设置为 bash
if [ "$BASH" != "/bin/bash" ] && [ "$BASH" != "/usr/bin/bash" ]
then
   bash "$0" "$@"
   exit $?
fi

function error(){
    echo -e "\033[1;31m$1\033[0m"
    exit 1
}

# linux kernel版本
KERNEL_VERSION=$(uname -r)

# 本地依赖包目录
packages_dir="/home/kbox11_packages"

################################################################################
# Function Name: start_docker
# Description  : 启动docker服务。
# Parameter    : 
# Returns      : 0 on success, otherwise on fail
################################################################################
function start_docker(){
    echo "----- 启动docker服务"
    local lxcfs_status=$(systemctl status lxcfs.service | grep -w "inactive")
    [[ -z ${lxcfs_status} ]] && return 0
    systemctl start lxcfs > /dev/null 2>&1
    systemctl enable lxcfs > /dev/null 2>&1
    echo "docker服务已启动"
}

################################################################################
# Function Name: set_a32a64
# Description  : 转码使能。
# Parameter    : 
# Returns      : 0 on success, otherwise on fail
################################################################################
function set_a32a64(){
    echo "----- Exagear转码使能"
    if [ -e "/proc/sys/fs/binfmt_misc/ubt_a32a64" ]
    then
        < /proc/sys/fs/binfmt_misc/ubt_a32a64 grep "enabled" > /dev/null 2>&1
        [ $? -eq 0 ] && return 0
    fi
    [ -e "/proc/sys/fs/binfmt_misc/ubt_a32a64" ] && echo -1 > /proc/sys/fs/binfmt_misc/ubt_a32a64
    mount |grep "binfmt_misc on" > /dev/null 2>&1
    [ $? -ne 0 ] && mount -t binfmt_misc none /proc/sys/fs/binfmt_misc
    chmod -R 700 /opt/exagear
    chmod +x /opt/exagear/ubt_a32a64
    echo "配置Exagear转码"
    translator_name=ubt_a32a64
    echo ":${translator_name}:M::\x7fELF\x01\x01\x01\x00\x00\x00\x00\x00\x00\x00\x00\x00\x02\x00\x28\x00:\xff\xff\xff\xff\xff\xff\xff\x00\xff\xff\xff\xff\xff\xff\xff\xff\xfe\xff\xff\xff:/opt/exagear/ubt_a32a64:POCF" > /proc/sys/fs/binfmt_misc/register
    < /proc/sys/fs/binfmt_misc/ubt_a32a64 grep "enabled"
    [ $? -ne 0 ] && error "Exagear规则注册失败"
    echo "Exagear转码已开启"
}

################################################################################
# Function Name: insmod_ashmem_binder
# Description  : 安装ashmem、binder两个内核模块。
# Parameter    : 
# Returns      : 0 on success, otherwise on fail
################################################################################
function insmod_ashmem_binder(){
    local count_time=0
    local binder_num
    echo "----- 安装ashmem binder模块"
    cd /lib/modules/${KERNEL_VERSION}/kernel/lib/ || exit
    lsmod |grep -w aosp_binder_linux
    if [ $? -ne 0 ]
    then
        echo "安装binder模块"
        [ -e "aosp_binder_linux.ko" ] || cp ${packages_dir}/ashmem_binder_ko/aosp_binder_linux.ko .
        insmod aosp_binder_linux.ko num_devices=400
        lsmod |grep -w aosp_binder_linux
        [ $? -ne 0 ] && error "binder安装失败"
    fi
    cd /lib/modules/${KERNEL_VERSION}/kernel/lib/ || exit
    lsmod |grep -w ashmem_linux
    if [ $? -ne 0 ]
    then
        echo "安装ashmem模块"
        [ -e "ashmem_linux.ko" ] || cp ${packages_dir}/ashmem_binder_ko/ashmem_linux.ko .
        insmod ashmem_linux.ko
        lsmod |grep -w ashmem_linux
        [ $? -ne 0 ] && error "ashmem安装失败"
    fi
    while true
    do
        sleep 1
        binder_num=$(ls /dev/|grep -c "^aosp_binder[0-9]\{1,3\}$")
        [ "${binder_num}" -eq 400 ] && break
        if [ ${count_time} -gt 15 ]
        then
            echo -e "\033[1;31m insmod aosp_binder failed\033[0m"
            break
        fi
        (( count_time++ )) || true
    done
    echo "配置dev可执行权限"
    chmod 600 /dev/aosp_binder*
    chmod 600 /dev/ashmem
    chmod 600 /dev/dri/*
    chmod 600 /dev/input
}

################################################################################
# Function Name: import_android_image
# Description  : 导入android镜像到docker。
# Parameter    : 
# Returns      : 0 on success, otherwise on fail
################################################################################
function import_android_image(){
    cd ${packages_dir} || exit
    docker images|awk '{print $1" "$2}'|grep -w "kbox"|grep -w "origin" > /dev/null 2>&1
    if [ $? -ne 0 ]
    then
        if [ -f "android.tar" ]
        then
            echo "----- 正在导入镜像"
            docker import android.tar kbox:origin || exit
        fi
    fi
}

################################################################################
# Function Name: start_kbox_default
# Description  : 尝试启动一个kbox11，判断环境是否已经配置好。
# Parameter    : 
# Returns      : 0 on success, otherwise on fail
################################################################################
function start_kbox_default(){
    echo "----- 启动Kbox"
    local kbox_name
    cd ${packages_dir}/deploy_scripts || exit
    chmod +x android11_kbox.sh
    docker ps -a|awk '{print $(NF-1)" "$NF}'|grep -E -w "8501|kbox_1" > /dev/null 2>&1
    [ $? -eq 0 ] && return 0

    ./android11_kbox.sh start kbox:origin 1 > /dev/null 2>&1

    kbox_name=kbox_1
    local cmd="docker exec -i ${kbox_name} getprop sys.boot_completed | grep 1 &"
    local result=$(bash ${packages_dir}/deploy_scripts/base_box.sh wait_async_cmd "${cmd}")
    if [ "${result}" == "1" ]
    then
        echo "${kbox_name} 已启动, 环境部署完毕。"
    elif [ "${result}" == "-1" ]
    then
        echo "${kbox_name} 启动状态检验超时"
    else
        echo "${kbox_name} 启动失败, 请重新检查环境。"
    fi
}

main(){
    start_docker
    set_a32a64
    insmod_ashmem_binder
    import_android_image
    start_kbox_default
}

main "$@"
exit 0
