#!/bin/bash
# build external vulkan driver.
# Copyright © Huawei Technologies Co., Ltd. 2022-2022. All rights reserved.

source environment

build_zlib_arm() {
    cd ${SRC_PATH}/zlib
    set_env_arm
    chmod +x configure

    ./configure \
    --static \
    --prefix=${INSTALL_PATH} \
    --libdir=${INSTALL_PATH}/lib
    make install
    clean_env
    cd ${SRC_PATH}
}

build_zlib_aarch64() {
    cd ${SRC_PATH}/zlib
    set_env_aarch64
    chmod +x configure

    ./configure \
    --static \
    --prefix=${INSTALL_PATH} \
    --libdir=${INSTALL_PATH}/lib64
    make install
    clean_env
    cd ${SRC_PATH}
}

build_expat_arm() {
    cd ${SRC_PATH}/expat/expat
    set_env_arm

    ./buildconf.sh
    ./configure \
    --host=arm-linux-android \
    --enable-shared=false \
    --prefix=${INSTALL_PATH} \
    --libdir=${INSTALL_PATH}/lib
    make install
    clean_env
    cd ${SRC_PATH}
}

build_expat_aarch64() {
    cd ${SRC_PATH}/expat/expat
    set_env_aarch64

    ./buildconf.sh
    ./configure \
    --host=aarch64-linux-android \
    --enable-shared=false \
    --prefix=${INSTALL_PATH} \
    --libdir=${INSTALL_PATH}/lib64
    make install
    clean_env
    cd ${SRC_PATH}
}

build_libelf_arm() {
    cd ${SRC_PATH}/elfutils
    if [ -d "build-android-arm" ];then
        rm -rf build-android-arm
    fi
    mkdir build-android-arm
    cd build-android-arm

    ${CMAKE_BIN} .. \
    -GNinja \
    -DCMAKE_BUILD_TYPE=Release \
    -DCMAKE_TOOLCHAIN_FILE=${NDK_PATH}/build/cmake/android.toolchain.cmake \
    -DANDROID_ABI=armeabi-v7a \
    -DANDROID_NDK=${NDK_PATH} \
    -DANDROID_PLATFORM=android-28 \
    -DCMAKE_ANDROID_ARCH_ABI=armeabi-v7a \
    -DCMAKE_EXPORT_COMPILE_COMMANDS=ON \
    -DCMAKE_SYSTEM_NAME=Android \
    -DCMAKE_SYSTEM_VERSION=28 \
    -DCMAKE_INSTALL_PREFIX=${INSTALL_PATH} \
    -DCMAKE_INSTALL_LIBDIR=lib

    ninja install
    cd ${SRC_PATH}
}

build_libelf_aarch64() {
    cd ${SRC_PATH}/elfutils
    if [ -d "build-android-aarch64" ];then
        rm -rf build-android-aarch64
    fi
    mkdir build-android-aarch64
    cd build-android-aarch64

    ${CMAKE_BIN} .. \
    -GNinja \
    -DCMAKE_BUILD_TYPE=Release \
    -DCMAKE_TOOLCHAIN_FILE=${NDK_PATH}/build/cmake/android.toolchain.cmake \
    -DANDROID_ABI=arm64-v8a \
    -DANDROID_NDK=${NDK_PATH} \
    -DANDROID_PLATFORM=android-28 \
    -DCMAKE_ANDROID_ARCH_ABI=arm64-v8a \
    -DCMAKE_EXPORT_COMPILE_COMMANDS=ON \
    -DCMAKE_SYSTEM_NAME=Android \
    -DCMAKE_SYSTEM_VERSION=28 \
    -DCMAKE_INSTALL_PREFIX=${INSTALL_PATH} \
    -DCMAKE_INSTALL_LIBDIR=lib64

    ninja install
    cd ${SRC_PATH}
}

build_llvm_aarch64() {
    cd ${SRC_PATH}/llvm
    #先编译host64位llvm库，以获取64位llvm-config
    if [ -d "build64" ];then
        rm -rf build64
    fi
    mkdir build64
    cd build64
    ${CMAKE_BIN} .. \
    -GNinja \
    -DCMAKE_BUILD_TYPE=Release \
    -DCMAKE_EXPORT_COMPILE_COMMANDS=ON \
    -DLLVM_BUILD_LLVM_DYLIB=True \
    -DLLVM_ENABLE_PIC=True \
    -DLLVM_ENABLE_RTTI=True \
    -DLLVM_TARGETS_TO_BUILD="AMDGPU;AArch64;ARM" \
    -DCMAKE_INSTALL_PREFIX=${INSTALL_PATH} \
    -DLLVM_LIBDIR_SUFFIX=64
    ninja
    cd ..

    if [ -d "build-android-aarch64" ];then
        rm -rf build-android-aarch64
    fi
    mkdir build-android-aarch64
    cd build-android-aarch64
    ${CMAKE_BIN} .. \
    -GNinja \
    -DCMAKE_BUILD_TYPE=Release \
    -DCMAKE_TOOLCHAIN_FILE=${NDK_PATH}/build/cmake/android.toolchain.cmake \
    -DANDROID_ABI=arm64-v8a \
    -DANDROID_NDK=${NDK_PATH} \
    -DANDROID_PLATFORM=android-28 \
    -DCMAKE_ANDROID_ARCH_ABI=arm64-v8a \
    -DCMAKE_EXPORT_COMPILE_COMMANDS=ON \
    -DCMAKE_SYSTEM_NAME=Android \
    -DCMAKE_SYSTEM_VERSION=28 \
    -DLLVM_TARGET_ARCH=AArch64 \
    -DLLVM_TARGETS_TO_BUILD="AMDGPU;AArch64;ARM" \
    -DLLVM_DEFAULT_TARGET_TRIPLE=aarch64-linux-android \
    -DCMAKE_INSTALL_PREFIX=${INSTALL_PATH} \
    -DLLVM_TABLEGEN=../build64/bin/llvm-tblgen \
    -DLLVM_BUILD_LLVM_DYLIB=True \
    -DLLVM_LIBDIR_SUFFIX=64 \
    -DLLVM_ENABLE_PIC=False \
    -DLLVM_ENABLE_RTTI=True
    ninja install
    ${NDK_PATH}/toolchains/llvm/prebuilt/linux-x86_64/bin/llvm-strip -s ${INSTALL_PATH}/lib64/libLLVM-12.so
    cp ${INSTALL_PATH}/lib64/libLLVM-12.so ../build64/lib64/
    cd ${SRC_PATH}
}

build_llvm_arm() {
    cd ${SRC_PATH}/llvm
    #先编译host32位llvm库，以获取32位llvm-config
    if [ -d "build" ];then
        rm -rf build
    fi
    mkdir build
    cd build
    ${CMAKE_BIN} .. \
    -GNinja \
    -DCMAKE_BUILD_TYPE=Release \
    -DCMAKE_EXPORT_COMPILE_COMMANDS=ON \
    -DLLVM_BUILD_LLVM_DYLIB=True \
    -DLLVM_ENABLE_PIC=True \
    -DLLVM_ENABLE_RTTI=True \
    -DLLVM_TARGETS_TO_BUILD="AMDGPU;AArch64;ARM" \
    -DCMAKE_INSTALL_PREFIX=${INSTALL_PATH}
    ninja
    cd ..

    if [ -d "build-android-arm" ];then
        rm -rf build-android-arm
    fi
    mkdir build-android-arm
    cd build-android-arm
    ${CMAKE_BIN} .. \
    -GNinja \
    -DCMAKE_BUILD_TYPE=Release \
    -DCMAKE_TOOLCHAIN_FILE=${NDK_PATH}/build/cmake/android.toolchain.cmake \
    -DANDROID_ABI=armeabi-v7a \
    -DANDROID_NDK=${NDK_PATH} \
    -DANDROID_PLATFORM=android-28 \
    -DCMAKE_ANDROID_ARCH_ABI=armeabi-v7a \
    -DCMAKE_EXPORT_COMPILE_COMMANDS=ON \
    -DCMAKE_SYSTEM_NAME=Android \
    -DCMAKE_SYSTEM_VERSION=28 \
    -DLLVM_TARGET_ARCH=arm \
    -DLLVM_TARGETS_TO_BUILD="AMDGPU;AArch64;ARM" \
    -DLLVM_DEFAULT_TARGET_TRIPLE=arm-linux-android \
    -DCMAKE_INSTALL_PREFIX=${INSTALL_PATH} \
    -DLLVM_TABLEGEN=../build/bin/llvm-tblgen \
    -DLLVM_BUILD_LLVM_DYLIB=True \
    -DLLVM_ENABLE_PIC=False \
    -DLLVM_ENABLE_RTTI=True
    ninja install
    ${NDK_PATH}/toolchains/llvm/prebuilt/linux-x86_64/bin/llvm-strip -s ${INSTALL_PATH}/lib/libLLVM-12.so
    cp ${INSTALL_PATH}/lib/libLLVM-12.so ../build/lib/
    cd ${SRC_PATH}
}

build_drm_arm() {
    cd ${SRC_PATH}/drm-libdrm
    if [ -d "build-android-arm" ];then
        rm -rf build-android-arm
    fi

    ${MESON} \
    build-android-arm \
    --cross-file=${CROSS_PATH}/android-arm \
    --prefix=${INSTALL_PATH} \
    --libdir=lib \
    -Dnouveau=false \
    -Dfreedreno=false

    ninja -C build-android-arm install
    cd ${SRC_PATH}
}

build_drm_aarch64() {
    cd ${SRC_PATH}/drm-libdrm
    if [ -d "build-android-aarch64" ];then
        rm -rf build-android-aarch64
    fi

    ${MESON} \
    build-android-aarch64 \
    --cross-file=${CROSS_PATH}/android-aarch64 \
    --prefix=${INSTALL_PATH} \
    --libdir=lib64 \
    -Dnouveau=false \
    -Dfreedreno=false

    ninja -C build-android-aarch64 install
    cd ${SRC_PATH}
}

build_libva_arm() {
    cd ${SRC_PATH}/libva
    if [ -d "build-android-arm" ];then
        rm -rf build-android-arm
    fi

    ${MESON} \
    build-android-arm \
    --cross-file=${CROSS_PATH}/android-arm \
    --prefix=${INSTALL_PATH} \
    --libdir=lib \
    -Ddriverdir=/system/vendor/lib/dri

    ninja -C build-android-arm install
    cd ${SRC_PATH}
}

build_libva_aarch64() {
    cd ${SRC_PATH}/libva
    if [ -d "build-android-aarch64" ];then
        rm -rf build-android-aarch64
    fi

    ${MESON} \
    build-android-aarch64 \
    --cross-file=${CROSS_PATH}/android-aarch64 \
    --prefix=${INSTALL_PATH} \
    --libdir=lib64 \
    -Ddriverdir=/system/vendor/lib64/dri

    ninja -C build-android-aarch64 install
    cd ${SRC_PATH}
}

build_mesa_arm() {
    cd ${SRC_PATH}/mesa
    if [ -d "build-android-arm" ];then
        rm -rf build-android-arm
    fi

    ${MESON} \
    build-android-arm \
    --cross-file=${CROSS_PATH}/android-arm \
    --prefix=${INSTALL_PATH} \
    --libdir=lib \
    -Dplatforms=android \
    -Dplatform-sdk-version=28 \
    -Dandroid-stub=true \
    -Degl-native-platform=android \
    -Dplatform-sdk-version=28 \
    -Dgallium-drivers=radeonsi \
    -Degl=enabled \
    -Dgbm=enabled \
    -Ddri-search-path="/system/vendor/lib/dri" \
    -Dgallium-va=enabled \
    -Dvulkan-drivers=amd

    ninja -C build-android-arm install
    cd ${SRC_PATH}
}

build_mesa_aarch64() {
    cd ${SRC_PATH}/mesa
    if [ -d "build-android-aarch64" ];then
        rm -rf build-android-aarch64
    fi

    ${MESON} \
    build-android-aarch64 \
    --cross-file=${CROSS_PATH}/android-aarch64 \
    --prefix=${INSTALL_PATH} \
    --libdir=lib64 \
    -Dplatforms=android \
    -Dplatform-sdk-version=28 \
    -Dandroid-stub=true \
    -Degl-native-platform=android \
    -Dplatform-sdk-version=28 \
    -Dgallium-drivers=radeonsi \
    -Degl=enabled \
    -Dgbm=enabled \
    -Ddri-search-path="/system/vendor/lib64/dri" \
    -Dgallium-va=enabled \
    -Dvulkan-drivers=amd

    ninja -C build-android-aarch64 install
    cd ${SRC_PATH}
}

clean() {
    if [ -d "${INSTALL_PATH}" ];then
        rm -rf ${INSTALL_PATH}/*
    fi
}

build() {
    clean
    gen_meson_cross_file

    build_zlib_aarch64
    build_expat_aarch64
    build_libelf_aarch64
    build_llvm_aarch64
    build_drm_aarch64
    build_libva_aarch64
    build_mesa_aarch64

    build_zlib_arm
    build_expat_arm
    build_libelf_arm
    build_llvm_arm
    build_drm_arm
    build_libva_arm
    build_mesa_arm
}

build

if [ -d ${OUTPUT_PATH} ]; then
    rm -r ${OUTPUT_PATH}/*
fi

mkdir -p ${OUTPUT_PATH}/system/lib
mkdir -p ${OUTPUT_PATH}/system/lib64
mkdir -p ${OUTPUT_PATH}/system/vendor/lib/egl
mkdir -p ${OUTPUT_PATH}/system/vendor/lib64/egl
mkdir -p ${OUTPUT_PATH}/system/vendor/lib/dri
mkdir -p ${OUTPUT_PATH}/system/vendor/lib64/dri
mkdir -p ${OUTPUT_PATH}/system/vendor/lib/hw
mkdir -p ${OUTPUT_PATH}/system/vendor/lib64/hw

#32bit
cp ${INSTALL_PATH}/lib/libdrm.so  ${OUTPUT_PATH}/system/lib/
cp ${INSTALL_PATH}/lib/libdrm_amdgpu.so  ${OUTPUT_PATH}/system/vendor/lib/libdrm_amdgpu.so.1
cp ${INSTALL_PATH}/lib/libdrm_radeon.so  ${OUTPUT_PATH}/system/vendor/lib/libdrm_radeon.so.1
cp ${INSTALL_PATH}/lib/libLLVM-12.so  ${OUTPUT_PATH}/system/vendor/lib/libLLVM-12.so

cp ${INSTALL_PATH}/lib/libva-drm.so.2.1400.0 ${OUTPUT_PATH}/system/vendor/lib/libva-drm.so.2
cp ${INSTALL_PATH}/lib/libva.so.2.1400.0 ${OUTPUT_PATH}/system/vendor/lib/libva.so.2
cp ${INSTALL_PATH}/lib/libva-drm.so ${OUTPUT_PATH}/system/vendor/lib/
cp ${INSTALL_PATH}/lib/libva.so ${OUTPUT_PATH}/system/vendor/lib/
cp ${INSTALL_PATH}/lib/libva-drm.so.2.1400.0 ${OUTPUT_PATH}/system/lib/libva-drm.so.2
cp ${INSTALL_PATH}/lib/libva.so.2.1400.0 ${OUTPUT_PATH}/system/lib/libva.so.2
cp ${INSTALL_PATH}/lib/libva-drm.so ${OUTPUT_PATH}/system/lib/
cp ${INSTALL_PATH}/lib/libva.so ${OUTPUT_PATH}/system/lib/

cp ${INSTALL_PATH}/lib/libglapi.so.0.0.0  ${OUTPUT_PATH}/system/vendor/lib/libglapi.so.0
cp ${INSTALL_PATH}/lib/libgbm.so.1.0.0 ${OUTPUT_PATH}/system/vendor/lib/libgbm.so
cp ${INSTALL_PATH}/lib/libvulkan_radeon.so  ${OUTPUT_PATH}/system/vendor/lib/hw/vulkan.radv.so
cp ${INSTALL_PATH}/lib/libGLESv1_CM.so.1.1.0  ${OUTPUT_PATH}/system/vendor/lib/egl/libGLESv1_CM_mesa2.so
cp ${INSTALL_PATH}/lib/libGLESv2.so.2.0.0  ${OUTPUT_PATH}/system/vendor/lib/egl/libGLESv2_mesa2.so
cp ${INSTALL_PATH}/lib/libEGL.so.1.0.0  ${OUTPUT_PATH}/system/vendor/lib/egl/libEGL_mesa2.so
cp ${INSTALL_PATH}/lib/dri/radeonsi_dri.so  ${OUTPUT_PATH}/system/vendor/lib/dri/radeonsi_dri.so
cp ${INSTALL_PATH}/lib/dri/radeonsi_drv_video.so  ${OUTPUT_PATH}/system/vendor/lib/dri/radeonsi_drv_video.so

#64bit
cp ${INSTALL_PATH}/lib64/libdrm.so  ${OUTPUT_PATH}/system/lib64/
cp ${INSTALL_PATH}/lib64/libdrm_amdgpu.so  ${OUTPUT_PATH}/system/vendor/lib64/libdrm_amdgpu.so.1
cp ${INSTALL_PATH}/lib64/libdrm_radeon.so  ${OUTPUT_PATH}/system/vendor/lib64/libdrm_radeon.so.1
cp ${INSTALL_PATH}/lib64/libLLVM-12.so  ${OUTPUT_PATH}/system/vendor/lib64/libLLVM-12.so

cp ${INSTALL_PATH}/lib64/libva-drm.so.2.1400.0 ${OUTPUT_PATH}/system/vendor/lib64/libva-drm.so.2
cp ${INSTALL_PATH}/lib64/libva.so.2.1400.0 ${OUTPUT_PATH}/system/vendor/lib64/libva.so.2
cp ${INSTALL_PATH}/lib64/libva-drm.so ${OUTPUT_PATH}/system/vendor/lib64/
cp ${INSTALL_PATH}/lib64/libva.so ${OUTPUT_PATH}/system/vendor/lib64/
cp ${INSTALL_PATH}/lib64/libva-drm.so.2.1400.0 ${OUTPUT_PATH}/system/lib64/libva-drm.so.2
cp ${INSTALL_PATH}/lib64/libva.so.2.1400.0 ${OUTPUT_PATH}/system/lib64/libva.so.2
cp ${INSTALL_PATH}/lib64/libva-drm.so ${OUTPUT_PATH}/system/lib64/
cp ${INSTALL_PATH}/lib64/libva.so ${OUTPUT_PATH}/system/lib64/

cp ${INSTALL_PATH}/lib64/libglapi.so.0.0.0  ${OUTPUT_PATH}/system/vendor/lib64/libglapi.so.0
cp ${INSTALL_PATH}/lib64/libgbm.so.1.0.0 ${OUTPUT_PATH}/system/vendor/lib64/libgbm.so
cp ${INSTALL_PATH}/lib64/libvulkan_radeon.so  ${OUTPUT_PATH}/system/vendor/lib64/hw/vulkan.radv.so
cp ${INSTALL_PATH}/lib64/libGLESv1_CM.so.1.1.0  ${OUTPUT_PATH}/system/vendor/lib64/egl/libGLESv1_CM_mesa2.so
cp ${INSTALL_PATH}/lib64/libGLESv2.so.2.0.0  ${OUTPUT_PATH}/system/vendor/lib64/egl/libGLESv2_mesa2.so
cp ${INSTALL_PATH}/lib64/libEGL.so.1.0.0  ${OUTPUT_PATH}/system/vendor/lib64/egl/libEGL_mesa2.so
cp ${INSTALL_PATH}/lib64/dri/radeonsi_dri.so  ${OUTPUT_PATH}/system/vendor/lib64/dri/radeonsi_dri.so
cp ${INSTALL_PATH}/lib64/dri/radeonsi_drv_video.so  ${OUTPUT_PATH}/system/vendor/lib64/dri/radeonsi_drv_video.so

cd ${OUTPUT_PATH}
tar -zcvf vulkan.tar.gz *