# Interfaces

## interface DHParameters

```cangjie
public interface DHParameters <: Key {
    override func encodeToPem(): PemEntry
    static func decodeDer(blob: DerBlob): DHParameters
    static func decodeFromPem(text: String): DHParameters
}
```

Function: Provides the DH parameters interface.

Parent Types:

- [Key](#interface-key)

### static func decodeDer(DerBlob)

```cangjie
static func decodeDer(blob: DerBlob): DHParameters
```

Function: Decodes DH key parameters from DER format.

> **Note:**
>
> - The DH (Diffie-Hellman) key exchange protocol is a method to ensure secure shared KEY transmission over insecure networks.
> - DER and PEM are two common encoding formats.

Parameters:

- blob: [DerBlob](x509_package_structs.md#struct-derblob) - DER-formatted DH key parameters object.

Return Value:

- [DHParameters](x509_package_interfaces.md#interface-dhparameters) - DH key parameters decoded from DER format.

Exceptions:

- [X509Exception](./x509_package_exceptions.md#class-x509exception) - Thrown when the DER-formatted DH key parameters content is incorrect and cannot be parsed.

### static func decodeFromPem(String)

```cangjie
static func decodeFromPem(text: String): DHParameters
```

Function: Decodes DH key parameters from PEM format.

> **Note:**
>
> PEM is a certificate encoded in ASCII (BASE64).

Parameters:

- text: String - PEM-formatted DH key parameters character stream.

Return Value:

- [DHParameters](x509_package_interfaces.md#interface-dhparameters) - DH key parameters decoded from PEM format.

Exceptions:

- [X509Exception](./x509_package_exceptions.md#class-x509exception) - Thrown when the character stream does not conform to PEM format or when the file header does not match the DH key parameters header standard ("-----BEGIN DH PARAMETERS-----").

### func encodeToPem()

```cangjie
override func encodeToPem(): PemEntry
```

Function: Encodes DH key parameters into PEM format.

Return Value:

- [PemEntry](x509_package_structs.md#struct-pementry) - Object generated by PEM format encoding of DH key parameters data.

## interface Key

```cangjie
public interface Key <: ToString {
    func encodeToDer(): DerBlob
    func encodeToPem(): PemEntry
    static func decodeDer(encoded: DerBlob): Key
    static func decodeFromPem(text: String): Key
}
```

Function: Provides the key interface. Public keys are used for signature verification or encryption, while private keys are used for signing or decryption. Public and private keys must match each other and form a pair. This class is an abstract key class with no concrete implementation, serving as an extension interface for [PrivateKey](x509_package_interfaces.md#interface-privatekey)/[PublicKey](x509_package_interfaces.md#interface-publickey) and user extensions.

Parent Types:

- ToString

### static func decodeDer(DerBlob)

```cangjie
static func decodeDer(encoded: DerBlob): Key
```

Function: Decodes a key from DER format.

Parameters:

- encoded: [DerBlob](x509_package_structs.md#struct-derblob) - DER-formatted object.

Return Value:

- [Key](x509_package_interfaces.md#interface-key) - Key decoded from DER format.

Exceptions:

- [X509Exception](./x509_package_exceptions.md#class-x509exception) - Thrown when the DER-formatted private key content is incorrect and cannot be parsed.

### static func decodeFromPem(String)

```cangjie
static func decodeFromPem(text: String): Key
```

Function: Decodes a key from PEM format.

Parameters:

- text: String - PEM-formatted character stream.

Return Value:

- [Key](x509_package_interfaces.md#interface-key) - Key decoded from PEM format.

### func encodeToDer()

```cangjie
func encodeToDer(): DerBlob
```

Function: Encodes a key into DER format.

Return Value:

- [DerBlob](x509_package_structs.md#struct-derblob) - Object generated by DER format encoding of key data.

### func encodeToPem()

```cangjie
func encodeToPem(): PemEntry
```

Function: Encodes a key into PEM format.

Return Value:

- [PemEntry](x509_package_structs.md#struct-pementry) - Object generated by PEM format encoding of key data.

## interface PrivateKey

```cangjie
public interface PrivateKey <: Key {
    static func decodeDer(blob: DerBlob): PrivateKey
    static func decodeFromPem(text: String): PrivateKey
    static func decodeDer(blob: DerBlob, password!: ?String): PrivateKey
    static func decodeFromPem(text: String, password!: ?String): PrivateKey
    func encodeToDer(password!: ?String): DerBlob
    override func encodeToPem(): PemEntry
    func encodeToPem(password!: ?String): PemEntry
}
```

Function: Provides the private key interface.

Parent Types:

- [Key](#interface-key)

### static func decodeDer(DerBlob)

```cangjie
static func decodeDer(blob: DerBlob): PrivateKey
```

Function: Decodes a private key from DER format.

Parameters:

- blob: [DerBlob](x509_package_structs.md#struct-derblob) - DER-formatted private key object.

Return Value:

- [PrivateKey](x509_package_interfaces.md#interface-privatekey) - Private key decoded from DER format.

Exceptions:

- [X509Exception](./x509_package_exceptions.md#class-x509exception) - Thrown when the DER-formatted private key content is incorrect and cannot be parsed.

### static func decodeDer(DerBlob, ?String)

```cangjie
static func decodeDer(blob: DerBlob, password!: ?String): PrivateKey 
```

Function: Decrypts and decodes a DER-formatted private key into a [PrivateKey](x509_package_interfaces.md#interface-privatekey) object. If the password is None, no decryption is performed.

Parameters:

- blob: [DerBlob](x509_package_structs.md#struct-derblob) - DER-formatted private key.
- password!: ?String - Decryption password.

Return Value:

- [PrivateKey](x509_package_interfaces.md#interface-privatekey) - Decrypted and decoded private key object.

Exceptions:

- [X509Exception](./x509_package_exceptions.md#class-x509exception) - Thrown when decryption/decoding fails or when `password` is an empty string.

### static func decodeFromPem(String)

```cangjie
static func decodeFromPem(text: String): PrivateKey
```

Function: Decodes a private key from PEM format.

Parameters:

- text: String - PEM-formatted private key character stream.

Return Value:

- [PrivateKey](x509_package_interfaces.md#interface-privatekey) - Private key decoded from PEM format.

Exceptions:

- [X509Exception](./x509_package_exceptions.md#class-x509exception) - Thrown when the character stream does not conform to PEM format or when the file header does not match the public key header standard.

### static func decodeFromPem(String, ?String)

```cangjie
static func decodeFromPem(text: String, password!: ?String): PrivateKey 
```

Function: Decrypts and decodes a PEM-formatted private key into a [PrivateKey](x509_package_interfaces.md#interface-privatekey) object. If the password is None, no decryption is performed.

Parameters:

- text: String - PEM-formatted private key.
- password!: ?String - Decryption password.

Return Value:

- [PrivateKey](x509_package_interfaces.md#interface-privatekey) - Decrypted and decoded private key object.

Exceptions:

- [X509Exception](./x509_package_exceptions.md#class-x509exception) - Thrown when decryption/decoding fails or when `password` is an empty string.

### func encodeToDer(?String)

```cangjie
func encodeToDer(password!: ?String): DerBlob
```

Function: Encrypts and encodes a private key into DER format. If the password is None, no encryption is performed.

Parameters:

- password!: ?String - Encryption password.

Return Value:

- [DerBlob](x509_package_structs.md#struct-derblob) - Encrypted DER-formatted private key.

Exceptions:

- [X509Exception](./x509_package_exceptions.md#class-x509exception) - Thrown when encryption fails or when `password` is an empty string.

### func encodeToPem()

```cangjie
override func encodeToPem(): PemEntry
```

Function: Encodes a private key into PEM format.

Return Value:

- [PemEntry](x509_package_structs.md#struct-pementry) - Encoded PEM-formatted private key.

Exceptions:

- [X509Exception](./x509_package_exceptions.md#class-x509exception) - Thrown when encoding fails.

### func encodeToPem(?String)

```cangjie
func encodeToPem(password!: ?String): PemEntry
```

Function: Encrypts and encodes a private key into PEM format. No encryption is applied when the password is None.

Parameters:

- password!: ?String - The encryption password.

Return Value:

- [PemEntry](x509_package_structs.md#struct-pementry) - The encrypted private key in PEM format.

Exceptions:

- [X509Exception](./x509_package_exceptions.md#class-x509exception) - Encryption failed, or `password` is an empty string.

## interface PublicKey

```cangjie
public interface PublicKey <: Key {
    override func encodeToPem(): PemEntry
    static func decodeDer(blob: DerBlob): PublicKey
    static func decodeFromPem(text: String): PublicKey
}
```

Function: Public key interface.

Parent Type:

- [Key](#interface-key)

### static func decodeDer(DerBlob)

```cangjie
static func decodeDer(blob: DerBlob): PublicKey
```

Function: Decodes a public key from DER format.

Parameters:

- blob: [DerBlob](x509_package_structs.md#struct-derblob) - The public key object in DER format.

Return Value:

- [PublicKey](x509_package_interfaces.md#interface-publickey) - The public key decoded from DER format.

Exceptions:

- [X509Exception](./x509_package_exceptions.md#class-x509exception) - Thrown when the DER-formatted public key content is incorrect and cannot be parsed.

### static func decodeFromPem(String)

```cangjie
static func decodeFromPem(text: String): PublicKey
```

Function: Decodes a public key from PEM format.

Parameters:

- text: String - The character stream of the public key in PEM format.

Return Value:

- [PublicKey](x509_package_interfaces.md#interface-publickey) - The public key decoded from PEM format.

Exceptions:

- [X509Exception](./x509_package_exceptions.md#class-x509exception) - Thrown when the character stream does not conform to PEM format or when the file header does not meet the public key header standard.

### func encodeToPem()

```cangjie
override func encodeToPem(): PemEntry
```

Function: Encodes a public key into PEM format.

Return Value:

- [PemEntry](x509_package_structs.md#struct-pementry) - The object generated by encoding the public key data into PEM format.