/********************************************************************************
 * Copyright (c) 2002, 2007 IBM Corporation and others. All rights reserved.
 * This program and the accompanying materials are made available under the terms
 * of the Eclipse Public License v1.0 which accompanies this distribution, and is 
 * available at http://www.eclipse.org/legal/epl-v10.html
 * 
 * Initial Contributors:
 * The following IBM employees contributed to the Remote System Explorer
 * component that contains this file: David McKnight, Kushal Munir, 
 * Michael Berger, David Dykstal, Phil Coulthard, Don Yantzi, Eric Simpson, 
 * Emily Bruner, Mazen Faraj, Adrian Storisteanu, Li Ding, and Kent Hawley.
 * 
 * Contributors:
 * Martin Oberhuber (Wind River) - [168975] Move RSE Events API to Core
 * Martin Oberhuber (Wind River) - [177523] Unify singleton getter methods
 * Martin Oberhuber (Wind River) - [186773] split ISystemRegistryUI from ISystemRegistry
 ********************************************************************************/

package org.eclipsecon.tmtutorial.multishell;

import java.util.ArrayList;
import java.util.List;

import org.eclipse.core.runtime.IAdaptable;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.NullProgressMonitor;
import org.eclipse.core.runtime.Status;
import org.eclipse.core.runtime.jobs.Job;
import org.eclipse.jface.action.Action;
import org.eclipse.jface.action.IAction;
import org.eclipse.jface.action.IContributionItem;
import org.eclipse.jface.action.IStatusLineManager;
import org.eclipse.jface.action.IToolBarManager;
import org.eclipse.jface.action.MenuManager;
import org.eclipse.jface.resource.ImageDescriptor;
import org.eclipse.jface.text.ITextOperationTarget;
import org.eclipse.jface.viewers.CheckboxTableViewer;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.ISelectionChangedListener;
import org.eclipse.jface.viewers.SelectionChangedEvent;
import org.eclipse.jface.viewers.StructuredSelection;
import org.eclipse.jface.viewers.Viewer;
import org.eclipse.rse.core.RSECorePlugin;
import org.eclipse.rse.core.events.ISystemResourceChangeEvent;
import org.eclipse.rse.core.events.ISystemResourceChangeEvents;
import org.eclipse.rse.core.events.ISystemResourceChangeListener;
import org.eclipse.rse.core.model.IHost;
import org.eclipse.rse.core.model.ISystemRegistry;
import org.eclipse.rse.core.subsystems.ISubSystem;
import org.eclipse.rse.internal.shells.ui.ShellResources;
import org.eclipse.rse.internal.shells.ui.actions.SystemBaseShellAction;
import org.eclipse.rse.internal.ui.SystemResources;
import org.eclipse.rse.services.clientserver.messages.SystemMessage;
import org.eclipse.rse.shells.ui.RemoteCommandHelpers;
import org.eclipse.rse.shells.ui.view.CommandEntryViewerConfiguration;
import org.eclipse.rse.shells.ui.view.SystemCommandEditor;
import org.eclipse.rse.shells.ui.view.SystemViewRemoteOutputAdapter;
import org.eclipse.rse.subsystems.shells.core.subsystems.IRemoteCmdSubSystem;
import org.eclipse.rse.subsystems.shells.core.subsystems.IRemoteCommandShell;
import org.eclipse.rse.ui.ISystemIconConstants;
import org.eclipse.rse.ui.RSEUIPlugin;
import org.eclipse.rse.ui.SystemWidgetHelpers;
import org.eclipse.rse.ui.messages.ISystemMessageLine;
import org.eclipse.rse.ui.model.ISystemShellProvider;
import org.eclipse.rse.ui.view.IRSEViewPart;
import org.eclipse.rse.ui.view.ISystemViewElementAdapter;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.events.SelectionListener;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Event;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Listener;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.swt.widgets.Table;
import org.eclipse.swt.widgets.TableItem;
import org.eclipse.swt.widgets.Widget;
import org.eclipse.ui.IActionBars;
import org.eclipse.ui.ISelectionListener;
import org.eclipse.ui.ISelectionService;
import org.eclipse.ui.IWorkbenchPart;
import org.eclipse.ui.part.CellEditorActionHandler;
import org.eclipse.ui.part.ViewPart;



/**
 * This is the desktop view wrapper of the System View viewer.
 */
public class SystemMultiShellViewPart
	extends ViewPart
	implements
		ISelectionListener,
		SelectionListener,
		ISelectionChangedListener,
		ISystemResourceChangeListener,
		ISystemShellProvider,
		IRSEViewPart,
		ISystemMessageLine
{
	
	class SendInputJob extends Job
	{
		private String _input;
		public SendInputJob(String input)
		{
			super("Send Input");
			_input = input;
		}
		
		public IStatus run(IProgressMonitor monitor)
		{
			sendInput(monitor, _input);
			return Status.OK_STATUS;
		}
		
		public void sendInput(IProgressMonitor monitor, String inputStr)
		{
			IRemoteCmdSubSystem[] sses = getCmdSubSystems();
			for (int i = 0; i < sses.length; i++)
			{
				// get shell for 
				IRemoteCmdSubSystem ss = sses[i];
				IRemoteCommandShell input = getShellFor(ss);
				if (input != null)
				{
					try
					{
						ss.sendCommandToShell(inputStr, input, monitor);
					}
					catch (Exception e)
					{
						e.printStackTrace();
					}
				}
			}

		}

	}
	
	class RemoveHostRunnable implements Runnable
	{
		private IHost _host;
		public RemoveHostRunnable(IHost host)
		{
			_host = host;
		}
		
		public void run()
		{
			removeHost(_host);
		}
		
		private void removeHost(IHost host)
		{
			_connectionList.remove(host);
		}				
	}
	
	class AddHostRunnable implements Runnable
	{
		private IHost _host;
		public AddHostRunnable(IHost host)
		{
			_host = host;
		}
		
		public void run()
		{
			addHost(_host);
		}
		
		private void addHost(IHost host)
		{
			if (isHostApplicable(host))
			{
				if (!isHostListed(host))
				{
					_connectionList.add(host);
				}
			}
		}
		
		private boolean isHostListed(IHost chost)
		{
			Table table = _connectionList.getTable();
			TableItem[] items = table.getItems();
			for (int i = 0; i < items.length; i++)
			{
				IHost host = (IHost)items[i].getData();
				if (chost == host)
				{
					return true;
				}
			}
			return false;
		}
		
	}

	// keyboard handling for "enter" key
	private Listener _keyListener = new Listener()
	{
		private boolean checkState(Event event)
		{
			boolean isEnabled = !_inputEntry.isInCodeAssist();
			if (event.character == '\r')
			{
				_inputEntry.setInCodeAssist(false);
			}
			
			return isEnabled;
		}

		public void handleEvent(Event e)
		{
			if (checkState(e))
			{
				
				if (e.character == '\r' && e.stateMask!=262144) // "enter" key, but not Ctrl+M (bug 160786)
				{
					sendInput();
				}
				else if (e.keyCode == 13)
				{
					sendInput();
				}
				else if (e.stateMask == 262144)
				{
					if (e.character == 3)
					{
					    if (_inputEntry.getSelectedText().length() == 0)
					    {
					       // sendBreak();
					    }
					    else
					    {
					        _inputEntry.doOperation(ITextOperationTarget.COPY);
					    }
					}
				}
			}
		}
	};

	class BrowseAction extends Action
	{
		public BrowseAction(String label, ImageDescriptor des)
		{
			super(label, des);

			setToolTipText(label);
		}

		public void checkEnabledState()
		{
			if (_folder != null && _folder.getInput() != null)
			{
				setEnabled(true);
			}
			else
			{
				setEnabled(false);
			}
		}

		public void run()
		{
		}
	}


	// view action for clearing shells
	public class ClearAction extends BrowseAction
	{
		public ClearAction()
		{

			super(SystemResources.ACTION_CLEAR_LABEL,
			RSEUIPlugin.getDefault().getImageDescriptor(ISystemIconConstants.ICON_SYSTEM_CLEAR_ID));

			ImageDescriptor des = RSEUIPlugin.getDefault().getImageDescriptor(ISystemIconConstants.ICON_SYSTEM_CLEAR_ID);

			setImageDescriptor(des);

			setEnabled(false);
		}

		public void checkEnabledState()
		{
			Object input = _folder.getInput();
			if (input != null)
			{
				if (input instanceof IRemoteCommandShell)
				{
					setEnabled(((IRemoteCommandShell) input).isActive());
					return;
				}
			}

			setEnabled(false);
		}

		public void run()
		{
			if (_folder.getInput() != null)
			{
				clear();
			}
		}

		// clear contents of the current command viewer
		private void clear()
		{
			_folder.getViewer().clearAllItems();
		}
	}

	/*
	 * Removes all shells from view when disconnect happens
	 */
	public class CommandSubSystemDisconnectedRunnable implements Runnable
	{
		private IRemoteCmdSubSystem _subsystem;
		public CommandSubSystemDisconnectedRunnable(IRemoteCmdSubSystem subsystem)
		{
			_subsystem = subsystem;
		}

		public void run()
		{
			IRemoteCommandShell[] cmds = _subsystem.getShells();
			if (cmds != null)
			{
			for (int i = 0; i < cmds.length; i++)
			{
				_folder.remove(cmds[i]);
			}
			}
		}
	}

	public class CommandMenuManager extends MenuManager
	{
		public CommandMenuManager()
		{
			super(ShellResources.ACTION_LAUNCH_LABEL);
		}
	}
	
	public class CommandSubmenuManager extends MenuManager
	{
	    private IRemoteCmdSubSystem[] _subsystems;
		public CommandSubmenuManager(IHost connection, IRemoteCmdSubSystem[] subsystems)
		{
			super(connection.getAliasName());
			_subsystems = subsystems;
		}
		
		public IRemoteCmdSubSystem[] getSubSystems()
		{
		    return _subsystems;
		}
	}

	private ClearAction _clearAction = null;


	private SystemCommandEditor _inputEntry;

	private Button _upButton;
	private Button _downButton;
	private Composite _inputContainer;
	
	private CommandEntryViewerConfiguration _entryViewerConfiguration;
	
	private List _shellActions = null;

	private CheckboxTableViewer _connectionList = null;
	
	private MultiShellViewWorkbook _folder = null;
	private IRemoteCommandShell _lastSelected = null;
	private CellEditorActionHandler _editorActionHandler = null;

	//  for ISystemMessageLine
	private String _message, _errorMessage;
	private SystemMessage sysErrorMessage;
	private IStatusLineManager _statusLine = null;

	// constants			
	public static final String ID = "org.eclipsecon.tmtutorial.multishell"; //$NON-NLS-1$
	// matches id in plugin.xml, view tag	

	public SystemMultiShellViewPart()
	{
		super();
		_entryViewerConfiguration = new CommandEntryViewerConfiguration();
	}
	
	public void setFocus()
	{
		_folder.showCurrentPage();
	}

	public Shell getShell()
	{
		return _folder.getShell();
	}

	public Viewer getRSEViewer()
	{
		return _folder.getViewer();
	}
	
	public CellEditorActionHandler getEditorActionHandler()
	{
	    if (_editorActionHandler == null)
	    {
	        _editorActionHandler = new CellEditorActionHandler(getViewSite().getActionBars());
	    }
	    return _editorActionHandler;
	}
	
	/**
	 * Creates the folder that holds each shell view
	 * @param parent
	 */
	protected void createFolderArea(Composite parent)
	{
		// FOLDER
		_folder = new MultiShellViewWorkbook(parent, this);
		_folder.getFolder().addSelectionListener(this);
		
		GridLayout ilayout = new GridLayout();
		ilayout.numColumns = 4;
		GridData idata = new GridData(GridData.FILL_BOTH);
		_folder.setLayout(ilayout);
		_folder.setLayoutData(idata);
		
		ISelectionService selectionService = getSite().getWorkbenchWindow().getSelectionService();
		selectionService.addSelectionListener(this);		
		SystemWidgetHelpers.setHelp(_folder, RSEUIPlugin.HELPPREFIX + "ucmd0000"); //$NON-NLS-1$
	}
	
	protected void createInputArea(Composite parent)
	{
		_inputContainer = new Composite(parent, SWT.NONE);
		GridLayout ilayout = new GridLayout();
		ilayout.numColumns = 4;

		Label label = new Label(_inputContainer, SWT.NONE);
		label.setText(ShellResources.RESID_COMMANDSVIEW_COMMAND_LABEL);

		_inputEntry = new SystemCommandEditor(getViewSite(), _inputContainer, SWT.SINGLE | SWT.BORDER, 50, _entryViewerConfiguration, "", "Content Assist"); //$NON-NLS-1$
		_inputEntry.getTextWidget().setToolTipText(ShellResources.RESID_COMMANDSVIEW_COMMAND_TOOLTIP);
		_inputEntry.getTextWidget().addListener(SWT.KeyUp, _keyListener);

			
		_upButton = new Button(_inputContainer, SWT.ARROW | SWT.UP);
		_upButton.addSelectionListener(this);
		_upButton.setToolTipText(ShellResources.RESID_COMMANDSVIEW_PREVIOUS_TOOLTIP);

		_downButton = new Button(_inputContainer, SWT.ARROW | SWT.DOWN);
		_downButton.addSelectionListener(this);
		_downButton.setToolTipText(ShellResources.RESID_COMMANDSVIEW_PREVIOUS_TOOLTIP);

		GridData idata = new GridData(GridData.VERTICAL_ALIGN_END | GridData.FILL_VERTICAL | GridData.FILL_HORIZONTAL);
		idata.heightHint = 22;
		_inputEntry.getTextWidget().setLayoutData(idata);
		_inputEntry.getTextWidget().addListener(SWT.KeyUp, _keyListener);


		GridData gridData1 = new GridData(GridData.FILL_HORIZONTAL);
		gridData1.horizontalSpan = 2;
		_inputContainer.setLayout(ilayout);
		_inputContainer.setLayoutData(gridData1);
	}
	

	
	public void createPartControl(Composite parent)
	{
		GridLayout gridLayout = new GridLayout();
		gridLayout.numColumns = 2;
		parent.setLayout(gridLayout);
				
		// connection selection
		createConnectionSelectionArea(parent);
		
		// tab folder
		createFolderArea(parent);
		
		// input entry
		createInputArea(parent);

		
		// connect as listener
		ISystemRegistry registry = RSECorePlugin.getTheSystemRegistry();
		registry.addSystemResourceChangeListener(this);

		
		fillLocalToolBar();
	}

	public void selectionChanged(IWorkbenchPart part, ISelection sel)
	{
	}


	
	public void dispose()
	{
		ISelectionService selectionService = getSite().getWorkbenchWindow().getSelectionService();
		selectionService.removeSelectionListener(this);
		_folder.dispose();

		ISystemRegistry registry = RSECorePlugin.getTheSystemRegistry();
		registry.removeSystemResourceChangeListener(this);
		super.dispose();
	}

	public void updateActionStates()
	{
		if (_shellActions == null || (_shellActions.size() == 0 && _folder != null && _folder.getInput() != null))
			fillLocalToolBar();

		if (_folder != null && _folder.getInput() != null)
		{
		    IRemoteCommandShell currentSelected = (IRemoteCommandShell)_folder.getInput();
		    if (currentSelected != null)
		    {
		        if (currentSelected != _lastSelected)
		        {
		            // reset the actions
		    		IActionBars actionBars = getViewSite().getActionBars();
					IToolBarManager toolBarManager = actionBars.getToolBarManager();
					updateShellActions();
					addToolBarItems(toolBarManager);		            
		        }
		        _lastSelected = currentSelected;
		        
				_folder.updateActionStates();

				_clearAction.checkEnabledState();
	
				StructuredSelection cmdSelection = new StructuredSelection(_folder.getInput());
				for (int i =0; i < _shellActions.size(); i++)
				{
				    Object action = _shellActions.get(i);
				    if (action instanceof SystemBaseShellAction)
				    {
				        SystemBaseShellAction shellAction = (SystemBaseShellAction)action;
				    	shellAction.setEnabled(shellAction.updateSelection(cmdSelection));
				    }
				}
		    }
		}


	}

	protected void updateShellActions()
	{
	    if (_folder != null && _folder.getInput() != null)
	    {
	        IRemoteCommandShell cmdShell = (IRemoteCommandShell)_folder.getInput();
	        SystemViewRemoteOutputAdapter adapter = (SystemViewRemoteOutputAdapter)((IAdaptable)cmdShell).getAdapter(ISystemViewElementAdapter.class);
	        
	        _shellActions = adapter.getShellActions(cmdShell.getCommandSubSystem().getParentRemoteCmdSubSystemConfiguration());
	    }
	    else if (_shellActions != null)
	    {
	        _shellActions.clear();
	    }
	    else
	    {
	    	_shellActions = new ArrayList();
	    }
	}

	public void fillLocalToolBar()
	{
		boolean firstCall = false;
		if (_folder != null )
		{
			firstCall = (_shellActions == null || _shellActions.size() == 0);

			if (firstCall) {															
				updateShellActions();
			}
		
			updateActionStates();
	
			IActionBars actionBars = getViewSite().getActionBars();
	
			if (firstCall)
			{				
				_clearAction = new ClearAction();
				_statusLine = actionBars.getStatusLineManager();
			}
			IToolBarManager toolBarManager = actionBars.getToolBarManager();
			addToolBarItems(toolBarManager);
		}
	}

	private void addToolBarItems(IToolBarManager toolBarManager)
	{
		toolBarManager.removeAll();
		if (_clearAction != null)
		{
			toolBarManager.add(_clearAction);
		}
		if (_shellActions != null && _shellActions.size() > 0)
		{
			for (int i =0; i < _shellActions.size(); i++)
			{
			    Object shellAction = _shellActions.get(i);
			    if (shellAction instanceof IContributionItem)
			    {
			        toolBarManager.add((IContributionItem)shellAction);
			    }
			    else if (shellAction instanceof IAction)
			    {
			        toolBarManager.add((IAction)shellAction);
			    }
			}
		}
		toolBarManager.update(true);
	}




	public void selectionChanged(SelectionChangedEvent e)
	{
	}

	/**
	 * Update or create a view tab for the specified command
	 * If there is no tab for the command, create one.
	 * 
	 * @param root command to view
	 * 
	 */
	public void updateOutput(IRemoteCommandShell root)
	{
		updateOutput(root, true);
	}

	private void updateOutput(IRemoteCommandShell root, boolean createTab)
	{
		if (root != null)
		{
			_folder.updateOutput(root, createTab);
			if (createTab)
			    updateActionStates();
		}
	}

	public void setInput(IAdaptable object)
	{
		setInput(object, true);
	}

	public void setInput(IAdaptable object, boolean updateHistory)
	{	
		_folder.setInput(object);
	}

	/**
	 * Creates the table of connections to use with multishell
	 * @param parent
	 */
	protected void createConnectionSelectionArea(Composite parent)
	{
		_connectionList = CheckboxTableViewer.newCheckList(parent, SWT.BORDER);
		GridLayout tlayout = new GridLayout();
		Table table = _connectionList.getTable();
		
		table.setLayout(tlayout);
		GridData tdata = new GridData(GridData.FILL_VERTICAL);
		table.setLayoutData(tdata);
		ISystemRegistry registry = RSECorePlugin.getTheSystemRegistry();
		IHost[] hosts = registry.getHosts();
		for (int i = 0; i < hosts.length; i++)
		{
			IHost host = hosts[i];
			addHost(host);
		}
	}
	
	private void addHost(IHost host)
	{
		AddHostRunnable runnable = new AddHostRunnable(host);
		Display.getDefault().asyncExec(runnable);
	}
	
	private void removeHost(IHost host)
	{
		RemoveHostRunnable runnable = new RemoveHostRunnable(host);
		Display.getDefault().asyncExec(runnable);
	}

	
	private boolean isHostApplicable(IHost host)
	{
		ISubSystem[] sses = host.getSubSystems();
		for (int i = 0; i < sses.length; i++)
		{
			ISubSystem ss = sses[i];
			if (ss instanceof IRemoteCmdSubSystem)
			{
				return ss.isConnected();
			}
		}
		return false;
	}


	/**
	 * Resource change listening
	 */
	public void systemResourceChanged(ISystemResourceChangeEvent event)
	{ 
		int type = event.getType();
		switch (type)
		{
		
		// if a new host is added, need to add it to the list of hosts to do multishell with
		case ISystemResourceChangeEvents.EVENT_ADD_RELATIVE:
		case ISystemResourceChangeEvents.EVENT_ADD:
		{
			Object source = event.getSource();
			if (source instanceof IHost)
			{
				IHost host = (IHost)source;
				addHost(host);
			}
		}
		break;
		
		// if a host is deleted, need to remove it from the list of hosts to do multishell with
		case ISystemResourceChangeEvents.EVENT_DELETE:
		{
			Object source = event.getSource();
			if (source instanceof IHost)
			{
				removeHost((IHost)source);
			}
			
		}
		break;
		
		// connection could have disconnected or connected
		// need to update the list of hosts
		case ISystemResourceChangeEvents.EVENT_PROPERTY_CHANGE:
		{
			Object source = event.getSource();
			if (source instanceof ISubSystem)
			{
				ISubSystem ss = (ISubSystem)source;
				IHost host = ss.getHost();
				if (ss.isConnected())
				{				
					addHost(host);
				}
				else
				{
					removeHost(host);
				}
			}
		}
		break;
		
		// indicates that a given shell is finished
		case ISystemResourceChangeEvents.EVENT_COMMAND_SHELL_FINISHED:
		{
			Object source = event.getSource();
			// case where the entire shell subsystem is down
			if (source instanceof IRemoteCmdSubSystem)
			{
				Shell shell = RSEUIPlugin.getTheSystemRegistryUI().getShell();
				shell.getDisplay().asyncExec(new CommandSubSystemDisconnectedRunnable((IRemoteCmdSubSystem) source));
			}
			// case where the shell is complete
			else if (source instanceof IRemoteCommandShell)
			{
				// find out if we're listening to this

				updateOutput((IRemoteCommandShell) source, false);
				updateActionStates();
			}
		} 
		break;
		
		// indicates that a given shell is no longer being viewed
		case ISystemResourceChangeEvents.EVENT_COMMAND_SHELL_REMOVED:
		{
			Object source = event.getSource();
			if (source instanceof IRemoteCommandShell)
			{
			    updateOutput((IRemoteCommandShell) source, false);
			    _folder.remove(source);
				updateActionStates();
			}
		}
		break;
		
		// indicates the output has been updated 
		case ISystemResourceChangeEvents.EVENT_REFRESH:
		{
			Object parent = event.getParent();
			
			// only updating if the output is for this shell
			if (parent instanceof IRemoteCommandShell)
			{
				updateOutput((IRemoteCommandShell) parent, false);
			}
		}
		break;
		
	
		// used for cleaning up on a disconnect
		case ISystemResourceChangeEvents.EVENT_MUST_COLLAPSE:
		{
		    Object source = event.getSource();
			if (source instanceof IRemoteCmdSubSystem)
			{
				Shell shell = RSEUIPlugin.getTheSystemRegistryUI().getShell();
				shell.getDisplay().asyncExec(new CommandSubSystemDisconnectedRunnable((IRemoteCmdSubSystem) source));
			}		    
		}
		break;
		
		default:
			break;
		}
	}

	public void widgetDefaultSelected(SelectionEvent e)
	{
		widgetSelected(e);
	}

	public void widgetSelected(SelectionEvent e)
	{
		Widget source = e.widget;

		if (source == _folder.getFolder())
		{
			if (_folder.getInput() != null)
			{
				updateActionStates();
			}
		}
	}
	
	
//	 -------------------------------
	// ISystemMessageLine interface...
	// -------------------------------
	/**
	 * Clears the currently displayed error message and redisplayes
	 * the message which was active before the error message was set.
	 */
	public void clearErrorMessage()
	{
		_errorMessage = null;
		sysErrorMessage = null;
		if (_statusLine != null)
			_statusLine.setErrorMessage(_errorMessage);
	}
	/**
	 * Clears the currently displayed message.
	 */
	public void clearMessage()
	{
		_message = null;
		if (_statusLine != null)
			_statusLine.setMessage(_message);
	}
	/**
	 * Get the currently displayed error text.
	 * @return The error message. If no error message is displayed <code>null</code> is returned.
	 */
	public String getErrorMessage()
	{
		return _errorMessage;
	}
	/**
	 * Get the currently displayed message.
	 * @return The message. If no message is displayed <code>null<code> is returned.
	 */
	public String getMessage()
	{
		return _message;
	}
	/**
	 * Display the given error message. A currently displayed message
	 * is saved and will be redisplayed when the error message is cleared.
	 */
	public void setErrorMessage(String message)
	{
		this._errorMessage = message;
		if (_statusLine != null)
			_statusLine.setErrorMessage(message);
	}
	/**
	 * Get the currently displayed error text.
	 * @return The error message. If no error message is displayed <code>null</code> is returned.
	 */
	public SystemMessage getSystemErrorMessage()
	{
		return sysErrorMessage;
	}

	/**
	 * Display the given error message. A currently displayed message
	 * is saved and will be redisplayed when the error message is cleared.
	 */
	public void setErrorMessage(SystemMessage message)
	{
		sysErrorMessage = message;
		setErrorMessage(message.getLevelOneText());
	}
	/**
	 * Display the given error message. A currently displayed message
	 * is saved and will be redisplayed when the error message is cleared.
	 */
	public void setErrorMessage(Throwable exc)
	{
		setErrorMessage(exc.getMessage());
	}

	/**
	 * Set the message text. If the message line currently displays an error,
	 * the message is stored and will be shown after a call to clearErrorMessage
	 */
	public void setMessage(String message)
	{
		this._message = message;
		if (_statusLine != null)
			_statusLine.setMessage(message);
	}
	/** 
	 *If the message line currently displays an error,
	 * the message is stored and will be shown after a call to clearErrorMessage
	 */
	public void setMessage(SystemMessage message)
	{
		setMessage(message.getLevelOneText());
	}
	

		public void sendInput()
		{
			String inputStr = _inputEntry.getTextWidget().getText();
			sendInput(new NullProgressMonitor(), inputStr);
		}
		
		public void sendInput(IProgressMonitor monitor, String inputStr)
		{
			IRemoteCmdSubSystem[] sses = getCmdSubSystems();
			for (int i = 0; i < sses.length; i++)
			{
				// get shell for 
				IRemoteCmdSubSystem ss = sses[i];
				IRemoteCommandShell input = getShellFor(ss);
				if (input != null)
				{
					try
					{
						ss.sendCommandToShell(inputStr, input, monitor);
					}
					catch (Exception e)
					{
						e.printStackTrace();
					}
				}
			}
			
			_inputEntry.getTextWidget().setText(""); //$NON-NLS-1$
			_inputEntry.getTextWidget().setFocus();

		}

		protected IRemoteCmdSubSystem[] getCmdSubSystems()
		{
			List sses = new ArrayList();
			TableItem[] items = _connectionList.getTable().getItems();
			for (int i = 0; i < items.length; i++)
			{
				TableItem item = items[i];
				if (item.getChecked())
				{
					IHost host = (IHost)item.getData();
					IRemoteCmdSubSystem ss = RemoteCommandHelpers.getCmdSubSystem(host);
					sses.add(ss);
				}
			}
			
			return (IRemoteCmdSubSystem[])sses.toArray(new IRemoteCmdSubSystem[sses.size()]);
		}
		
		protected IRemoteCommandShell getShellFor(IRemoteCmdSubSystem ss)
		{
			IRemoteCommandShell shell =  _folder.getShellFor(ss);
			if (shell == null)
			{
				// create shell
				try
				{
					if (!ss.isConnected())
					{
						//TODO must run synchronously since returning the shell?
						//What if we are on the dispatch thread?
						//ss.connect();
						ss.connect(null, false);
					}
					
					shell = ss.runShell(null, new NullProgressMonitor());
					updateOutput(shell);
				}
				catch (Exception e)
				{
					
				}				
			}
			return shell;
		}
		


}