package org.eclipse.eodm;

import java.net.URISyntaxException;
import org.eclipse.eodm.exceptions.InvalidLexicalFormException;
import org.eclipse.eodm.rdf.rdfbase.BlankNode;
import org.eclipse.eodm.rdf.rdfbase.PlainLiteral;
import org.eclipse.eodm.rdf.rdfbase.RDFGraph;
import org.eclipse.eodm.rdf.rdfbase.RDFProperty;
import org.eclipse.eodm.rdf.rdfbase.RDFSLiteral;
import org.eclipse.eodm.rdf.rdfbase.RDFSResource;
import org.eclipse.eodm.rdf.rdfbase.RDFStatement;
import org.eclipse.eodm.rdf.rdfbase.RDFXMLLiteral;
import org.eclipse.eodm.rdf.rdfbase.TypedLiteral;
import org.eclipse.eodm.rdf.rdfbase.URIReference;
import org.eclipse.eodm.rdf.rdfbase.URIReferenceNode;
import org.eclipse.eodm.rdf.rdfbase.UniformResourceIdentifier;
import org.eclipse.eodm.rdf.rdfs.RDFAlt;
import org.eclipse.eodm.rdf.rdfs.RDFBag;
import org.eclipse.eodm.rdf.rdfs.RDFList;
import org.eclipse.eodm.rdf.rdfs.RDFSClass;
import org.eclipse.eodm.rdf.rdfs.RDFSContainer;
import org.eclipse.eodm.rdf.rdfs.RDFSContainerMembershipProperty;
import org.eclipse.eodm.rdf.rdfs.RDFSDatatype;
import org.eclipse.eodm.rdf.rdfs.RDFSeq;
import org.eclipse.eodm.rdf.rdfweb.Document;
import org.eclipse.eodm.rdf.rdfweb.LocalName;
import org.eclipse.eodm.rdf.rdfweb.Namespace;
import org.eclipse.eodm.rdf.rdfweb.NamespaceDefinition;

/**
 * RDFFactory - An abstract factory interface for creating all kinds of RDF objects.
 * 
 * <p>The creation method for RDF resources has the following variants:
 * <ul>
 *  <li><code>createXYZ( RDFGraph graph, String uriRefString )</code>
 *  <li><code>createXYZ( RDFGraph graph, URIReferecne uriRef)</code>
 *  <li><code>createXYZ( RDFGraph graph, Namespace namespace, String localNameString )</code>
 * </ul>
 * All of the above methods return an object that uses the <code>graph</code> as its context
 * graph and corresponds to a resource identified by a RDF URI Reference. 
 * Both the identity and the context graph can not be changed after the object 
 * is returned from the above methods. However, a resource may have multiple
 * RDF URI References that are equal to each other.<p>
 * 
 * <p>To create an anonymous resource, you need first create a BlankNode using the 
 * method {@link org.eclipse.eodm.RDF.RDFFactory#createBlankNode(RDFGraph) createBlankNode(RDFGraph g)}
 * and force that blank node to the type of resource you need using the method
 * {@link org.eclipse.eodm.RDF.RDFS.RDFBase.RDFSResource#asType(Class) asType(Class)}.
 * The factory also provides methods for creating some kind of anonymous resource,
 * such as RDFList, RDFAlt, etc.</p>
 * 
 * <a name="uriRefString"></a>
 * <p>The <code>uriRefString</code> parameter in the first kind of method for creating
 * a non-anonymous resource must be a valid <strong>RDF URI Reference</strong>. 
 * Defined by <a href="http://www.w3.org/TR/2004/REC-rdf-concepts-20040210/#dfn-URI-reference">
 * W3C RDF Concepts and Abstract Syntaxes</a>, a RDF URI reference is a Unicode 
 * string that:
 * <ul>
 *   <li>does not contain any control characters ( #x00 - #x1F, #x7F-#x9F)
 *   <li>and would produce a valid URI character sequence (per RFC-2396, 
 *   sections 2.1) representing an absolute URI with optional fragment identifier 
 *   when subjected to the encoding described below.
 * </ul>
 * The encoding consists of:
 * <ul>
 *   <li>encoding the Unicode string as UTF-8 [RFC-2279], giving a sequence of octet values.
 *   <li>%-escaping octets that do not correspond to permitted US-ASCII characters.
 * </ul>
 * </p>
 *   
 * <p></p>
 * 
 * @eodmext  This factory interface is added by EODM.
 * @author zhanglei
 */

public interface RDFFactory {
	/**
	 * The singleton instance of the factory.
	 * <!-- begin-user-doc -->
	 * <!-- end-user-doc -->
	 * @generated
	 */
	RDFFactory eINSTANCE = new org.eclipse.eodm.impl.RDFFactoryImpl();
	


    
    
	/**
     * Creates a new unique '<em>BlankNode</em>'.
     * The nodeID of the new blank node will be different from all blank 
     * nodes created using this factory.
     * 
     * @param graph  the context graph of this blank node.
     * @return a new object of class '<em>BlankNode</em>' with a unique nodeID.
     */
    BlankNode createBlankNode( RDFGraph graph, String nodeid );

    /**
     * Returns an object of class '<em>RDFSResource</em>'.
     * 
     * @param graph   the context graph of this resource.
     * @param uriRefString  the RDF URI Reference of this object.
     *        <a href="#uriRefString">The string must be a valid RDF URI Reference</a>.
     * @return a new object of class '<em>RDFSResource</em>'.
     * @throws URISyntaxException if the uriRefString is not a valid RDF URI Reference.
     */
    RDFSResource createRDFSResource( RDFGraph graph, String uriRefString )
    throws URISyntaxException;

    /**
     * Returns an object of class '<em>RDFSResource</em>'.
     * 
     * @param graph   the context graph of this resource.
     * @param uriRef  the RDF URI Reference of this object.
     * @return a new object of class '<em>RDFSResource</em>'.
     */
    RDFSResource createRDFSResource( RDFGraph graph, URIReference uriRef );
    
    /**
     * Returns an object of class '<em>RDFSResource</em>'.
     * 
     * @param graph     the context graph of this resource.
     * @param namespace the namespace of the RDF URI Reference of this object.
     * @param localNameString the fragment identifier string of the RDF URI Reference.
     * 
     * @return a new object of class '<em>RDFSResource</em>'.
     */
    RDFSResource createRDFSResource( RDFGraph graph, Namespace namespace, String localNameString );
        
    /**
     * Returns an object of class '<em>PlainLiteral</em>'.
     * 
     * @param lexicalForm the lexical form of the plain literal. It may be 
     *        <code>null</code> and be set later.
     * @return an object of class '<em>PlainLiteral</em>'. The returned plain literal
     *         does not have a language tag (the value of the language tag is
     *         <code>null</code>) and can be set later.
     */
    PlainLiteral createPlainLiteral( String lexicalForm );

    /**
     * Returns an object of class '<em>PlainLiteral</em>'
     * 
     * @param lexicalForm the lexical form of the plain literal. It may be
     *        <code>null</code> and be set later.
     * @param languageTag the language tag of the plain literal. It may be
     *        <code>null</code> to signify that no language tag is set.
     * @return an object of class '<em>PlainLiteral</em>'. 
     */
    PlainLiteral createPlainLiteral( String lexicalForm, String languageTag );

    /**
     * Returns an object of class '<em>URIReference</em>'.
     * 
     * @param uriRefString  <a href="#uriRefString">The string must be a valid RDF URI Reference</a>.
     * @return an object of class '<em>URIReference</em>'.
     * @throws URISyntaxException if the uriRefString is not a valid RDF URI Reference.
     */
    URIReference createURIReference( String uriRefString )
    throws URISyntaxException;

    /**
     * Returns an object of class '<em>URIReference</em>'. The URI of this
     * URI Reference is the namespace URI appended with the localNameString.
     * 
     * @param namespace the namespace of this URI Reference
     * @param localNameString the fragment identifier of this URI Reference
     * @return an object of class '<em>URIReference</em>'.
     */
    URIReference createURIReference( Namespace namespace, String localNameString );
    
    /**
     * Returns an object of class '<em>URIReference</em>'. The URI of this
     * URI Reference is the namespace URI appended with the localName string.
     * 
     * @param namespace the namespace of this URI Reference
     * @param localName the fragment identifier of this URI Reference
     * @return an object of class '<em>URIReference</em>'.
     */
    URIReference createURIReference( Namespace namespace, LocalName localName );
    
    /**
     * Returns an object of class '<em>RDFStatement</em>'.
     * If this statement is reified, it is an anonymous statement without a RDF URI
     * reference and can not be given a URI reference.
     *  
     * @return an object of class '<em>RDFStatement</em>'.
     */
    RDFStatement createRDFStatement(RDFGraph graph);

    /**
     * Returns an object of class '<em>RDFStatement</em>'.
     * If this statement is reified, the reified statement resource will use
     * <code>uriRefString</code> as its RDF URI Reference and can not be changed.
     * 
     * @param uriRefString  <a href="#uriRefString">The string must be a valid RDF URI Reference</a>.
     * @return an object of class '<em>RDFStatement</em>'.
     * @throws URISyntaxException if the uriRefString is not a valid RDF URI Reference.
     */
    RDFStatement createRDFStatement(RDFGraph graph, String uriRefString )
    throws URISyntaxException;
    
    /**
     * Returns an object of class '<em>RDFProperty</em>'.
     * 
     * @param graph   the context graph of this property.
     * @param uriRefString  the RDF URI Reference of this property.
     *        <a href="#uriRefString">The string must be a valid RDF URI Reference</a>. 
     * @return an object of class '<em>RDFProperty</em>'.
     * @throws URISyntaxException if the uriRefString is not a valid RDF URI Reference.
     */
    RDFProperty createRDFProperty( RDFGraph graph, String uriRefString )
    throws URISyntaxException;

    /**
     * Returns an object of class '<em>RDFProperty</em>'.
     * 
     * @param graph   the context graph of this property.
     * @param uriRef  the RDF URI Reference of this property.
     * @return an object of class '<em>RDFProperty</em>'.
     */
    RDFProperty createRDFProperty( RDFGraph graph, URIReference uriRef );
    
    /**
     * Returns an object of class '<em>RDFProperty</em>'.
     * The URI of this property is the namespace URI appended with the localNameString.
     * 
     * @param graph   the context graph of this property.
     * @param namespace the namespace of the URI Reference of this property.
     * @param localNameString the fragment identifier of the URI Reference of this property.
     * @return an object of class '<em>RDFProperty</em>'.
     */
    RDFProperty createRDFProperty( RDFGraph graph, Namespace namespace, String localNameString );
    
    /**
     * Returns an object of class '<em>UniformResourceIdentifier</em>'.
     * 
     * @param uriString the URI string of this UniformResourceIdentifier.
     * @return an object of class '<em>UniformResourceIdentifier</em>'.
     * @throws URISyntaxException if the uriString is not a valid URI.
     */
    UniformResourceIdentifier createUniformResourceIdentifier( String uriString )
    throws URISyntaxException;

    /**
     * Returns an object of class '<em>RDFGraph</em>'. 
     * 
     * @param uriRefString  The string must be <code>null</code> or <a href="#uriRefString"> a valid RDF URI Reference</a>.
     *        If it is a valid RDF URI Reference, it serves as the name of the graph
     *        and can not be changed. If it is <code>null</code>, this graph then
     *        does not have a name and can not be given a name later.
     * @return an object of class '<em>RDFGraph</em>'.
     * @throws URISyntaxException if the uriRefString is not <code>null</code> and
     *         is not a valid RDF URI Reference.
     */
    RDFGraph createGraph( String uriRefString )
    throws URISyntaxException;
    /**
     * Returns an object of class '<em>RDFGraph</em>'.
     * 
     * @param uriRef the RDF URI Reference of the RDFGraph.
     * @return an object of class '<em>RDFGraph</em>'.
     */
    RDFGraph createGraph( URIReference uriRef );
    
    /**
     * Returns an object of class '<em>RDFGraph</em>'.
     * 
     * @param namespace       the namespace of the RDF URI Reference of the RDFGraph
     * @param localNameString the fragment identifier of the RDF URI Reference of the RDFGraph
     * @return an object of class '<em>RDFGraph</em>'.
     */
    RDFGraph createGraph( Namespace namespace, String localNameString );

    /**
     * Returns an object of class '<em>URIReferenceNode</em>'.
     * 
     * @param graph  the context graph of this node.
     * @param uriRefString the RDF URI Reference of this node. <a href="#uriRefString">The string must be a valid RDF URI Reference</a>.
     * @return an object of class '<em>URIReferenceNode</em>'.
     * @throws URISyntaxException if the uriRefString is not a valid RDF URI Reference.
     */
    URIReferenceNode createURIReferenceNode( RDFGraph graph, String uriRefString )
    throws URISyntaxException;

    /**
     * Returns an object of class '<em>URIReferenceNode</em>'
     * 
     * @param graph the context graph of this node.
     * @param uriRef the RDF URI Reference of this node.
     * @return an object of class '<em>URIReferenceNode</em>'
     */
    URIReferenceNode createURIReferenceNode( RDFGraph graph, URIReference uriRef );
    
    /**
     * Returns an object of class '<em>URIReferenceNode</em>'
     *  
     * @param graph the context graph of this node
     * @param namespace the namespace of the RDF URI Reference of this node
     * @param localNameString the fragment identifier of the RDF URI Reference of this node
     * @return an object of class '<em>URIReferenceNode</em>'
     */
    URIReferenceNode createURIReferenceNode( RDFGraph graph, Namespace namespace, String localNameString );
    
    /**
     * Returns an object of class '<em>RDFSLiteral</em>'.
     * 
     * @param lexicalForm the lexical form of the literal, may be <code>null</code> to
     *        be set later.
     * @return an object of class '<em>RDFSLiteral</em>'.
     */
    RDFSLiteral createRDFSLiteral( String lexicalForm ) ;
    
    /**
     * Returns an object of class '<em>TypedLiteral</em>'.
     * 
     * @param lexicalForm the lexical form of this typed literal
     * @param dataTypeURIRef the RDF URI Reference of the data type of thie literal.
     *        <a href="#uriRefString">The string must be a valid RDF URI Reference</a>.
     * @return an object of class '<em>TypedLiteral</em>'.
     * @throws URISyntaxException if the dataTypeURIRef is not a valid RDF URI Reference.
     * @throws InvalidLexicalFormException if the <code>lexicalForm</code> string is not <code>
     *         null</code> and does not conforms to the data type identified by 
     *         the data type URI Reference.
     */
    TypedLiteral createTypedLiteral( String lexicalForm, String dataTypeURIRef )
    throws URISyntaxException, InvalidLexicalFormException;

    /**
     * Returns an object of class '<em>RDFXMLLiteral</em>'. This literal
     * has an XML content as its lexical form and a URI Reference for identifying
     * the data type: <code>http://www.w3.org/1999/02/22-rdf-syntax-ns#XMLLiteral</code>.
     * 
     * @param xmlString the lexical form of this literal in the form of a XML content.
     *        Please see <a href="http://www.w3.org/TR/rdf-concepts/#section-XMLLiteral">
     *        here</a> for the specification on this XML lexical form. It may be
     *        <code>null</code> and changed later.
     * @return an object of class '<em>RDFXMLLiteral</em>'.
     * @throws InvalidLexicalFormException  if the <code>xmlString</code> is not
     *         <code>null</code> and does not conforms to the rdf:XMLLiteral data
     *         type specification.
     */
    RDFXMLLiteral createRDFXMLLiteral( String xmlString )
    throws InvalidLexicalFormException;
    
    /**
     * Returns an object of class '<em>RDFSClass</em>'.
     * 
     * @param graph the context graph of this class.
     * @param uriRefString the RDF URI Reference of the class. 
     *        <a href="#uriRefString">The string must be a valid RDF URI Reference</a>.
     * @return an object of class '<em>RDFSClass</em>'.
     * @throws URISyntaxException if the <code>uriRefString</code> is not a 
     *         valid RDF URI Reference.
     */
    RDFSClass createRDFSClass( RDFGraph graph, String uriRefString )
    throws URISyntaxException;

    /**
     * Returns an object of class '<em>RDFSClass</em>'.
     * 
     * @param graph the context graph of this class.
     * @param uriRef the RDF URI Reference of this class.
     * @return an object of class '<em>RDFSClass</em>'
     */
    RDFSClass createRDFSClass( RDFGraph graph, URIReference uriRef );
    
    /**
     * Returns an object of class '<em>RDFSClass</em>'.
     * 
     * @param graph           the context graph of this class.
     * @param namespace       the namespace of the RDF URI Reference of this class.
     * @param localNameString the fragment identifier of the RDF URI Reference of this class.
     * @return an object of class '<em>RDFSClass</em>'
     */
    RDFSClass createRDFSClass( RDFGraph graph, Namespace namespace, String localNameString );
    
    /**
     * Returns an object of class '<em>RDFSDatatype</em>'.
     * 
     * @param graph the context graph of this class.
     * @param uriRefString the RDF URI Reference of the data type. 
     *        <a href="#uriRefString">The string must be a valid RDF URI Reference</a>.
     * @return an object of class '<em>RDFSDatatype</em>'.
     * @throws URISyntaxException if the <code>uriRefString</code> is not a 
     *         valid RDF URI Reference.
     */
    RDFSDatatype createRDFSDatatype( RDFGraph graph, String uriRefString )
    throws URISyntaxException ;
    
    /**
     * Returns an object of class '<em>RDFSDatatype</em>'.
     * 
     * @param graph the context graph of this data type.
     * @param uriRef the RDF URI Reference of the data type.
     * @return an object of class '<em>RDFSDatatype</em>'.
     */
    RDFSDatatype createRDFSDatatype( RDFGraph graph, URIReference uriRef );

    /**
     * Returns an object of class '<em>RDFSDatatype</em>'.
     * 
     * @param graph            the context graph of the data type
     * @param namespace        the namespace of the RDF URI Reference of the data type
     * @param localNameString  the fragment identifier of the RDF URI Reference of the data type
     * @return an object of '<em>RDFSDatatype</em>'.
     */
    RDFSDatatype createRDFSDatatype( RDFGraph graph, Namespace namespace, String localNameString );
    
    /**
     * Returns a new unique blank node that is of type rdf:List.
     * This list can not be given a URL Reference later.
     * 
     * @param graph the context graph of this RDFList.
     * @return a new unique blank node that is of type rdf:List.
     */
    RDFList createRDFList( RDFGraph graph ) ;
    
    /**
     * Returns an object of class '<em>RDFList</em>'.
     * 
     * @param graph the context graph of this list.
     * @param uriRefString the RDF URI Reference of the list.
     *        <a href="#uriRefString">The string must be a valid RDF URI Reference</a>.
     * @return an object of class '<em>RDFList</em>'.
     * @throws URISyntaxException if the <code>uriRefString</code> is not a 
     *         valid RDF URI Reference. 
     */
    RDFList createRDFList( RDFGraph graph, String uriRefString )
    throws URISyntaxException;

    /**
     * Returns an object of class '<em>RDFList</em>'.
     * 
     * @param graph  the context graph of this list
     * @param uriRef the RDF URI Reference of the list.
     * @return an object of class '<em>RDFList</em>'.
     */
    RDFList createRDFList( RDFGraph graph, URIReference uriRef );
    
    /**
     * Returns an object of class '<em>RDFList</em>'.
     * 
     * @param graph           the context graph of this list
     * @param namespace       the namespace of the RDF URI Reference of the list
     * @param localNameString the fragment identifier of the RDF URI Reference of the list
     * @return an object of class '<em>RDFList</em>'.
     */
    RDFList createRDFList( RDFGraph graph, Namespace namespace, String localNameString );
    
    /**
     * Returns a new unique blank node that is of type rdfs:Container.
     * This resouce can not be given a URI Reference later.
     *  
     * @param graph the context graph of this RDFSContainer.
     * @return a new unique blank node that is of type rdfs:Container.
     */
    RDFSContainer createRDFSContainer( RDFGraph graph );
    
    /**
     * Returns an object of class '<em>RDFSContainer</em>'.
     * 
     * @param graph the context graph of this RDFSContainer.
     * @param uriRefString the RDF URI Reference of the RDFSContainer.
     *        <a href="#uriRefString">The string must be a valid RDF URI Reference</a>.
     * @return an object of class '<em>RDFSContainer</em>'.
     * @throws URISyntaxException if the <code>uriRefString</code> is not a 
     *         valid RDF URI Reference. 
     */
    RDFSContainer createRDFSContainer( RDFGraph graph, String uriRefString )
    throws URISyntaxException;
    
    /**
     * Returns an object of class '<em>RDFSContainer</em>'.
     * 
     * @param graph  the context graph of this RDFSContainer
     * @param uriRef the RDF URI Reference of the RDFSContainer.
     * @return an object of class '<em>RDFSContainer</em>'.
     */
    RDFSContainer createRDFSContainer( RDFGraph graph, URIReference uriRef );
    
    /**
     * Returns an object of class '<em>RDFSContainer</em>'.
     * 
     * @param graph           the context graph of this RDFSContainer
     * @param namespace       the namespace of the RDF URI Reference of the RDFSContainer
     * @param localNameString the fragment identifier of the RDF URI Reference of the RDFSContainer
     * @return an object of class '<em>RDFSContainer</em>'.
     */
    RDFSContainer createRDFSContainer( RDFGraph graph, Namespace namespace, String localNameString );    
    
    /**
     * Returns a new unique blank node of type RDFAlt.
     * The resource can not be given a URI Reference later.
     *
     * @param graph the context graph of this RDFAlt
     * 
     * @return a new unique blank node of type RDFAlt
     */
    RDFAlt createRDFAlt( RDFGraph graph );
    
    /**
     * Returns an object of class '<em>RDFAlt</em>'.
     * 
     * @param graph the context graph of this RDFAlt.
     * @param uriRefString the RDF URI Reference of the RDFAlt.
     *        <a href="#uriRefString">The string must be a valid RDF URI Reference</a>.
     * @return an object of class '<em>RDFAlt</em>'.
     * @throws URISyntaxException if the <code>uriRefString</code> is not a 
     *         valid RDF URI Reference. 
     */
    RDFAlt createRDFAlt( RDFGraph graph, String uriRefString )
    throws URISyntaxException;
    
    /**
     * Returns an object of class '<em>RDFAlt</em>'.
     * 
     * @param graph  the context graph of this RDFAlt
     * @param uriRef the RDF URI Reference of the RDFAlt.
     * @return an object of class '<em>RDFAlt</em>'.
     */
    RDFAlt createRDFAlt( RDFGraph graph, URIReference uriRef );
    
    /**
     * Returns an object of class '<em>RDFAlt</em>'.
     * 
     * @param graph           the context graph of this RDFAlt
     * @param namespace       the namespace of the RDF URI Reference of the RDFAlt
     * @param localNameString the fragment identifier of the RDF URI Reference of the RDFAlt
     * @return an object of class '<em>RDFAlt</em>'.
     */
    RDFAlt createRDFAlt( RDFGraph graph, Namespace namespace, String localNameString );
    
    /**
     * Returns a new unique blank node of type RDFBag.
     * The resource can not be given a URI Reference later.
     *
     * @param graph the context graph of this RDFBag
     * @return a new unique blank node of type RDFBag
     */
    RDFBag createRDFBag( RDFGraph graph );   
    
    /**
     * Returns an object of class '<em>RDFBag</em>'.
     * 
     * @param graph the context graph of this RDFBag.
     * @param uriRefString the RDF URI Reference of the RDFBag.
     *        <a href="#uriRefString">The string must be a valid RDF URI Reference</a>.
     * @return an object of class '<em>RDFBag</em>'.
     * @throws URISyntaxException if the <code>uriRefString</code> is not a 
     *         valid RDF URI Reference. 
     */
    RDFBag createRDFBag( RDFGraph graph, String uriRefString )
    throws URISyntaxException;
    
    /**
     * Returns an object of class '<em>RDFBag</em>'.
     * 
     * @param graph  the context graph of this RDFBag
     * @param uriRef the RDF URI Reference of the RDFBag.
     * @return an object of class '<em>RDFBag</em>'.
     */
    RDFBag createRDFBag( RDFGraph graph, URIReference uriRef );
    
    /**
     * Returns an object of class '<em>RDFBag</em>'.
     * 
     * @param graph           the context graph of this RDFBag
     * @param namespace       the namespace of the RDF URI Reference of the RDFBag
     * @param localNameString the fragment identifier of the RDF URI Reference of the RDFBag
     * @return an object of class '<em>RDFBag</em>'.
     */
    RDFBag createRDFBag( RDFGraph graph, Namespace namespace, String localNameString );

    /**
     * Returns a new unique blank node of type RDFSeq.
     * The resource can not be given a URI Reference later.
     *
     * @param graph the context graph of this RDFSeq
     * @return a new unique blank node of type RDFSeq
     */
    RDFSeq createRDFSeq( RDFGraph graph );
    
    /**
     * Returns an object of class '<em>RDFSeq</em>'.
     * 
     * @param graph the context graph of this RDFSeq.
     * @param uriRefString the RDF URI Reference of the RDFSeq.
     *        <a href="#uriRefString">The string must be a valid RDF URI Reference</a>.
     * @return an object of class '<em>RDFSeq</em>'.
     * @throws URISyntaxException if the <code>uriRefString</code> is not a 
     *         valid RDF URI Reference. 
     */
    RDFSeq createRDFSeq( RDFGraph graph, String uriRefString )
    throws URISyntaxException;
    
    /**
     * Returns an object of class '<em>RDFSeq</em>'.
     * 
     * @param graph  the context graph of this RDFSeq
     * @param uriRef the RDF URI Reference of the RDFSeq.
     * @return an object of class '<em>RDFSeq</em>'.
     */
    RDFSeq createRDFSeq( RDFGraph graph, URIReference uriRef );
    
    /**
     * Returns an object of class '<em>RDFSeq</em>'.
     * 
     * @param graph           the context graph of this RDFSeq
     * @param namespace       the namespace of the RDF URI Reference of the RDFSeq
     * @param localNameString the fragment identifier of the RDF URI Reference of the RDFSeq
     * @return an object of class '<em>RDFSeq</em>'.
     */
    RDFSeq createRDFSeq( RDFGraph graph, Namespace namespace, String localNameString );

    /**
     * Returns an object of class '<em>RDFSContainerMembershipProperty</em>'.
     * 
     * @param graph the context graph of this RDFSContainerMembershipProperty.
     * @param uriRefString the RDF URI Reference of the RDFSContainerMembershipProperty.
     *        <a href="#uriRefString">The string must be a valid RDF URI Reference</a>.
     * @return an object of class '<em>RDFSContainerMembershipProperty</em>'.
     * @throws URISyntaxException if the <code>uriRefString</code> is not a 
     *         valid RDF URI Reference. 
     */
    RDFSContainerMembershipProperty createRDFSContainerMembershipProperty( RDFGraph graph, String uriRefString )
    throws URISyntaxException;
    
    /**
     * Returns an object of class '<em>RDFSContainerMembershipProperty</em>'.
     * 
     * @param graph  the context graph of this RDFSContainerMembershipProperty
     * @param uriRef the RDF URI Reference of the RDFSContainerMembershipProperty.
     * @return an object of class '<em>RDFSContainerMembershipProperty</em>'.
     */
    RDFSContainerMembershipProperty createRDFSContainerMembershipProperty( RDFGraph graph, URIReference uriRef );
    
    /**
     * Returns an object of class '<em>RDFSContainerMembershipProperty</em>'.
     * 
     * @param graph           the context graph of this RDFSContainerMembershipProperty
     * @param namespace       the namespace of the RDF URI Reference of the RDFSContainerMembershipProperty
     * @param localNameString the fragment identifier of the RDF URI Reference of the RDFSContainerMembershipProperty
     * @return an object of class '<em>RDFSContainerMembershipProperty</em>'.
     */
    RDFSContainerMembershipProperty createRDFSContainerMembershipProperty( RDFGraph graph, Namespace namespace, String localNameString );

    /**
     * Returns an object of class '<em>Document</em>'.
     * 
     * @param uriRefString the RDF URI Reference of the Document.
     *        <a href="#uriRefString">The string must be a valid RDF URI Reference</a>.
     * @return an object of class '<em>Document</em>'.
     * @throws URISyntaxException if the <code>uriRefString</code> is not a 
     *         valid RDF URI Reference. 
     */
    Document createDocument( String uriRefString )
    throws URISyntaxException;
    
    /**
     * Returns an object of class '<em>Document</em>'.
     * 
     * @param uriRef the RDF URI Reference of the Document.
     * @return an object of class '<em>Document</em>'.
     */
    Document createDocument( URIReference uriRef );
    
    /**
     * Returns an object of class '<em>Document</em>'.
     * 
     * @param namespace       the namespace of the RDF URI Reference of the Document
     * @param localNameString the fragment identifier of the RDF URI Reference of the Document
     * @return an object of class '<em>Document</em>'.
     */
    Document  createDocument( Namespace namespace, String localNameString );
    
    /**
     * Returns an object of class '<em>LocalName</em>'.
     * 
     * @param name the local name string. The string can not be changed after
     *        the LocalName object is created.
     * @return an object of class '<em>LocalName</em>'.
     */
    LocalName createLocalName( String name );

    /**
     * Returns an object of class '<em>Namespace</em>'.
     * 
     * @param uriRefString  the RDF URI Reference of this namespace which can
     *        not be changed after the namespace object is created.
     *        <a href="#uriRefString">The string must be a valid RDF URI Reference</a>.
     * @return an object of class '<em>Namespace</em>'.
     * @throws URISyntaxException if the <code>uriRefString</code> is not a valid
     *         RDF URI Reference.
     */
    Namespace createNamespace( String uriRefString )
    throws URISyntaxException;
    
    /**
     * Returns an object of class '<em>Namespace</em>'.
     * 
     * @param uriRef the RDF URI Reference of this namespace which can not 
     *        be changed after the namespace object is created.
     * @return an object of class '<em>Namespace</em>'.
     */
    Namespace createNamespace( URIReference uriRef );

    /**
     * Returns an object of class '<em>NamespaceDefinition</em>'.
     * 
     * @param prefix the prefix of the namespace definition. It may be
     *        <code>null</code> and be set later.
     * @param namespace the namespace of this definition. It may be <code>null</code>
     *        and be set later. 
     * @return an object of class '<em>NamespaceDefinition</em>'.
     */
    NamespaceDefinition createNamespaceDefinition( String prefix, Namespace namespace );


}
