/**
 * <copyright> 
 * 
 * Copyright (c) 2004-2005 IBM Corporation and others. 
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License - v 1.0 
 * which accompanies this distribution, and is available at 
 * http://opensource.org/licenses/eclipse-1.0.txt 
 * 
 * Contributors: 
 *   IBM - Initial API and implementation 
 * 
 * </copyright> 
 * 
 * $Id: OWLDocument.java,v 1.2 2007/03/18 09:07:06 lzhang Exp $
 */

package org.eclipse.eodm.owl.resource.parser;

import java.io.InputStream;
import java.util.LinkedList;

import org.eclipse.eodm.owl.resource.parser.exception.OWLParserException;
import org.eclipse.eodm.owl.resource.parser.impl.OWLOntologyGraph;

/**
 * OWLDocument - OWL otology document.
 * 
 * OWL document (local file or web resource with linked uri) describes an OWL
 * file in which all elements and statements of an ontology are defined.
 * 
 */
public interface OWLDocument {
    /**
     * Check if the OWL document is a local file or a web resource with linked
     * uri.
     * 
     * @return true if the OWL document is a local file
     */
    boolean isLocalFile();

    /**
     * Get the document local uri. <br>
     * If the OWL document is a local file, the current file path is used as the
     * document's local uri.
     * 
     * @return the document local uri
     */
    String getLocalURI();

    /**
     * Get the document public uri. <br>
     * If OWL document is a web resource, the web uri is used as the document's
     * public uri.
     * 
     * @return the document public uri
     */
    String getPublicURI();

    /**
     * Get OWL Document charset(encoding).
     * 
     * @return the charset
     */
    String getCharset();

    /**
     * Set the OWL Document charset(encoding).
     * 
     * @param charset
     *            the charset
     */
    void setCharset(String charset);

    /**
     * Get the default base uri.
     * 
     * @return default base uri
     */
    String getDefaultBaseURI();

    /**
     * Set the default base uri.
     * 
     * @param uri
     *            default base uri
     */
    void setDefaultBaseURI(String uri);

    /**
     * Read the OWL document from the input.
     * 
     * @return the input stream
     * @exception OWLParserException
     *                if io error
     */
    InputStream readInput() throws OWLParserException;

    /**
     * Set the public URI of the document
     * 
     * @param URI
     *            the URI of the document
     */
    void setPublicURI(String URI);

    /**
     * Check if the document has been parsed already
     * 
     * @return true if the document is parsed, or false
     */
    boolean isParsed();

    /**
     * Set a flag to denote that the document has been parsed.
     */
    void setParsed();

    /**
     * Get a list of ontology URIs
     * 
     * @param flag
     *            indicate which URIs will be returned.
     * @return if the flag is ture, return URIs of ontologies which are imported
     *         by this document, if false, return URIs of ontologies which
     *         imports this document.
     */
    LinkedList getImportOntologyURIs(boolean flag);

    /**
     * add an ontology URI into the list which are uris imported by this
     * document
     * 
     * @param str
     *            the ontology uri
     */
    void addImportOntologyURIs(String str);

    /**
     * Get an ontology graph which includes document, graph and ontology
     * generated from an OWL document.
     * 
     * @return an ontology graph
     */
    OWLOntologyGraph getOntologyGraph();

    /**
     * Set an ontology graph for an OWL document
     * 
     * @param ontologygraph
     *            an ontology graph
     */
    void setOntologyGraph(OWLOntologyGraph ontologygraph);

    /**
     * Set an input stream for the document
     * 
     * @param input
     *            an input stream
     */
    void setInputStream(InputStream input);
}