/**
 * <copyright> 
 * 
 * Copyright (c) 2004-2005 IBM Corporation and others. 
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License - v 1.0 
 * which accompanies this distribution, and is available at 
 * http://opensource.org/licenses/eclipse-1.0.txt 
 * 
 * Contributors: 
 *   IBM - Initial API and implementation 
 * 
 * </copyright> 
 * 
 * $Id: OWLXMLResourceImpl.java,v 1.3 2007/03/18 09:07:04 lzhang Exp $
 */

package org.eclipse.eodm.owl.resource;

import java.io.BufferedInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.util.Iterator;
import java.util.Map;

import org.eclipse.emf.common.util.URI;
import org.eclipse.emf.ecore.resource.impl.ResourceImpl;
import org.eclipse.emf.ecore.xmi.impl.XMLHandler;
import org.eclipse.eodm.exceptions.UnsupportedViewTypeException;
import org.eclipse.eodm.owl.resource.parser.exception.OWLParserException;
import org.eclipse.eodm.owl.resource.parser.impl.OWLOntologyGraph;
import org.eclipse.eodm.rdf.rdfweb.Document;

/**
 * OWLXMLResourceImpl
 * 
 */
public class OWLXMLResourceImpl extends ResourceImpl implements OWLXMLResource {

    String defaultEncoding = null;

    public OWLXMLResourceImpl() {
        super();
    }

    public OWLXMLResourceImpl(URI uri) {
        super(uri);
    }

    /**
     * Read a buffer for guessing encoding.
     * 
     * @param is
     *            the input stream that supports mark
     * @return a buffer of bytes of the input stream
     * @throws IOException
     */
    protected byte[] readGuessBuffer(InputStream is) throws IOException {
        int BUFFER_SIZE = 200;

        if (is.available() == 0) {
            return new byte[0];
        }

        byte[] buffer = new byte[BUFFER_SIZE];
        is.mark(BUFFER_SIZE);
        int bytesRead = is.read(buffer, 0, BUFFER_SIZE);
        int totalBytesRead = bytesRead;

        while (bytesRead != -1 && (totalBytesRead < BUFFER_SIZE)) {
            bytesRead = is.read(buffer, totalBytesRead, BUFFER_SIZE
                                                        - totalBytesRead);

            if (bytesRead != -1)
                totalBytesRead += bytesRead;
        }

        if (totalBytesRead < BUFFER_SIZE) {
            byte[] smallerBuffer = new byte[totalBytesRead];
            System.arraycopy(buffer, 0, smallerBuffer, 0, totalBytesRead);
            smallerBuffer = buffer;
        }

        is.reset();
        return buffer;
    }

    /**
     * Guess the encoding of the input stream. Note that the input stream must
     * support mark.
     * 
     * @param is
     *            the input stream
     * @return the guessed encoding or null
     * @throws IOException
     */
    protected String guessEncoding(InputStream is) throws IOException {
        if (!is.markSupported())
            return null;
        return XMLHandler.getXMLEncoding(readGuessBuffer(is));
    }

    /**
     * Get the encoding to use from a options map.
     * 
     * @param options
     *            the map that contains all the user-specified resource options
     * @return the name of the encoding specified in the options map or
     *         <code>null</code> if not specified.
     */
    protected String getEncodingOption(Map options) {
        if (options == null)
            return null;
        Object obj = options.get(OWLXMLResource.OPTION_ENCODING);
        if (obj != null)
            if (obj instanceof String)
                return (String) obj;
        return null;
    }

    /**
     * Load an OWL Model from input stream. All OWLDocuments should be set into
     * options.
     * 
     * @param input
     *            the input stream
     * @param options
     *            option set
     * @exception IOException
     *                if io errors
     */
    protected void doLoad(InputStream inputStream, Map options)
            throws IOException {
        String enc = getEncodingOption(options);
        if (enc == null) {
            if (!inputStream.markSupported())
                inputStream = new BufferedInputStream(inputStream);
            enc = guessEncoding(inputStream);
        }
        defaultEncoding = enc; // remember this encoding for save

        try {

            OWLOntologyGraph[] ontograph = OWLXMLLoader.loadFromStream(
                    inputStream, enc);
            if (ontograph != null)
                getContents().add(ontograph);

        } catch (OWLParserException e) {
            IOException ie = new IOException(
                    "Exception in loading OWL Model from stream");
            ie.initCause(e);
            throw ie;
        }
    }

    /**
     * Save an OWL Model to output stream. Following options: 1) The URI of the
     * Ontology to be saved. 2) Encoding of the output stream.
     * 
     * @param output
     *            the output stream
     * @param options
     *            option set
     * @exception IOException
     *                if io errors
     */
    protected void doSave(OutputStream output, Map options) throws IOException {
        String encoding = getEncodingOption(options);
        if (encoding == null) {
            encoding = defaultEncoding;
        }

        Document document = null;
        for (Iterator iterator = getContents().iterator(); iterator.hasNext();) {
            Object obj = iterator.next();
            if (obj instanceof Document) {
                document = (Document) obj;
                break;
            }
        }

        if (document != null) {
            if (encoding == null)
                try {
                    OWLXMLSaver.saveToStream(document, output);
                } catch (UnsupportedViewTypeException e) {
                    // TODO Auto-generated catch block
                    e.printStackTrace();
                }
            else
                try {
                    OWLXMLSaver.saveToStream(document, output, encoding);
                } catch (UnsupportedViewTypeException e) {
                    // TODO Auto-generated catch block
                    e.printStackTrace();
                }
        }
    }
}