/*
 * Decompiled with CFR 0.152.
 */
package org.glassfish.main.jul.rotation;

import java.io.BufferedOutputStream;
import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.nio.charset.Charset;
import java.nio.file.CopyOption;
import java.nio.file.Files;
import java.nio.file.StandardCopyOption;
import java.time.LocalDateTime;
import java.time.format.DateTimeFormatter;
import org.glassfish.main.jul.rotation.LogFileArchiver;
import org.glassfish.main.jul.rotation.MeteredFileWriter;
import org.glassfish.main.jul.rotation.MeteredStream;
import org.glassfish.main.jul.tracing.GlassFishLoggingTracer;

public class LogFileManager {
    private static final System.Logger LOG = System.getLogger(LogFileManager.class.getName());
    private static final DateTimeFormatter SUFFIX_FORMATTER = DateTimeFormatter.ofPattern("yyyy-MM-dd'T'HH-mm-ss");
    private final File logFile;
    private final LogFileArchiver archiver;
    private final Charset fileEncoding;
    private final long maxFileSize;
    private MeteredFileWriter writer;

    public LogFileManager(File logFile, Charset fileEncoding, long maxFileSize, boolean compressOldLogFiles, int maxCountOfOldLogFiles) {
        this.logFile = logFile;
        this.fileEncoding = fileEncoding;
        this.maxFileSize = maxFileSize;
        this.archiver = new LogFileArchiver(logFile, compressOldLogFiles, maxCountOfOldLogFiles);
    }

    public synchronized void write(String text) throws IllegalStateException {
        if (!this.isOutputEnabled()) {
            throw new IllegalStateException("The file output is disabled!");
        }
        try {
            this.writer.write(text);
        }
        catch (Exception e) {
            GlassFishLoggingTracer.error(this.getClass(), "Could not write to the output stream.", e);
        }
    }

    public synchronized void flush() {
        if (this.isOutputEnabled()) {
            try {
                this.writer.flush();
            }
            catch (IOException e) {
                GlassFishLoggingTracer.error(this.getClass(), "Could not flush the writer.", e);
            }
        }
        this.rollIfFileTooBig();
    }

    public synchronized long getFileSize() {
        return this.writer == null ? this.logFile.length() : this.writer.getBytesWritten();
    }

    public synchronized void rollIfFileTooBig() {
        if (this.isRollFileSizeLimitReached()) {
            this.roll();
        }
    }

    public synchronized void rollIfFileNotEmpty() {
        if (this.getFileSize() > 0L) {
            this.roll();
        }
    }

    public synchronized void roll() {
        boolean wasOutputEnabled = this.isOutputEnabled();
        LOG.log(System.Logger.Level.DEBUG, "Rolling the file {0}; output was originally enabled: {1}", this.logFile, wasOutputEnabled);
        this.disableOutput();
        LOG.log(System.Logger.Level.DEBUG, "Output disabled for now.");
        File archivedFile = this.rollToNewFile();
        LOG.log(System.Logger.Level.INFO, "Archived file: {0} - if null, action failed.", archivedFile);
        if (wasOutputEnabled) {
            this.enableOutput();
            LOG.log(System.Logger.Level.DEBUG, "Output to {0} enabled again.", this.logFile);
        }
        if (archivedFile != null) {
            this.archiver.archive(archivedFile);
        }
    }

    public synchronized boolean isOutputEnabled() {
        return this.writer != null;
    }

    public synchronized void enableOutput() {
        if (this.isOutputEnabled()) {
            throw new IllegalStateException("Output is already enabled!");
        }
        File parent = this.logFile.getParentFile();
        if (parent != null && !parent.exists() && !parent.mkdirs()) {
            throw new IllegalStateException("Failed to create the parent directory " + parent.getAbsolutePath());
        }
        try {
            FileOutputStream fout = new FileOutputStream(this.logFile, true);
            BufferedOutputStream bout = new BufferedOutputStream(fout);
            MeteredStream stream = new MeteredStream(bout, this.logFile.length());
            this.writer = new MeteredFileWriter(stream, this.fileEncoding);
        }
        catch (Exception e) {
            throw new IllegalStateException("Could not open the log file for writing: " + this.logFile, e);
        }
    }

    public synchronized void disableOutput() {
        if (!this.isOutputEnabled()) {
            return;
        }
        try {
            LOG.log(System.Logger.Level.DEBUG, "Closing writer: {0}", this.writer);
            this.writer.close();
        }
        catch (IOException e) {
            GlassFishLoggingTracer.error(this.getClass(), "Could not close the output stream.", e);
        }
        this.writer = null;
    }

    private boolean isRollFileSizeLimitReached() {
        if (this.maxFileSize <= 0L) {
            return false;
        }
        long fileSize = this.getFileSize();
        return fileSize >= this.maxFileSize;
    }

    private File rollToNewFile() {
        try {
            if (this.logFile.createNewFile()) {
                LOG.log(System.Logger.Level.DEBUG, "Created new log file: {0}", this.logFile);
                return null;
            }
            LOG.log(System.Logger.Level.DEBUG, "Rolling log file: {0}", this.logFile);
            File archivedLogFile = this.prepareAchivedLogFileTarget();
            this.moveFile(this.logFile, archivedLogFile);
            this.forceOSFilesync(this.logFile);
            return archivedLogFile;
        }
        catch (Exception e) {
            this.logError("Error, could not rotate log file", e);
            return null;
        }
    }

    private File prepareAchivedLogFileTarget() {
        String archivedFileNameBase = this.logFile.getName() + "_" + SUFFIX_FORMATTER.format(LocalDateTime.now());
        int counter = 1;
        String archivedFileName = archivedFileNameBase;
        while (true) {
            File archivedLogFile = new File(this.logFile.getParentFile(), archivedFileName);
            File archivedGzLogFile = this.archiver.getGzArchiveFile(archivedLogFile);
            if (!archivedLogFile.exists() && !archivedGzLogFile.exists()) {
                return archivedLogFile;
            }
            archivedFileName = archivedFileNameBase + "_" + ++counter;
        }
    }

    private void forceOSFilesync(File file) throws IOException {
        new FileOutputStream(file).close();
    }

    private void moveFile(File logFileToArchive, File target) throws IOException {
        LOG.log(System.Logger.Level.DEBUG, "moveFile(logFileToArchive={0}, target={1})", logFileToArchive, target);
        try {
            Files.move(logFileToArchive.toPath(), target.toPath(), StandardCopyOption.ATOMIC_MOVE);
        }
        catch (IOException | UnsupportedOperationException e) {
            this.logError(String.format("File %s could not be renamed to %s atomically, now trying to move it without this request.", logFileToArchive, target), e);
            Files.move(logFileToArchive.toPath(), target.toPath(), new CopyOption[0]);
        }
    }

    private void logError(String message, Throwable t) {
        GlassFishLoggingTracer.error(this.getClass(), message, t);
        LOG.log(System.Logger.Level.ERROR, message, t);
    }
}

