/*
 * Decompiled with CFR 0.152.
 */
package org.glassfish.web.loader;

import com.sun.enterprise.util.io.FileUtils;
import java.io.Closeable;
import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.net.MalformedURLException;
import java.net.URL;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReadWriteLock;
import java.util.concurrent.locks.ReentrantReadWriteLock;
import java.util.jar.JarEntry;
import java.util.jar.JarFile;
import org.glassfish.web.loader.LogFacade;
import org.glassfish.web.loader.ResourceEntry;

class JarFileManager
implements Closeable {
    private static final int SECONDS_TO_CLOSE_UNUSED_JARS = Integer.getInteger("org.glassfish.web.loader.unusedJars.secondsToClose", 60);
    private static final int SECONDS_TO_CHECK_UNUSED_JARS = Integer.getInteger("org.glassfish.web.loader.unusedJars.secondsToRunCheck", 15);
    private static final System.Logger LOG = System.getLogger(JarFileManager.class.getName());
    private final List<JarResource> files = new ArrayList<JarResource>();
    private final ScheduledExecutorService scheduler = Executors.newScheduledThreadPool(1, new JarFileManagerThreadFactory());
    private final ReadWriteLock lock = new ReentrantReadWriteLock();
    private final Lock readLock = this.lock.readLock();
    private final Lock writeLock = this.lock.writeLock();
    private volatile long lastJarFileAccess;
    private ScheduledFuture<?> unusedJarsCheck;
    private volatile boolean resourcesExtracted;

    JarFileManager() {
    }

    void addJarFile(File file) {
        this.writeLock.lock();
        try {
            this.files.add(new JarResource(file));
        }
        finally {
            this.writeLock.unlock();
        }
    }

    JarFile[] getJarFiles() {
        if (!this.isJarsOpen() && !this.openJARs()) {
            return null;
        }
        this.readLock.lock();
        try {
            this.lastJarFileAccess = System.currentTimeMillis();
            JarFile[] jarFileArray = (JarFile[])this.files.stream().map(r -> r.jarFile).toArray(JarFile[]::new);
            return jarFileArray;
        }
        finally {
            this.readLock.unlock();
        }
    }

    File[] getJarRealFiles() {
        this.readLock.lock();
        try {
            File[] fileArray = (File[])this.files.stream().map(r -> r.file).toArray(File[]::new);
            return fileArray;
        }
        finally {
            this.readLock.unlock();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    ResourceEntry findResource(String name, String path, File loaderDir, boolean antiJARLocking) {
        LOG.log(System.Logger.Level.TRACE, "findResource(name={0}, path={1}, loaderDir={2}, antiJARLocking={3})", name, path, loaderDir, antiJARLocking);
        if (!this.isJarsOpen() && !this.openJARs()) {
            return null;
        }
        this.readLock.lock();
        try {
            this.lastJarFileAccess = System.currentTimeMillis();
            for (JarResource jarResource : this.files) {
                JarFile jarFile = jarResource.jarFile;
                JarEntry jarEntry = jarFile.getJarEntry(path);
                if (jarEntry == null) continue;
                ResourceEntry entry = this.createResourceEntry(name, jarResource.file, jarFile, jarEntry, path);
                if (entry == null) {
                    ResourceEntry resourceEntry = null;
                    return resourceEntry;
                }
                if (antiJARLocking && !path.endsWith(".class")) {
                    File resourceFile = new File(loaderDir, jarEntry.getName());
                    if (!this.resourcesExtracted && !resourceFile.exists()) {
                        this.extractResources(loaderDir, path);
                    }
                }
                ResourceEntry resourceEntry = entry;
                return resourceEntry;
            }
        }
        finally {
            this.readLock.unlock();
        }
        return null;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    void extractResources(File loaderDir, String canonicalLoaderDir) {
        LOG.log(System.Logger.Level.DEBUG, "extractResources(loaderDir={0}, canonicalLoaderDir={1})", loaderDir, canonicalLoaderDir);
        if (this.resourcesExtracted) {
            return;
        }
        this.readLock.lock();
        try {
            for (JarResource jarResource : this.files) {
                JarFileManager.extractResource(jarResource.jarFile, loaderDir, canonicalLoaderDir);
            }
        }
        finally {
            this.readLock.unlock();
        }
        this.resourcesExtracted = true;
    }

    void closeJarFiles() {
        LOG.log(System.Logger.Level.DEBUG, "closeJarFiles()");
        this.writeLock.lock();
        try {
            this.lastJarFileAccess = 0L;
            JarFileManager.closeJarFiles(this.files);
        }
        finally {
            if (this.unusedJarsCheck != null) {
                this.unusedJarsCheck.cancel(false);
            }
            this.writeLock.unlock();
        }
    }

    @Override
    public void close() throws IOException {
        this.closeJarFiles();
        this.scheduler.shutdown();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private boolean openJARs() {
        LOG.log(System.Logger.Level.DEBUG, "openJARs()");
        this.writeLock.lock();
        try {
            if (this.isJarsOpen()) {
                boolean bl = true;
                return bl;
            }
            this.lastJarFileAccess = System.currentTimeMillis();
            for (JarResource jarResource : this.files) {
                if (jarResource.jarFile != null) continue;
                try {
                    jarResource.jarFile = new JarFile(jarResource.file, true, 1, Runtime.version());
                }
                catch (IOException e) {
                    LOG.log(System.Logger.Level.DEBUG, "Failed to open JAR", (Throwable)e);
                    this.lastJarFileAccess = 0L;
                    JarFileManager.closeJarFiles(this.files);
                    boolean bl = false;
                    this.writeLock.unlock();
                    return bl;
                }
            }
            LOG.log(System.Logger.Level.DEBUG, "JAR files are open. If unused, will be closed after {0} s", SECONDS_TO_CLOSE_UNUSED_JARS);
            this.unusedJarsCheck = this.scheduler.scheduleAtFixedRate(this::closeJarFilesIfNotUsed, SECONDS_TO_CHECK_UNUSED_JARS, SECONDS_TO_CHECK_UNUSED_JARS, TimeUnit.SECONDS);
            boolean bl = true;
            return bl;
        }
        finally {
            this.writeLock.unlock();
        }
    }

    private boolean isJarsOpen() {
        return this.lastJarFileAccess > 0L;
    }

    private ResourceEntry createResourceEntry(String name, File file, JarFile jarFile, JarEntry jarEntry, String entryPath) {
        URL source;
        URL codeBase;
        try {
            codeBase = file.getCanonicalFile().toURI().toURL();
        }
        catch (IOException e) {
            LOG.log(System.Logger.Level.WARNING, "Invalid file: " + file, (Throwable)e);
            return null;
        }
        try {
            source = new URL("jar:" + codeBase + "!/" + entryPath);
        }
        catch (MalformedURLException e) {
            LOG.log(System.Logger.Level.WARNING, "Cannot create valid URL of file " + file + " and entry path " + entryPath, (Throwable)e);
            return null;
        }
        ResourceEntry entry = new ResourceEntry(codeBase, source);
        try {
            entry.manifest = jarFile.getManifest();
        }
        catch (IOException e) {
            LOG.log(System.Logger.Level.WARNING, "Failed to get manifest from " + jarFile.getName(), (Throwable)e);
            return null;
        }
        entry.lastModified = file.lastModified();
        int contentLength = (int)jarEntry.getSize();
        try (InputStream binaryStream = jarFile.getInputStream(jarEntry);){
            if (binaryStream != null) {
                entry.readEntryData(name, binaryStream, contentLength, jarEntry);
            }
        }
        catch (IOException e) {
            LOG.log(System.Logger.Level.WARNING, "Failed to read entry data for " + name, (Throwable)e);
            return null;
        }
        return entry;
    }

    private static void extractResource(JarFile jarFile, File loaderDir, String pathPrefix) {
        LOG.log(System.Logger.Level.DEBUG, "extractResource(jarFile={0}, loaderDir={1}, pathPrefix={2})", jarFile, loaderDir, pathPrefix);
        Iterator jarEntries = jarFile.versionedStream().filter(jarEntry -> !jarEntry.isDirectory() && !jarEntry.getName().endsWith(".class")).iterator();
        while (jarEntries.hasNext()) {
            JarEntry jarEntry2 = (JarEntry)jarEntries.next();
            File resourceFile = new File(loaderDir, jarEntry2.getName());
            try {
                if (!resourceFile.getCanonicalPath().startsWith(pathPrefix)) {
                    throw new IllegalArgumentException(LogFacade.getString("AS-WEB-UTIL-00021", jarEntry2.getName()));
                }
            }
            catch (IOException ioe) {
                throw new IllegalArgumentException(LogFacade.getString("AS-WEB-UTIL-00022", jarEntry2.getName()), ioe);
            }
            if (!FileUtils.mkdirsMaybe(resourceFile.getParentFile())) {
                LOG.log(System.Logger.Level.WARNING, "AS-WEB-UTIL-00023", resourceFile.getParentFile());
            }
            try {
                InputStream is = jarFile.getInputStream(jarEntry2);
                try (FileOutputStream os = new FileOutputStream(resourceFile);){
                    FileUtils.copy(is, os, Long.MAX_VALUE);
                }
                finally {
                    if (is == null) continue;
                    is.close();
                }
            }
            catch (IOException e) {
                LOG.log(System.Logger.Level.WARNING, "Failed to copy entry " + jarEntry2, (Throwable)e);
            }
        }
    }

    private void closeJarFilesIfNotUsed() {
        if (!this.isJarsOpen()) {
            return;
        }
        long unusedFor = (System.currentTimeMillis() - this.lastJarFileAccess) / 1000L;
        if (unusedFor <= (long)SECONDS_TO_CLOSE_UNUSED_JARS) {
            return;
        }
        LOG.log(System.Logger.Level.DEBUG, "Closing jar files, because they were not used for {0} s.", unusedFor);
        this.closeJarFiles();
    }

    private static void closeJarFiles(List<JarResource> files) {
        for (JarResource jarResource : files) {
            if (jarResource.jarFile == null) continue;
            JarFile toClose = jarResource.jarFile;
            jarResource.jarFile = null;
            JarFileManager.closeJarFile(toClose);
        }
        LOG.log(System.Logger.Level.DEBUG, "JAR files were closed.");
    }

    private static void closeJarFile(JarFile jarFile) {
        try {
            jarFile.close();
        }
        catch (IOException e) {
            LOG.log(System.Logger.Level.WARNING, "Could not close the jarFile " + jarFile, (Throwable)e);
        }
    }

    private static class JarFileManagerThreadFactory
    implements ThreadFactory {
        private int counter = 1;

        private JarFileManagerThreadFactory() {
        }

        @Override
        public Thread newThread(Runnable runnable) {
            Thread thread = new Thread(runnable, "JarFileManager-" + this.counter++);
            thread.setDaemon(true);
            thread.setPriority(1);
            return thread;
        }
    }

    private static class JarResource {
        final File file;
        JarFile jarFile;

        JarResource(File file) {
            this.file = file;
        }
    }
}

