/*******************************************************************************
 * Copyright (c) 2004, 2005 Sybase, Inc. and others.
 *
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     Sybase, Inc. - initial API and implementation
 *******************************************************************************/
package org.eclipse.datatools.sqltools.sql.parser;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Iterator;
import java.util.List;

import org.eclipse.datatools.modelbase.sql.datatypes.UserDefinedType;

/**
 * This class knows how to convert information generated by a SQL parser into
 * completion proposals for content assist. Vendor extenders may need to use
 * different advisors based on their parser generator technoligy and grammar.
 * 
 * @author Hui Cao
 * 
 */
public class ParserProposalAdvisor {

	private final static String[] EMPTY = new String[0];
	public String getLocalVariablePrefix()
	{
		return "@";
	}
	
	public String getGlobalVariablePrefix()
	{
		return "@@";
	}
	
	/**
	 * Tells whether the given token represents a local variable token definition.
	 */
	public boolean isLocalVariableTokenDefinition(String token) {
		return token != null && token.equalsIgnoreCase("<VAR_NAME>");
	}

	/**
	 * Tells whether the given token represents a global variable token definition.
	 */
	public boolean isGlobalVariableTokenDefinition(String token) {
		return token != null && token.equalsIgnoreCase("<GLOBAL_VAR_NAME>");
	}

	/**
	 * Tells whether the given token represents an identifier token definition.
	 */
	public boolean isIdentifierTokenDefinition(String token) {
		return token != null && token.equalsIgnoreCase("<ID>");
	}
	
	/**
	 * Tells whether the given token represents a token definition.
	 */
	public boolean isTokenDefinition(String token) {
		return token.startsWith("<") && token.endsWith(">") || token.charAt(0) == '"';
	}
	
	/**
	 * Tests whether the given string represents a token name
	 */
	public boolean isTokenName(String token) {
		boolean isName = token != null
				&& (token.startsWith("<") && token.endsWith(">"));
		isName = isName || (token != null && token.charAt(0) == '"');
		return isName;
	}

	public String[] getParserProposals(ParsingResult result) {
        if (result.getExceptions() == null
                || result.getExceptions().size() == 0) {
            return EMPTY;
        } else {
            List list = new ArrayList();
            // provide only the deepest lookahead tokens
            // 205684: contributed by Charles Eutsler
            int lookaheadDepth = 1;

            for (int i = 0; i < result.getExceptions().size(); i++) {
                if (result.getExceptions().get(i) instanceof ParseException) {
                    ParseException pe = (ParseException) result.getExceptions()
                            .get(i);
                    if (pe.expectedTokenSequences == null) {
                        continue;
                    }
                    for (int j = 0; j < pe.expectedTokenSequences.length; j++)
{
                        String image;
                        int[] sequence = pe.expectedTokenSequences[j];
                        if (sequence.length < lookaheadDepth) {
                            // this is for another lookahead path
                            continue;
                        }
                        if (sequence.length > lookaheadDepth) {
                            // what we've found so far are not applicable
                            list = new ArrayList();
                            lookaheadDepth = sequence.length;
                        }
                        // get the last token in the array
                        image = pe.tokenImage[sequence[sequence.length - 1]];
                        String expected = removeQuotes(image);
                        if (!list.contains(expected)) {
                            list.add(expected);
                        }
                    }
                }
            }
            return (String[]) list.toArray(new String[list.size()]);

		}
	}

	private String removeQuotes(String string) {
		StringBuffer sb = new StringBuffer();
		for (int i = 0; i < string.length(); i++) {
			char c = string.charAt(i);
			if (c != '"' && c != '\\') {
				sb.append(c);
			} else if (c == '\\' && string.charAt(i + 1) == '"') {
				sb.append('"');
				i = i + 1;
			}

		}
		return sb.toString();
	}

	/**
	 * Returns whether the proposal list contains datatype. The default
	 * implementation uses a simple algorithm: as long as the proposals contains
	 * "char", "varchar" and "int".
	 * 
	 * @param config
	 * @param result
	 * @return list the user defined datatype name list.
	 */
	public boolean containsDataTypeProposals(String[] parserProposals,
			Collection unreservedKeywords) {
		int count = 0;
		for (int i = 0; i < parserProposals.length; i++) {
			String proposal = parserProposals[i].toLowerCase();
			if (proposal.equals("char") || proposal.equals("varchar")
					|| proposal.equals("int")) {
				count++;
				if (count == 3) {
					break;
				}
			}
		}

		if (count == 0) {
			for (Iterator iter = unreservedKeywords.iterator(); iter.hasNext();) {
				String proposal = (String) iter.next();
				if (proposal.equals("char") || proposal.equals("varchar")
						|| proposal.equals("int")) {
					count++;
					if (count == 3) {
						break;
					}
				}
			}
		}

		return count == 3;
	}

	/**
	 * Tells whether to accept the user defined data type in the content assist.
	 * Default implementation simply returns true;
	 * 
	 * @param udts
	 */
	public boolean acceptsUserDefinedDataType(UserDefinedType udt) {
		return true;
	}

}
