/** \file annotator.hpp .
-----------------------------------------------------------------------------



 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.

-----------------------------------------------------------------------------

    \brief  Contains Annotatorproxy an Annotator class

   Description:

-----------------------------------------------------------------------------


   4/14/1999   Initial creation
   8/03/1999   Language handling added
   9/06/1999   Type enabling/disabling added
   1/17/2000   Autom. priorisation of annotators added

-------------------------------------------------------------------------- */

#ifndef UIMA_ANNOTATOR_HPP
#define UIMA_ANNOTATOR_HPP

/* ----------------------------------------------------------------------- */
/*       Interface dependencies                                            */
/* ----------------------------------------------------------------------- */

#include "uima/pragmas.hpp" //must be included first to disable warnings
#include "uima/err_ids.h"
#include "uima/language.hpp"
#include "uima/annotator_timing.hpp"
#include "uima/cas.hpp"
#include "uima/types.h"
#include "uima/annotator_abase.hpp"

/* ----------------------------------------------------------------------- */
/*       Implementation dependencies                                       */
/* ----------------------------------------------------------------------- */

#include <map>

#include "uima/exceptions.hpp"
#include "uima/strtools.hpp"
#include "uima/timedatetools.hpp"

// temp, for SIngleProcesingUnit
#include "uima/engine.hpp"

#include "uima/annotator_context.hpp"

/* ----------------------------------------------------------------------- */
/*       Constants                                                         */
/* ----------------------------------------------------------------------- */

/* ----------------------------------------------------------------------- */
/*       Forward declarations                                              */
/* ----------------------------------------------------------------------- */

namespace uima {
  namespace internal {
    class AnnotatorFile;
  }
  class AnnotatorContext;
  class Annotator;
  namespace util {
    class Trace;
  }
}


/* ----------------------------------------------------------------------- */
/*       Types / Classes                                                   */
/* ----------------------------------------------------------------------- */

#ifdef UIMA_DEBUG_ANNOTATOR_TIMING
#  ifdef UIMA_ANNOTATOR_TIMING
#     undef UIMA_ANNOTATOR_TIMING
#  endif
/** A macro in which our timing statements are hidden,
   if UIMA_DEBUG_ANNOTATOR_TIMING is <EM>not</EM> defined.
   Using a timer <TT>clTimer</TT> of type <TT>Timer</TT> the macro can be used
   in statements like:
   <TT>UIMA_ANNOTATOR_TIMING(clTimer.start)</TT> and <TT>UIMA_ANNOTATOR_TIMING(clTimer.start)</TT>
*/
#  define UIMA_ANNOTATOR_TIMING( expr ) ( expr )
#else
#  ifdef UIMA_ANNOTATOR_TIMING
#     undef UIMA_ANNOTATOR_TIMING
#  endif
#  define UIMA_ANNOTATOR_TIMING( ignore ) ( ( void )0 )
#endif

namespace uima {
  UIMA_EXC_CLASSDECLARE(ExcEnumerationOverflow, ExcIllFormedInputError);



  /**
   * The class <TT>AnnotatorProxy</TT> represents an Annotator.
   * An AnnotatorProxy is a user-defined DLL which interacts with UIMACPP.
   * For a description of the Annotator architecture and Annotator
   * communication protocol, refer to the corresponding design document.
   * \code
     \endcode
   * @see
   */
  class UIMA_LINK_IMPORTSPEC AnnotatorProxy {
  public:
    /** @name Constructors */
    /*@{*/
    /** Create a new annotator with the specified filename denoting the UimaAnnotatorFile,
        the symbolic name used in the configuration, a target set for the
        AT types that should be generated by this annotator, and a target set of
        TT types that should be generated by this annotator. */
    AnnotatorProxy(const util::Filename & crclFilename,
                   const TCHAR * cpszSymbolicName,
                   bool  bDelayLoad = false);
    /*@}*/
    ~AnnotatorProxy(void);
    /** @name Properties */
    /*@{*/
    /** Return TRUE, if the annotator is valid and could be activated. */
    bool                       isValid(void) const;
    /** Return TRUE, if the annotator requires deinitialisation. */
    bool                       isInitialized(void) const;
    /** Return the complete filename of this annotator. */
    const util::Filename &     getFilename(void) const;
    /** Return the name of the annotator. */
    const std::string &             getSymbolicName(void) const                  {
      return(iv_strSymbolicName);
    }
    /** Return the detailed error message if the load() fails. */
    const char*                getErrorMsg() const;

    /** @name Miscellaneous */
    /*@{*/
    /** Load the annotator and return an error code. */
    TyErrorId                  load(void);
    /** Unload the annotator from memory. */
    void                       unload(void);
    /*@}*/
    /** @name Delayed load support */
    /*@{*/
    /** Return TRUE, if this annotator is supposed to load delayed. */
    bool                       isLoadedDelayed(void) const {
      return false;
    }
    /** Load the annotator delayed and return an error code. */
    TyErrorId                  loadDelayed(void);
    /*@}*/
    /** @name Annotator Functions */
    /*@{*/
    /** Call the Annotator to initialize itself based on an engine
        and a Configuration section, and return an error code. */
    TyErrorId                  init(AnnotatorContext &);
    TyErrorId                  typeSystemInit(uima::TypeSystem const &);
    /** Call the Annotator to deinitialize itself based on an engine,
        and return an error code. */
    TyErrorId                  deInit(void);
    /** Call the Annotator to reconfigure itself based on an Configuration
        section, and return an error code. */
    TyErrorId                  reConfig();
    /** Call the Annotator to perform its document-related duty based on an engine,
        and return an error code. */
    TyErrorId                  processDocument(CAS & cas, ResultSpecification const &);
    /** Call the Annotator to do batch processing
        and return an error code. */
    TyErrorId                  batchProcessComplete(void);
    /** Call the Annotator to do batch processing
        and return an error code. */
    TyErrorId                  collectionProcessComplete(void);

    /** Call the Annotator hasNext
        and return  true or false */
    bool        hasNext(void);
    /** Call the Annotator next and return a CAS */
    CAS &       next(void);

    /** Call the Annotator getRequiredCasInstances.
        This will determine the size of the CASPool for this analysis component. */
    int       getCasInstancesRequired(void);


    /*@}*/
#ifdef UIMA_DEBUG_ANNOTATOR_TIMING
    /** @name Timer Access Functions
       Note: The time returned accumulates over all calls to the functions.
       So getProcessDocumentTime() will return the time spent processing
       documents since the instance (this AnnotatorProxy object), was
       created.
    */
    /*@{*/
    /** Return the time spent in load(). */
    const Timer &              getLoadTimer(void) const                     {
      return(iv_clTimerLoad);
    }
    /** Return the time spent in init(). */
    const Timer &              getInitTimer(void) const                     {
      return(iv_clTimerInit);
    }
    /** Return the time spent in deInit(). */
    const Timer &              getDeInitTimer(void) const                   {
      return(iv_clTimerDeInit);
    }
    /** Return the time spent in config(). */
    const Timer &              getConfigTimer(void) const                   {
      return(iv_clTimerConfig);
    }
    /** Return the time spent in processDocument(). */
    const Timer &              getProcessDocumentTimer(void) const          {
      return(iv_clTimerProcessDocument);
    }
    /** Return the time spent in batchProcessComplete(). */
    const Timer &              getBatchProcessCompleteTimer(void) const          {
      return(iv_clTimerBatchProcessComplete);
    }
    /** Return the time spent in batchProcessComplete(). */
    const Timer &              getCollectionProcessCompleteTimer(void) const          {
      return(iv_clTimerCollectionProcessComplete);
    }
    /** Return the time spent in hasNext(). */
    const Timer &              hasNextTimer(void) const          {
      return(iv_clTimerHasNext);
    }
    /** Return the time spent in next(). */
    const Timer &              nextTimer(void) const          {
      return(iv_clTimerNext);
    }




    /*@}*/
#endif
  protected:
    /* --- functions --- */
  private:
    util::Filename             iv_clFilename;
    std::string                     iv_strSymbolicName;
    /** const  uima::internal::AnnotatorFile **/
    util::DllProcLoaderFile *  iv_cpclFile;
    bool                       iv_bProcessUnspecifiedTerritories;
    /**uima::TyHandle**/
    uima::Annotator *              iv_hUserData;
    bool                       iv_bDelayLoad;

    AnnotatorContext * iv_pAnnotatorContext;




#ifdef UIMA_DEBUG_ANNOTATOR_TIMING
    Timer                      iv_clTimerLoad;
    Timer                      iv_clTimerInit;
    Timer                      iv_clTimerDeInit;
    Timer                      iv_clTimerConfig;
    Timer                      iv_clTimerProcessDocument;
    Timer                      iv_clTimerBatchProcessComplete;
    Timer                      iv_clTimerCollectionProcessComplete;
    Timer                      iv_clTimerHasNext;
    Timer                      iv_clTimerNext;

#endif
    /* --- functions --- */
//   bool                       isTarget(const StAnnotatorTargetSetsPtr cpstTargets, icu::UnicodeString const & crclType) const;
    TyErrorId                  init_now(AnnotatorContext &);
    TyErrorId                  init_delayed(AnnotatorContext &);
    TyErrorId                  doProcessDocument(CAS &, AnnotatorContext &, ResultSpecification const &);
#ifndef NDEBUG
    void                       debugIndicateProcessHasBeenPerformed(CAS const &, AnnotatorContext &, TyErrorId utErrorId) const;
#endif
    /* BASE CONSTRUCTOR NOT SUPPORTED */
    AnnotatorProxy(void); //lint !e1704
    /* COPY CONSTRUCTOR NOT SUPPORTED */
    AnnotatorProxy(const AnnotatorProxy & ); //lint !e1704
    /* ASSIGNMENT OPERATOR NOT SUPPORTED */
    AnnotatorProxy & operator=(const AnnotatorProxy & crclObject);
  }
  ;                                                 /* AnnotatorProxy */


  typedef  Annotator* (* /**UIMA_ANNOTATOR_LINK_IMPORTSPEC**/ TyAnnotatorSignatureMakeAE)(void);

}

/* ----------------------------------------------------------------------- */
/*       Implementation                                                    */
/* ----------------------------------------------------------------------- */

namespace uima {

  inline bool AnnotatorProxy::isInitialized(void) const
  /* ----------------------------------------------------------------------- */
  {
    return((bool) (iv_hUserData != 0));
  }


}

#endif /* UIMA_ANNOTATOR_HPP */

/* <EOF> */

