/*
 * Copyright 1999,2004 The Apache Software Foundation.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *      http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 */
package org.apache.catalina.tribes.group.interceptors;

import org.apache.catalina.tribes.ChannelMessage;
import org.apache.catalina.tribes.Member;
import org.apache.catalina.tribes.group.ChannelInterceptorBase;
import org.apache.catalina.tribes.membership.MemberImpl;
import org.apache.catalina.tribes.membership.Membership;
import java.util.Arrays;

/**
 * <p>Title: Member domain filter interceptor </p>
 *
 * <p>Description: Filters membership based on domain.
 * </p>
 *
 * @author Filip Hanik
 * @version 1.0
 */
public class DomainFilterInterceptor extends ChannelInterceptorBase {

    private static org.apache.juli.logging.Log log = org.apache.juli.logging.LogFactory.getLog( DomainFilterInterceptor.class );

    protected Membership membership = null;
    
    protected byte[] domain = new byte[0];

    public void messageReceived(ChannelMessage msg) {
        //should we filter incoming based on domain?
        super.messageReceived(msg);
    }//messageReceived


    public void memberAdded(Member member) {
        if ( membership == null ) setupMembership();
        boolean notify = false;
        synchronized (membership) {
            notify = Arrays.equals(domain,member.getDomain());
            if ( notify ) notify = membership.memberAlive((MemberImpl)member);
        }
        if ( notify ) super.memberAdded(member);
    }

    public void memberDisappeared(Member member) {
        if ( membership == null ) setupMembership();
        boolean notify = false;
        synchronized (membership) {
            notify = Arrays.equals(domain,member.getDomain());
            membership.removeMember((MemberImpl)member);
        }
        if ( notify ) super.memberDisappeared(member);
    }

    public boolean hasMembers() {
        if ( membership == null ) setupMembership();
        return membership.hasMembers();
    }

    public Member[] getMembers() {
        if ( membership == null ) setupMembership();
        return membership.getMembers();
    }

    public Member getMember(Member mbr) {
        if ( membership == null ) setupMembership();
        return membership.getMember(mbr);
    }

    public Member getLocalMember(boolean incAlive) {
        return super.getLocalMember(incAlive);
    }


    protected synchronized void setupMembership() {
        if ( membership == null ) {
            membership = new Membership((MemberImpl)super.getLocalMember(true));
        }

    }

    public byte[] getDomain() {
        return domain;
    }

    public void setDomain(byte[] domain) {
        this.domain = domain;
    }
}
