/*
 * $Id: CachingTilesContainerTest.java 1045365 2010-12-13 20:46:46Z apetrelli $
 *
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package org.apache.tiles.impl.mgmt;

import static org.easymock.EasyMock.*;
import static org.easymock.classextension.EasyMock.*;
import static org.junit.Assert.*;

import java.util.HashMap;
import java.util.Map;

import org.apache.tiles.Definition;
import org.apache.tiles.TilesContainer;
import org.apache.tiles.definition.NoSuchDefinitionException;
import org.apache.tiles.request.Request;
import org.junit.Before;
import org.junit.Test;

/**
 * Tests {@link CachingTilesContainer}.
 *
 * @version $Rev: 1045365 $ $Date: 2010-12-13 21:46:46 +0100 (Mon, 13 Dec 2010) $
 */
public class CachingTilesContainerTest {

    /**
     * The default name of the attribute in which storing custom definitions.
     */
    private static final String DEFAULT_DEFINITIONS_ATTRIBUTE_NAME =
        "org.apache.tiles.impl.mgmt.DefinitionManager.DEFINITIONS";

    /**
     * The wrapped Tiles container.
     */
    private TilesContainer wrapped;

    /**
     * The Tiles container to test.
     */
    private CachingTilesContainer container;

    /**
     * Sets up the test.
     */
    @Before
    public void setUp() {
        wrapped = createMock(TilesContainer.class);
        container = new CachingTilesContainer(wrapped);
    }

    /**
     * Test method for
     * {@link CachingTilesContainer#CachingTilesContainer(TilesContainer, String)}
     * .
     */
    @SuppressWarnings("unchecked")
    @Test
    public void testCachingTilesContainerTilesContainerString() {
        TilesContainer wrapped = createMock(TilesContainer.class);
        Request request = createMock(Request.class);
        Map<String, Definition> definitions = createMock(Map.class);
        Map<String, Object> scope = createMock(Map.class);
        Definition definition = createMock(Definition.class);

        expect(request.getContext("request")).andReturn(scope);
        expect(scope.get("myattribute")).andReturn(definitions);
        expect(definitions.get("definition")).andReturn(definition);

        replay(wrapped, request, definitions, scope, definition);
        CachingTilesContainer container = new CachingTilesContainer(wrapped,
                "myattribute");
        assertSame(definition, container.getDefinition("definition", request));
        verify(wrapped, request, definitions, scope, definition);
    }

    /**
     * Test method for
     * {@link org.apache.tiles.impl.mgmt.CachingTilesContainer#CachingTilesContainer(TilesContainer, String)}
     * .
     */
    @SuppressWarnings("unchecked")
    @Test
    public void testCachingTilesContainer() {
        Request request = createMock(Request.class);
        Map<String, Definition> definitions = createMock(Map.class);
        Map<String, Object> scope = createMock(Map.class);
        Definition definition = createMock(Definition.class);
        TilesContainer wrapped = createMock(TilesContainer.class);

        expect(request.getContext("request")).andReturn(scope);
        expect(scope.get(DEFAULT_DEFINITIONS_ATTRIBUTE_NAME)).andReturn(
                definitions);
        expect(definitions.get("definition")).andReturn(definition);

        replay(wrapped, request, definitions, scope, definition);
        CachingTilesContainer container = new CachingTilesContainer(wrapped,
                null);
        assertSame(definition, container.getDefinition("definition", request));
        verify(wrapped, request, definitions, scope, definition);
    }

    /**
     * Test method for
     * {@link CachingTilesContainer#getDefinition(String, Request)}
     * .
     */
    @SuppressWarnings("unchecked")
    @Test
    public void testGetDefinition() {
        Request request = createMock(Request.class);
        Map<String, Definition> definitions = createMock(Map.class);
        Map<String, Object> scope = createMock(Map.class);
        Definition definition = createMock(Definition.class);

        expect(request.getContext("request")).andReturn(scope);
        expect(scope.get(DEFAULT_DEFINITIONS_ATTRIBUTE_NAME)).andReturn(
                definitions);
        expect(definitions.get("definition")).andReturn(definition);

        replay(wrapped, request, definitions, scope, definition);
        assertSame(definition, container.getDefinition("definition", request));
        verify(wrapped, request, definitions, scope, definition);
    }

    /**
     * Test method for
     * {@link CachingTilesContainer#getDefinition(String, Request)}
     * .
     */
    @SuppressWarnings("unchecked")
    @Test
    public void testGetDefinitionContainer() {
        Request request = createMock(Request.class);
        Map<String, Object> scope = createMock(Map.class);
        Definition definition = createMock(Definition.class);

        expect(request.getContext("request")).andReturn(scope);
        expect(scope.get(DEFAULT_DEFINITIONS_ATTRIBUTE_NAME)).andReturn(
                null);
        expect(wrapped.getDefinition("definition", request)).andReturn(
                definition);

        replay(wrapped, request, scope, definition);
        assertSame(definition, container.getDefinition("definition", request));
        verify(wrapped, request, scope, definition);
    }

    /**
     * Test method for
     * {@link org.apache.tiles.impl.mgmt.CachingTilesContainer#isValidDefinition(String, Request)}
     * .
     */
    @SuppressWarnings("unchecked")
    @Test
    public void testIsValidDefinition() {
        Request request = createMock(Request.class);
        Map<String, Definition> definitions = createMock(Map.class);
        Map<String, Object> scope = createMock(Map.class);
        Definition definition = createMock(Definition.class);

        expect(request.getContext("request")).andReturn(scope);
        expect(scope.get(DEFAULT_DEFINITIONS_ATTRIBUTE_NAME)).andReturn(
                definitions);
        expect(definitions.get("definition")).andReturn(definition);

        replay(wrapped, request, definitions, scope, definition);
        assertTrue(container.isValidDefinition("definition", request));
        verify(wrapped, request, definitions, scope, definition);
    }

    /**
     * Test method for
     * {@link org.apache.tiles.impl.mgmt.CachingTilesContainer#isValidDefinition(String, Request)}
     * .
     */
    @SuppressWarnings("unchecked")
    @Test
    public void testIsValidDefinitionContainer() {
        Request request = createMock(Request.class);
        Map<String, Definition> definitions = createMock(Map.class);
        Map<String, Object> scope = createMock(Map.class);

        expect(request.getContext("request")).andReturn(scope);
        expect(scope.get(DEFAULT_DEFINITIONS_ATTRIBUTE_NAME)).andReturn(
                definitions);
        expect(definitions.get("definition")).andReturn(null);
        expect(wrapped.isValidDefinition("definition", request)).andReturn(true);

        replay(wrapped, request, definitions, scope);
        assertTrue(container.isValidDefinition("definition", request));
        verify(wrapped, request, definitions, scope);
    }

    /**
     * Test method for
     * {@link CachingTilesContainer#register(Definition, Request)}
     * .
     */
    @SuppressWarnings("unchecked")
    @Test
    public void testRegister() {
        Request request = createMock(Request.class);
        Map<String, Definition> definitions = createMock(Map.class);
        Map<String, Object> scope = createMock(Map.class);
        Definition definition = createMock(Definition.class);

        expect(request.getContext("request")).andReturn(scope);
        expect(scope.get(DEFAULT_DEFINITIONS_ATTRIBUTE_NAME)).andReturn(
                definitions);
        expect(definition.getName()).andReturn(null);
        expect(definitions.containsKey("$anonymousMutableDefinition1"))
                .andReturn(false);
        definition.setName("$anonymousMutableDefinition1");
        expect(definition.isExtending()).andReturn(true);
        // trick to test resolve definition separately.
        expect(definition.isExtending()).andReturn(false);
        expect(definition.getName()).andReturn("$anonymousMutableDefinition1");
        expect(definitions.put("$anonymousMutableDefinition1", definition))
                .andReturn(null);

        replay(wrapped, request, definitions, scope, definition);
        container.register(definition, request);
        verify(wrapped, request, definitions, scope, definition);
    }

    /**
     * Test method for
     * {@link CachingTilesContainer#register(Definition, Request)}
     * .
     */
    @SuppressWarnings("unchecked")
    @Test
    public void testRegisterInheritance() {
        Request request = createMock(Request.class);
        Map<String, Definition> definitions = createMock(Map.class);
        Map<String, Object> scope = createMock(Map.class);
        Definition definition = createMock(Definition.class);
        Definition parent = createMock(Definition.class);
        Definition grandparent = createMock(Definition.class);

        expect(request.getContext("request")).andReturn(scope).anyTimes();
        expect(scope.get(DEFAULT_DEFINITIONS_ATTRIBUTE_NAME)).andReturn(
                definitions).anyTimes();
        expect(definition.getName()).andReturn(null);
        expect(definitions.containsKey("$anonymousMutableDefinition1"))
                .andReturn(false);
        definition.setName("$anonymousMutableDefinition1");
        expect(definition.isExtending()).andReturn(true);
        // trick to test resolve definition separately.
        expect(definition.isExtending()).andReturn(true);
        expect(definition.getExtends()).andReturn("parent");
        expect(definitions.get("parent")).andReturn(parent);
        expect(parent.isExtending()).andReturn(true);
        expect(parent.getExtends()).andReturn("grandparent");
        expect(definition.getName()).andReturn("$anonymousMutableDefinition1");
        expect(definitions.get("grandparent")).andReturn(null);
        expect(wrapped.getDefinition("grandparent", request)).andReturn(
                grandparent);
        parent.inherit(grandparent);
        definition.inherit(parent);
        expect(definitions.put("$anonymousMutableDefinition1", definition))
                .andReturn(null);

        replay(wrapped, request, definitions, scope, definition, parent,
                grandparent);
        container.register(definition, request);
        verify(wrapped, request, definitions, scope, definition, parent,
                grandparent);
    }

    /**
     * Test method for
     * {@link CachingTilesContainer#register(Definition, Request)}
     * .
     */
    @SuppressWarnings("unchecked")
    @Test(expected = NoSuchDefinitionException.class)
    public void testRegisterInheritanceFail() {
        Request request = createMock(Request.class);
        Map<String, Definition> definitions = createMock(Map.class);
        Map<String, Object> scope = createMock(Map.class);
        Definition definition = createMock(Definition.class);

        expect(request.getContext("request")).andReturn(scope).anyTimes();
        expect(scope.get(DEFAULT_DEFINITIONS_ATTRIBUTE_NAME)).andReturn(
                definitions).anyTimes();
        expect(definition.getName()).andReturn(null);
        expect(definitions.containsKey("$anonymousMutableDefinition1"))
                .andReturn(false);
        definition.setName("$anonymousMutableDefinition1");
        expect(definition.isExtending()).andReturn(true);
        // trick to test resolve definition separately.
        expect(definition.isExtending()).andReturn(true);
        expect(definition.getExtends()).andReturn("parent");
        expect(definitions.get("parent")).andReturn(null);
        expect(wrapped.getDefinition("parent", request)).andReturn(null);
        expect(definition.getName()).andReturn("$anonymousMutableDefinition1");

        replay(wrapped, request, definitions, scope, definition);
        try {
            container.register(definition, request);
        } finally {
            verify(wrapped, request, definitions, scope, definition);
        }
    }

    /**
     * Test method for
     * {@link CachingTilesContainer#register(Definition, Request)}
     * .
     */
    @SuppressWarnings("unchecked")
    @Test
    public void testRegisterCreateDefinitions() {
        Request request = createMock(Request.class);
        Map<String, Object> scope = createMock(Map.class);
        Definition definition = createMock(Definition.class);

        expect(request.getContext("request")).andReturn(scope).anyTimes();
        expect(scope.get(DEFAULT_DEFINITIONS_ATTRIBUTE_NAME)).andReturn(
                null);
        expect(scope.put(eq(DEFAULT_DEFINITIONS_ATTRIBUTE_NAME), isA(HashMap.class))).andReturn(null);
        expect(definition.getName()).andReturn(null);
        definition.setName("$anonymousMutableDefinition1");
        expect(definition.isExtending()).andReturn(true);
        // trick to test resolve definition separately.
        expect(definition.isExtending()).andReturn(false);
        expect(definition.getName()).andReturn("$anonymousMutableDefinition1");

        replay(wrapped, request, scope, definition);
        container.register(definition, request);
        verify(wrapped, request, scope, definition);
    }

    /**
     * Test method for
     * {@link org.apache.tiles.impl.mgmt.CachingTilesContainer#render(String, Request)}
     * .
     */
    @SuppressWarnings("unchecked")
    @Test
    public void testRender() {
        Request request = createMock(Request.class);
        Map<String, Definition> definitions = createMock(Map.class);
        Map<String, Object> scope = createMock(Map.class);
        Definition definition = createMock(Definition.class);

        expect(request.getContext("request")).andReturn(scope);
        expect(scope.get(DEFAULT_DEFINITIONS_ATTRIBUTE_NAME)).andReturn(
                definitions);
        expect(definitions.get("definition")).andReturn(definition);
        container.render(definition, request);

        replay(wrapped, request, definitions, scope, definition);
        container.render("definition", request);
        verify(wrapped, request, definitions, scope, definition);
    }

    /**
     * Test method for
     * {@link org.apache.tiles.impl.mgmt.CachingTilesContainer#render(String, Request)}
     * .
     */
    @SuppressWarnings("unchecked")
    @Test(expected = NoSuchDefinitionException.class)
    public void testRenderFail() {
        Request request = createMock(Request.class);
        Map<String, Definition> definitions = createMock(Map.class);
        Map<String, Object> scope = createMock(Map.class);

        expect(request.getContext("request")).andReturn(scope);
        expect(scope.get(DEFAULT_DEFINITIONS_ATTRIBUTE_NAME)).andReturn(
                definitions);
        expect(definitions.get("definition")).andReturn(null);
        expect(wrapped.getDefinition("definition", request)).andReturn(null);

        replay(wrapped, request, definitions, scope);
        try {
            container.render("definition", request);
        } finally {
            verify(wrapped, request, definitions, scope);
        }
    }
}
