/*
 * Decompiled with CFR 0.152.
 */
package org.apache.tika.pipes;

import java.io.BufferedReader;
import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.Closeable;
import java.io.DataInputStream;
import java.io.DataOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.FutureTask;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import org.apache.tika.pipes.FetchEmitTuple;
import org.apache.tika.pipes.PipesConfigBase;
import org.apache.tika.pipes.PipesResult;
import org.apache.tika.pipes.async.AsyncConfig;
import org.apache.tika.pipes.emitter.EmitData;
import org.apache.tika.utils.ProcessUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class PipesClient
implements Closeable {
    private static final Logger LOG = LoggerFactory.getLogger(PipesClient.class);
    private Process process;
    private LogGobbler logGobbler;
    private Thread logGobblerThread;
    private final PipesConfigBase pipesConfig;
    private DataOutputStream output;
    private DataInputStream input;
    private final ExecutorService executorService = Executors.newFixedThreadPool(1);
    private int filesProcessed = 0;

    public PipesClient(PipesConfigBase pipesConfig) {
        this.pipesConfig = pipesConfig;
    }

    public int getFilesProcessed() {
        return this.filesProcessed;
    }

    private boolean ping() {
        if (this.process == null || !this.process.isAlive()) {
            return false;
        }
        try {
            this.output.write(3);
            this.output.flush();
            int ping = this.input.read();
            if (ping == 3) {
                return true;
            }
        }
        catch (IOException e) {
            return false;
        }
        return false;
    }

    @Override
    public void close() throws IOException {
        if (this.process != null) {
            this.process.destroyForcibly();
        }
        this.executorService.shutdownNow();
    }

    public PipesResult process(FetchEmitTuple t) throws IOException {
        if (!this.ping()) {
            this.restart();
        }
        if (this.pipesConfig.getMaxFilesProcessed() > 0 && this.filesProcessed >= this.pipesConfig.getMaxFilesProcessed()) {
            LOG.info("restarting server after hitting max files: " + this.filesProcessed);
            this.restart();
        }
        return this.actuallyProcess(t);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private PipesResult actuallyProcess(FetchEmitTuple t) {
        long start = System.currentTimeMillis();
        FutureTask<PipesResult> futureTask = new FutureTask<PipesResult>(() -> {
            ByteArrayOutputStream bos = new ByteArrayOutputStream();
            try (ObjectOutputStream objectOutputStream = new ObjectOutputStream(bos);){
                objectOutputStream.writeObject(t);
            }
            byte[] bytes = bos.toByteArray();
            this.output.write(2);
            this.output.writeInt(bytes.length);
            this.output.write(bytes);
            this.output.flush();
            return this.readResults(t, start);
        });
        try {
            this.executorService.execute(futureTask);
            PipesResult pipesResult = futureTask.get(this.pipesConfig.getTimeoutMillis(), TimeUnit.MILLISECONDS);
            return pipesResult;
        }
        catch (InterruptedException e) {
            this.process.destroyForcibly();
            PipesResult pipesResult = PipesResult.INTERRUPTED_EXCEPTION;
            return pipesResult;
        }
        catch (ExecutionException e) {
            long elapsed = System.currentTimeMillis() - start;
            this.destroyWithPause();
            if (!this.process.isAlive() && 17 == this.process.exitValue()) {
                LOG.warn("server timeout: {} in {} ms", (Object)t.getId(), (Object)elapsed);
                PipesResult pipesResult = PipesResult.TIMEOUT;
                return pipesResult;
            }
            try {
                this.process.waitFor(500L, TimeUnit.MILLISECONDS);
                if (this.process.isAlive()) {
                    LOG.warn("crash: {} in {} ms with no exit code available", (Object)t.getId(), (Object)elapsed);
                } else {
                    LOG.warn("crash: {} in {} ms with exit code {}", t.getId(), elapsed, this.process.exitValue());
                }
            }
            catch (InterruptedException interruptedException) {
                // empty catch block
            }
            PipesResult pipesResult = PipesResult.UNSPECIFIED_CRASH;
            return pipesResult;
        }
        catch (TimeoutException e) {
            long elapsed = System.currentTimeMillis() - start;
            this.process.destroyForcibly();
            LOG.warn("client timeout: {} in {} ms", (Object)t.getId(), (Object)elapsed);
            PipesResult pipesResult = PipesResult.TIMEOUT;
            return pipesResult;
        }
        finally {
            futureTask.cancel(true);
        }
    }

    private void destroyWithPause() {
        try {
            this.process.waitFor(200L, TimeUnit.MILLISECONDS);
        }
        catch (InterruptedException interruptedException) {
        }
        finally {
            this.process.destroyForcibly();
        }
    }

    private PipesResult readResults(FetchEmitTuple t, long start) throws IOException {
        int status = this.input.read();
        long millis = System.currentTimeMillis() - start;
        switch (status) {
            case 12: {
                LOG.warn("oom: {} in {} ms", (Object)t.getId(), (Object)millis);
                return PipesResult.OOM;
            }
            case 13: {
                LOG.warn("server response timeout: {} in {} ms", (Object)t.getId(), (Object)millis);
                return PipesResult.TIMEOUT;
            }
            case 10: {
                LOG.warn("emit exception: {} in {} ms", (Object)t.getId(), (Object)millis);
                return this.readMessage(PipesResult.STATUS.EMIT_EXCEPTION);
            }
            case 11: {
                LOG.warn("no emitter found: " + t.getId());
                return PipesResult.NO_EMITTER_FOUND;
            }
            case 5: 
            case 7: {
                LOG.info("parse success: {} in {} ms", (Object)t.getId(), (Object)millis);
                return this.deserializeEmitData();
            }
            case 6: {
                return this.readMessage(PipesResult.STATUS.PARSE_EXCEPTION_NO_EMIT);
            }
            case 8: {
                LOG.info("emit success: {} in {} ms", (Object)t.getId(), (Object)millis);
                return PipesResult.EMIT_SUCCESS;
            }
            case 9: {
                return this.readMessage(PipesResult.STATUS.EMIT_SUCCESS_PARSE_EXCEPTION);
            }
        }
        throw new IOException("problem reading response from server " + status);
    }

    private PipesResult readMessage(PipesResult.STATUS status) throws IOException {
        int length = this.input.readInt();
        byte[] bytes = new byte[length];
        this.input.readFully(bytes);
        String msg = new String(bytes, StandardCharsets.UTF_8);
        return new PipesResult(status, msg);
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    private PipesResult deserializeEmitData() throws IOException {
        int length = this.input.readInt();
        byte[] bytes = new byte[length];
        this.input.readFully(bytes);
        try (ObjectInputStream objectInputStream = new ObjectInputStream(new ByteArrayInputStream(bytes));){
            PipesResult pipesResult = new PipesResult((EmitData)objectInputStream.readObject());
            return pipesResult;
        }
        catch (ClassNotFoundException e) {
            LOG.error("class not found exception deserializing data", e);
            throw new RuntimeException(e);
        }
    }

    private void restart() throws IOException {
        if (this.process != null) {
            this.process.destroyForcibly();
            LOG.info("restarting process");
        } else {
            LOG.info("starting process");
        }
        if (this.logGobblerThread != null) {
            this.logGobblerThread.interrupt();
        }
        ProcessBuilder pb = new ProcessBuilder(this.getCommandline());
        this.process = pb.start();
        this.logGobbler = new LogGobbler(this.process.getErrorStream());
        this.logGobblerThread = new Thread(this.logGobbler);
        this.logGobblerThread.setDaemon(true);
        this.logGobblerThread.start();
        this.input = new DataInputStream(this.process.getInputStream());
        this.output = new DataOutputStream(this.process.getOutputStream());
        FutureTask<Integer> futureTask = new FutureTask<Integer>(() -> {
            int b = this.input.read();
            if (b != 1) {
                throw new RuntimeException("Couldn't start server: " + b);
            }
            return 1;
        });
        this.executorService.submit(futureTask);
        try {
            futureTask.get(this.pipesConfig.getStartupTimeoutMillis(), TimeUnit.MILLISECONDS);
        }
        catch (InterruptedException e) {
            this.process.destroyForcibly();
            return;
        }
        catch (ExecutionException e) {
            LOG.error("couldn't start server", e);
            this.process.destroyForcibly();
            throw new RuntimeException(e);
        }
        catch (TimeoutException e) {
            LOG.error("couldn't start server in time", e);
            this.process.destroyForcibly();
            throw new RuntimeException(e);
        }
        finally {
            futureTask.cancel(true);
        }
    }

    private String[] getCommandline() {
        List<String> configArgs = this.pipesConfig.getForkedJvmArgs();
        boolean hasClassPath = false;
        boolean hasHeadless = false;
        boolean hasExitOnOOM = false;
        for (String arg : configArgs) {
            if (arg.startsWith("-Djava.awt.headless")) {
                hasHeadless = true;
            }
            if (arg.equals("-cp") || arg.equals("--classpath")) {
                hasClassPath = true;
            }
            if (!arg.equals("-XX:+ExitOnOutOfMemoryError") && !arg.equals("-XX:+CrashOnOutOfMemoryError")) continue;
            hasExitOnOOM = true;
        }
        ArrayList<String> commandLine = new ArrayList<String>();
        String javaPath = this.pipesConfig.getJavaPath();
        commandLine.add(ProcessUtils.escapeCommandLine(javaPath));
        if (!hasClassPath) {
            commandLine.add("-cp");
            commandLine.add(System.getProperty("java.class.path"));
        }
        if (!hasHeadless) {
            commandLine.add("-Djava.awt.headless=true");
        }
        if (!hasExitOnOOM) {
            // empty if block
        }
        commandLine.addAll(configArgs);
        commandLine.add("org.apache.tika.pipes.PipesServer");
        commandLine.add(ProcessUtils.escapeCommandLine(this.pipesConfig.getTikaConfig().toAbsolutePath().toString()));
        String maxForEmitBatchBytes = "0";
        if (this.pipesConfig instanceof AsyncConfig) {
            maxForEmitBatchBytes = Long.toString(((AsyncConfig)this.pipesConfig).getMaxForEmitBatchBytes());
        }
        commandLine.add(maxForEmitBatchBytes);
        commandLine.add(Long.toString(this.pipesConfig.getTimeoutMillis()));
        commandLine.add(Long.toString(this.pipesConfig.getShutdownClientAfterMillis()));
        LOG.debug("commandline: " + ((Object)commandLine).toString());
        return commandLine.toArray(new String[0]);
    }

    public static class LogGobbler
    implements Runnable {
        private static final Logger SERVER_LOG = LoggerFactory.getLogger(LogGobbler.class);
        private final BufferedReader reader;

        public LogGobbler(InputStream is) {
            this.reader = new BufferedReader(new InputStreamReader(is, StandardCharsets.UTF_8));
        }

        @Override
        public void run() {
            String line = null;
            try {
                line = this.reader.readLine();
            }
            catch (IOException e) {
                return;
            }
            while (line != null) {
                if (line.startsWith("debug ")) {
                    SERVER_LOG.debug(line.substring(6));
                } else if (line.startsWith("info ")) {
                    SERVER_LOG.info(line.substring(5));
                } else if (line.startsWith("warn ")) {
                    SERVER_LOG.warn(line.substring(5));
                } else if (line.startsWith("error ")) {
                    SERVER_LOG.error(line.substring(6));
                } else {
                    SERVER_LOG.debug(line);
                }
                try {
                    line = this.reader.readLine();
                }
                catch (IOException e) {
                    return;
                }
            }
        }
    }
}

