/*
 * Copyright 2007 The Closure Compiler Authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.google.javascript.jscomp;

import com.google.common.base.Optional;
import com.google.common.collect.ImmutableList;
import com.google.javascript.jscomp.Requirement.WhitelistEntry;

/**
 * The error manager is in charge of storing, organizing and displaying
 * errors and warnings generated by the compiler.
 */
public interface ErrorManager extends ErrorHandler {
  /**
   * Reports an error. The errors will be displayed by the
   * {@link #generateReport()} at the discretion of the implementation.
   *
   * @param level the reporting level
   * @param error the error to report
   */
  @Override
  void report(CheckLevel level, JSError error);

  /**
   * Writes a report to an implementation-specific medium. The compiler calls
   * this method after any and all {@link #report} calls.
   */
  void generateReport();

  /**
   * Gets the number of reported errors.
   */
  int getErrorCount();

  /**
   * Gets the number of reported warnings.
   */
  int getWarningCount();

  /** Gets all the errors. */
  ImmutableList<JSError> getErrors();

  /** Gets all the warnings. */
  ImmutableList<JSError> getWarnings();

  /**
   * Sets the percentage of typed expressions.
   */
  void setTypedPercent(double typedPercent);

  /**
   * Gets the percentage of typed expressions.
   */
  double getTypedPercent();

  /**
   * Returns if the error manager has errors that should make compilation halt.
   * This, for example, omits errors that were promoted from warnings by using the --strict flag.
   */
  default boolean hasHaltingErrors() {
    if (getErrorCount() == 0) {
      return false;
    }
    for (JSError error : getErrors()) {
       if (error.getType().level == CheckLevel.ERROR) {
         return true;
       }
    }
    return false;
  }

  /**
   * Return true if the conformance violation should be reported. This is called even if the
   * violation is whitelisted.
   */
  default boolean shouldReportConformanceViolation(
      Requirement requirement, Optional<WhitelistEntry> whitelistEntry, JSError diagnostic) {
    return true;
  }
}
