/*
 * Decompiled with CFR 0.152.
 */
package org.apache.wicket.pageStore;

import java.io.Closeable;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.io.RandomAccessFile;
import java.io.Serializable;
import java.nio.ByteBuffer;
import java.nio.channels.FileChannel;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;
import org.apache.wicket.WicketRuntimeException;
import org.apache.wicket.page.IManageablePage;
import org.apache.wicket.pageStore.AbstractPersistentPageStore;
import org.apache.wicket.pageStore.IPersistedPage;
import org.apache.wicket.pageStore.IPersistentPageStore;
import org.apache.wicket.pageStore.SerializedPage;
import org.apache.wicket.pageStore.disk.NestedFolders;
import org.apache.wicket.pageStore.disk.PageWindowManager;
import org.apache.wicket.util.file.Files;
import org.apache.wicket.util.io.IOUtils;
import org.apache.wicket.util.lang.Args;
import org.apache.wicket.util.lang.Bytes;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class DiskPageStore
extends AbstractPersistentPageStore
implements IPersistentPageStore {
    private static final Logger log = LoggerFactory.getLogger(DiskPageStore.class);
    private static final String INDEX_FILE_NAME = "DiskPageStoreIndex";
    private final Bytes maxSizePerSession;
    private final NestedFolders folders;
    private final ConcurrentMap<String, DiskData> diskDatas;

    public DiskPageStore(String applicationName, File fileStoreFolder, Bytes maxSizePerSession) {
        super(applicationName);
        this.folders = new NestedFolders(new File(fileStoreFolder, applicationName + "-filestore"));
        this.maxSizePerSession = (Bytes)Args.notNull((Object)maxSizePerSession, (String)"maxSizePerSession");
        this.diskDatas = new ConcurrentHashMap<String, DiskData>();
        try {
            if (this.folders.getBase().exists() || this.folders.getBase().mkdirs()) {
                this.loadIndex();
            } else {
                log.warn("Cannot create file store folder for some reason.");
            }
        }
        catch (SecurityException e) {
            throw new WicketRuntimeException("SecurityException occurred while creating DiskPageStore. Consider using a non-disk based IPageStore implementation. See org.apache.wicket.Application.setPageManagerProvider(IPageManagerProvider)", e);
        }
    }

    @Override
    public boolean supportsVersioning() {
        return true;
    }

    @Override
    public void destroy() {
        log.debug("Destroying...");
        this.saveIndex();
        super.destroy();
        log.debug("Destroyed.");
    }

    @Override
    protected IManageablePage getPersistedPage(String sessionIdentifier, int id) {
        byte[] data;
        DiskData diskData = this.getDiskData(sessionIdentifier, false);
        if (diskData != null && (data = diskData.loadPage(id)) != null) {
            if (log.isDebugEnabled()) {
                log.debug("Returning page with id '{}' in session with id '{}'", (Object)id, (Object)sessionIdentifier);
            }
            return new SerializedPage(id, "unknown", data);
        }
        return null;
    }

    @Override
    protected void removePersistedPage(String sessionIdentifier, IManageablePage page) {
        DiskData diskData = this.getDiskData(sessionIdentifier, false);
        if (diskData != null) {
            if (log.isDebugEnabled()) {
                log.debug("Removing page with id '{}' in session with id '{}'", (Object)page.getPageId(), (Object)sessionIdentifier);
            }
            diskData.removeData(page.getPageId());
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    protected void removeAllPersistedPages(String sessionIdentifier) {
        DiskData diskData = this.getDiskData(sessionIdentifier, false);
        if (diskData != null) {
            ConcurrentMap<String, DiskData> concurrentMap = this.diskDatas;
            synchronized (concurrentMap) {
                this.diskDatas.remove(diskData.sessionIdentifier);
                diskData.unbind();
            }
        }
    }

    @Override
    protected void addPersistedPage(String sessionIdentifier, IManageablePage page) {
        if (!(page instanceof SerializedPage)) {
            throw new WicketRuntimeException("DiskPageStore works with serialized pages only");
        }
        SerializedPage serializedPage = (SerializedPage)page;
        DiskData diskData = this.getDiskData(sessionIdentifier, true);
        log.debug("Storing data for page with id '{}' in session with id '{}'", (Object)serializedPage.getPageId(), (Object)sessionIdentifier);
        byte[] data = serializedPage.getData();
        String type = serializedPage.getPageType();
        diskData.savePage(serializedPage.getPageId(), type, data);
    }

    protected DiskData getDiskData(String sessionIdentifier, boolean create) {
        if (!create) {
            return (DiskData)this.diskDatas.get(sessionIdentifier);
        }
        DiskData data = new DiskData(this, sessionIdentifier);
        DiskData existing = this.diskDatas.putIfAbsent(sessionIdentifier, data);
        return existing != null ? existing : data;
    }

    private void loadIndex() {
        File storeFolder = this.folders.getBase();
        File index = new File(storeFolder, INDEX_FILE_NAME);
        if (index.exists() && index.length() > 0L) {
            try (FileInputStream stream = new FileInputStream(index);){
                ObjectInputStream ois = new ObjectInputStream(stream);
                this.diskDatas.clear();
                for (DiskData diskData : (List)ois.readObject()) {
                    diskData.pageStore = this;
                    this.diskDatas.put(diskData.sessionIdentifier, diskData);
                }
            }
            catch (Exception e) {
                log.error("Couldn't load DiskPageStore index from file " + index + ".", (Throwable)e);
            }
        }
        Files.remove((File)index);
    }

    private void saveIndex() {
        File storeFolder = this.folders.getBase();
        if (storeFolder.exists()) {
            File index = new File(storeFolder, INDEX_FILE_NAME);
            Files.remove((File)index);
            try (FileOutputStream stream = new FileOutputStream(index);){
                ObjectOutputStream oos = new ObjectOutputStream(stream);
                ArrayList<DiskData> list = new ArrayList<DiskData>(this.diskDatas.size());
                for (DiskData diskData : this.diskDatas.values()) {
                    if (diskData.sessionIdentifier == null) continue;
                    list.add(diskData);
                }
                oos.writeObject(list);
            }
            catch (Exception e) {
                log.error("Couldn't write DiskPageStore index to file " + index + ".", (Throwable)e);
            }
        }
    }

    @Override
    public Set<String> getSessionIdentifiers() {
        return Collections.unmodifiableSet(this.diskDatas.keySet());
    }

    @Override
    public List<IPersistedPage> getPersistedPages(String sessionIdentifier) {
        ArrayList<IPersistedPage> pages = new ArrayList<IPersistedPage>();
        DiskData diskData = this.getDiskData(sessionIdentifier, false);
        if (diskData != null) {
            PageWindowManager windowManager = diskData.getManager();
            pages.addAll(windowManager.getFileWindows());
        }
        return pages;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public Bytes getTotalSize() {
        long size = 0L;
        ConcurrentMap<String, DiskData> concurrentMap = this.diskDatas;
        synchronized (concurrentMap) {
            for (DiskData diskData : this.diskDatas.values()) {
                size += diskData.size();
            }
        }
        return Bytes.bytes((long)size);
    }

    private String getSessionFileName(String sessionIdentifier) {
        File sessionFolder = this.folders.get(sessionIdentifier, true);
        return new File(sessionFolder, "data").getAbsolutePath();
    }

    protected static class DiskData
    implements Serializable {
        private static final long serialVersionUID = 1L;
        private transient DiskPageStore pageStore;
        private transient String fileName;
        private String sessionIdentifier;
        private PageWindowManager manager;

        protected DiskData(DiskPageStore pageStore, String sessionIdentifier) {
            this.pageStore = pageStore;
            this.sessionIdentifier = sessionIdentifier;
        }

        public long size() {
            return this.manager.getTotalSize();
        }

        public PageWindowManager getManager() {
            if (this.manager == null) {
                this.manager = new PageWindowManager(this.pageStore.maxSizePerSession.bytes());
            }
            return this.manager;
        }

        private String getFileName() {
            if (this.fileName == null) {
                this.fileName = this.pageStore.getSessionFileName(this.sessionIdentifier);
            }
            return this.fileName;
        }

        public String getKey() {
            return this.sessionIdentifier;
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        public synchronized void savePage(int pageId, String pageType, byte[] data) {
            if (this.sessionIdentifier == null) {
                return;
            }
            if (data != null) {
                PageWindowManager.FileWindow window = this.getManager().createPageWindow(pageId, pageType, data.length);
                FileChannel channel = this.getFileChannel(true);
                if (channel != null) {
                    try {
                        channel.write(ByteBuffer.wrap(data), window.getFilePartOffset());
                    }
                    catch (IOException e) {
                        log.error("Error writing to a channel " + channel, (Throwable)e);
                    }
                    finally {
                        IOUtils.closeQuietly((Closeable)channel);
                    }
                } else {
                    log.warn("Cannot save page with id '{}' because the data file cannot be opened.", (Object)pageId);
                }
            }
        }

        public synchronized void removeData(int pageId) {
            if (this.sessionIdentifier == null) {
                return;
            }
            this.getManager().removePage(pageId);
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        public byte[] loadData(PageWindowManager.FileWindow window) {
            byte[] result = null;
            FileChannel channel = this.getFileChannel(false);
            if (channel != null) {
                ByteBuffer buffer = ByteBuffer.allocate(window.getFilePartSize());
                try {
                    channel.read(buffer, window.getFilePartOffset());
                    if (buffer.hasArray()) {
                        result = buffer.array();
                    }
                }
                catch (IOException e) {
                    log.error("Error reading from file channel " + channel, (Throwable)e);
                }
                finally {
                    IOUtils.closeQuietly((Closeable)channel);
                }
            }
            return result;
        }

        private FileChannel getFileChannel(boolean create) {
            FileChannel channel = null;
            File file = new File(this.getFileName());
            if (create || file.exists()) {
                String mode = create ? "rw" : "r";
                try {
                    RandomAccessFile randomAccessFile = new RandomAccessFile(file, mode);
                    channel = randomAccessFile.getChannel();
                }
                catch (FileNotFoundException fnfx) {
                    log.error(fnfx.getMessage(), (Throwable)fnfx);
                }
            }
            return channel;
        }

        public synchronized byte[] loadPage(int id) {
            if (this.sessionIdentifier == null) {
                return null;
            }
            PageWindowManager.FileWindow window = this.getManager().getPageWindow(id);
            if (window == null) {
                return null;
            }
            return this.loadData(window);
        }

        public synchronized void unbind() {
            this.pageStore.folders.remove(this.sessionIdentifier);
            this.sessionIdentifier = null;
        }
    }
}

