/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package org.apache.syncope.core.rest.cxf;

import java.io.IOException;
import java.io.OutputStream;
import java.net.URI;
import java.nio.charset.StandardCharsets;
import java.util.Arrays;
import java.util.Collections;
import java.util.Objects;
import javax.ws.rs.WebApplicationException;
import javax.ws.rs.container.ContainerRequestContext;
import javax.ws.rs.container.ContainerRequestFilter;
import javax.ws.rs.container.PreMatching;
import javax.ws.rs.ext.Provider;
import javax.ws.rs.ext.WriterInterceptor;
import javax.ws.rs.ext.WriterInterceptorContext;
import org.apache.commons.io.IOUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.cxf.io.CachedOutputStream;
import org.apache.cxf.jaxrs.ext.MessageContext;
import org.apache.cxf.jaxrs.json.basic.JsonMapObject;
import org.apache.cxf.jaxrs.json.basic.JsonMapObjectReaderWriter;
import org.apache.cxf.jaxrs.swagger.openapi.OpenApiConfiguration;
import org.apache.cxf.jaxrs.swagger.openapi.SwaggerToOpenApiConversionUtils;
import org.apache.cxf.jaxrs.utils.JAXRSUtils;

@Provider
@PreMatching
public final class SyncopeSwaggerToOpenApiConversionFilter implements ContainerRequestFilter, WriterInterceptor {

    private static final String SWAGGER_PATH = "swagger.json";

    private static final String OPEN_API_PATH = "openapi.json";

    private static final String OPEN_API_PROPERTY = "openapi";

    private OpenApiConfiguration openApiConfig;

    private String openApiJsonPath = OPEN_API_PATH;

    @Override
    public void filter(final ContainerRequestContext reqCtx) throws IOException {
        String path = reqCtx.getUriInfo().getPath();
        if (path.endsWith(openApiJsonPath)) {
            reqCtx.setRequestUri(URI.create(SWAGGER_PATH));
            JAXRSUtils.getCurrentMessage().getExchange().put(OPEN_API_PROPERTY, Boolean.TRUE);
        }
    }

    public OpenApiConfiguration getOpenApiConfig() {
        return openApiConfig;
    }

    public void setOpenApiConfig(final OpenApiConfiguration openApiConfig) {
        this.openApiConfig = openApiConfig;
    }

    @Override
    public void aroundWriteTo(final WriterInterceptorContext context) throws IOException, WebApplicationException {
        if (isOpenApiRequested()) {
            OutputStream os = context.getOutputStream();
            CachedOutputStream cos = new CachedOutputStream();

            context.setOutputStream(cos);
            context.proceed();

            MessageContext ctx = createMessageContext();
            String swaggerJson = IOUtils.toString(cos.getInputStream(), StandardCharsets.UTF_8);
            String openApiJson = SwaggerToOpenApiConversionUtils.getOpenApiFromSwaggerJson(
                    ctx, swaggerJson, openApiConfig);

            JsonMapObjectReaderWriter readerWriter = new JsonMapObjectReaderWriter();
            JsonMapObject openapi = readerWriter.fromJsonToJsonObject(openApiJson);

            String basePath = StringUtils.substringBeforeLast(ctx.getHttpServletRequest().getRequestURI(), "/");
            openapi.setProperty("servers", Arrays.asList(Collections.singletonMap("url", basePath)));

            os.write(readerWriter.toJson(openapi).getBytes(StandardCharsets.UTF_8));
            os.flush();
        } else {
            context.proceed();
        }
    }

    private MessageContext createMessageContext() {
        return JAXRSUtils.createContextValue(JAXRSUtils.getCurrentMessage(), null, MessageContext.class);
    }

    private boolean isOpenApiRequested() {
        return Objects.equals(Boolean.TRUE, JAXRSUtils.getCurrentMessage().getExchange().get(OPEN_API_PROPERTY));
    }

    public void setOpenApiJsonPath(final String openApiJsonPath) {
        this.openApiJsonPath = openApiJsonPath;
    }
}
