/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package org.apache.syncope.core.persistence.dao;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertFalse;
import static org.junit.Assert.assertNotNull;
import static org.junit.Assert.assertTrue;

import java.util.ArrayList;
import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import org.apache.syncope.common.types.SubjectType;
import org.apache.syncope.core.persistence.beans.role.SyncopeRole;
import org.apache.syncope.core.persistence.beans.user.SyncopeUser;
import org.apache.syncope.core.persistence.dao.search.AttributeCond;
import org.apache.syncope.core.persistence.dao.search.MembershipCond;
import org.apache.syncope.core.persistence.dao.search.OrderByClause;
import org.apache.syncope.core.persistence.dao.search.ResourceCond;
import org.apache.syncope.core.persistence.dao.search.SearchCond;
import org.apache.syncope.core.persistence.dao.search.SubjectCond;
import org.apache.syncope.core.util.EntitlementUtil;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.test.context.ContextConfiguration;
import org.springframework.test.context.junit4.SpringJUnit4ClassRunner;
import org.springframework.transaction.annotation.Transactional;

@RunWith(SpringJUnit4ClassRunner.class)
@ContextConfiguration(locations = { "classpath:persistenceTestEnv.xml" })
@Transactional
public class AttributableSearchTest {

    @Autowired
    private UserDAO userDAO;

    @Autowired
    private RoleDAO roleDAO;

    @Autowired
    private SubjectSearchDAO searchDAO;

    @Autowired
    private EntitlementDAO entitlementDAO;

    @Test
    public void userMatch() {
        SyncopeUser user = userDAO.find(1L);
        assertNotNull(user);

        MembershipCond membershipCond = new MembershipCond();
        membershipCond.setRoleId(5L);

        assertFalse(searchDAO.matches(user, SearchCond.getLeafCond(membershipCond), SubjectType.USER));

        membershipCond.setRoleId(1L);

        assertTrue(searchDAO.matches(user, SearchCond.getLeafCond(membershipCond), SubjectType.USER));
    }

    @Test
    public void roleMatch() {
        SyncopeRole role = roleDAO.find(1L);
        assertNotNull(role);

        AttributeCond attrCond = new AttributeCond();
        attrCond.setSchema("show");
        attrCond.setType(AttributeCond.Type.ISNOTNULL);

        assertTrue(searchDAO.matches(role, SearchCond.getLeafCond(attrCond), SubjectType.ROLE));
    }

    @Test
    public void searchWithLikeCondition() {
        AttributeCond fullnameLeafCond = new AttributeCond(AttributeCond.Type.LIKE);
        fullnameLeafCond.setSchema("fullname");
        fullnameLeafCond.setExpression("%o%");

        MembershipCond membershipCond = new MembershipCond();
        membershipCond.setRoleId(1L);

        AttributeCond loginDateCond = new AttributeCond(AttributeCond.Type.EQ);
        loginDateCond.setSchema("loginDate");
        loginDateCond.setExpression("2009-05-26");

        SearchCond subCond = SearchCond.getAndCond(
                SearchCond.getLeafCond(fullnameLeafCond), SearchCond.getLeafCond(membershipCond));

        assertTrue(subCond.isValid());

        SearchCond cond = SearchCond.getAndCond(subCond, SearchCond.getLeafCond(loginDateCond));

        assertTrue(cond.isValid());

        List<SyncopeUser> users =
                searchDAO.search(EntitlementUtil.getRoleIds(entitlementDAO.findAll()), cond, SubjectType.USER);
        assertNotNull(users);
        assertEquals(1, users.size());
    }

    @Test
    public void searchCaseInsensitiveWithLikeCondition() {
        AttributeCond fullnameLeafCond = new AttributeCond(AttributeCond.Type.ILIKE);
        fullnameLeafCond.setSchema("fullname");
        fullnameLeafCond.setExpression("%O%");

        MembershipCond groupCond = new MembershipCond();
        groupCond.setRoleId(1L);

        AttributeCond loginDateCond = new AttributeCond(AttributeCond.Type.EQ);
        loginDateCond.setSchema("loginDate");
        loginDateCond.setExpression("2009-05-26");

        SearchCond subCond = SearchCond.getAndCond(
                SearchCond.getLeafCond(fullnameLeafCond), SearchCond.getLeafCond(groupCond));

        assertTrue(subCond.isValid());

        SearchCond cond = SearchCond.getAndCond(subCond, SearchCond.getLeafCond(loginDateCond));

        assertTrue(cond.isValid());

        List<SyncopeUser> users =
                searchDAO.search(EntitlementUtil.getRoleIds(entitlementDAO.findAll()), cond, SubjectType.USER);
        assertNotNull(users);
        assertEquals(1, users.size());
    }

    @Test
    public void searchWithNotCondition() {
        AttributeCond fullnameLeafCond = new AttributeCond(AttributeCond.Type.EQ);
        fullnameLeafCond.setSchema("fullname");
        fullnameLeafCond.setExpression("Giuseppe Verdi");

        SearchCond cond = SearchCond.getNotLeafCond(fullnameLeafCond);
        assertTrue(cond.isValid());

        List<SyncopeUser> users =
                searchDAO.search(EntitlementUtil.getRoleIds(entitlementDAO.findAll()), cond, SubjectType.USER);
        assertNotNull(users);
        assertEquals(4, users.size());

        Set<Long> ids = new HashSet<Long>(users.size());
        for (SyncopeUser user : users) {
            ids.add(user.getId());
        }
        assertTrue(ids.contains(1L));
        assertTrue(ids.contains(3L));
    }

    @Test
    public void searchCaseInsensitiveWithNotCondition() {
        AttributeCond fullnameLeafCond = new AttributeCond(AttributeCond.Type.IEQ);
        fullnameLeafCond.setSchema("fullname");
        fullnameLeafCond.setExpression("giuseppe verdi");

        SearchCond cond = SearchCond.getNotLeafCond(fullnameLeafCond);
        assertTrue(cond.isValid());

        List<SyncopeUser> users =
                searchDAO.search(EntitlementUtil.getRoleIds(entitlementDAO.findAll()), cond, SubjectType.USER);
        assertNotNull(users);
        assertEquals(4, users.size());

        Set<Long> ids = new HashSet<Long>(users.size());
        for (SyncopeUser user : users) {
            ids.add(user.getId());
        }
        assertTrue(ids.contains(1L));
        assertTrue(ids.contains(3L));
    }

    @Test
    public void searchByBoolean() {
        AttributeCond coolLeafCond = new AttributeCond(AttributeCond.Type.EQ);
        coolLeafCond.setSchema("cool");
        coolLeafCond.setExpression("true");

        SearchCond cond = SearchCond.getLeafCond(coolLeafCond);
        assertTrue(cond.isValid());

        List<SyncopeUser> users =
                searchDAO.search(EntitlementUtil.getRoleIds(entitlementDAO.findAll()), cond, SubjectType.USER);
        assertNotNull(users);
        assertEquals(1, users.size());

        assertEquals(Long.valueOf(4L), users.get(0).getId());
    }

    @Test
    public void searchByPageAndSize() {
        AttributeCond fullnameLeafCond = new AttributeCond(AttributeCond.Type.LIKE);
        fullnameLeafCond.setSchema("fullname");
        fullnameLeafCond.setExpression("%o%");

        MembershipCond membershipCond = new MembershipCond();
        membershipCond.setRoleId(1L);

        AttributeCond loginDateCond = new AttributeCond(AttributeCond.Type.EQ);
        loginDateCond.setSchema("loginDate");
        loginDateCond.setExpression("2009-05-26");

        SearchCond subCond = SearchCond.getAndCond(SearchCond.getLeafCond(fullnameLeafCond), SearchCond.getLeafCond(
                membershipCond));

        assertTrue(subCond.isValid());

        SearchCond cond = SearchCond.getAndCond(subCond, SearchCond.getLeafCond(loginDateCond));

        assertTrue(cond.isValid());

        List<SyncopeUser> users = searchDAO.search(EntitlementUtil.getRoleIds(entitlementDAO.findAll()),
                cond, 1, 2, Collections.<OrderByClause>emptyList(),
                SubjectType.USER);
        assertNotNull(users);
        assertEquals(1, users.size());

        users = searchDAO.search(EntitlementUtil.getRoleIds(entitlementDAO.findAll()),
                cond, 2, 2, Collections.<OrderByClause>emptyList(),
                SubjectType.USER);
        assertNotNull(users);
        assertTrue(users.isEmpty());
    }

    @Test
    public void searchByMembership() {
        MembershipCond membershipCond = new MembershipCond();
        membershipCond.setRoleId(1L);

        List<SyncopeUser> users = searchDAO.search(
                EntitlementUtil.getRoleIds(entitlementDAO.findAll()), SearchCond.getLeafCond(membershipCond),
                SubjectType.USER);
        assertNotNull(users);
        assertEquals(2, users.size());

        membershipCond = new MembershipCond();
        membershipCond.setRoleId(5L);

        users = searchDAO.search(
                EntitlementUtil.getRoleIds(entitlementDAO.findAll()), SearchCond.getNotLeafCond(membershipCond),
                SubjectType.USER);
        assertNotNull(users);
        assertEquals(5, users.size());
    }

    @Test
    public void searchByIsNull() {
        AttributeCond coolLeafCond = new AttributeCond(AttributeCond.Type.ISNULL);
        coolLeafCond.setSchema("cool");

        List<SyncopeUser> users = searchDAO.search(
                EntitlementUtil.getRoleIds(entitlementDAO.findAll()), SearchCond.getLeafCond(coolLeafCond),
                SubjectType.USER);
        assertNotNull(users);
        assertEquals(4, users.size());

        coolLeafCond = new AttributeCond(AttributeCond.Type.ISNOTNULL);
        coolLeafCond.setSchema("cool");

        users = searchDAO.search(
                EntitlementUtil.getRoleIds(entitlementDAO.findAll()), SearchCond.getLeafCond(coolLeafCond),
                SubjectType.USER);
        assertNotNull(users);
        assertEquals(1, users.size());
    }

    @Test
    public void searchByResource() {
        ResourceCond ws2 = new ResourceCond();
        ws2.setResourceName("ws-target-resource-2");

        ResourceCond ws1 = new ResourceCond();
        ws1.setResourceName("ws-target-resource-list-mappings-2");

        SearchCond searchCondition = SearchCond.getAndCond(SearchCond.getNotLeafCond(ws2), SearchCond.getLeafCond(ws1));

        assertTrue(searchCondition.isValid());

        List<SyncopeUser> users = searchDAO.search(
                EntitlementUtil.getRoleIds(entitlementDAO.findAll()), searchCondition,
                SubjectType.USER);

        assertNotNull(users);
        assertEquals(1, users.size());
    }

    @Test
    public void searchByUsernameAndId() {
        SubjectCond usernameLeafCond = new SubjectCond(SubjectCond.Type.EQ);
        usernameLeafCond.setSchema("username");
        usernameLeafCond.setExpression("rossini");

        SubjectCond idRightCond = new SubjectCond(SubjectCond.Type.LT);
        idRightCond.setSchema("id");
        idRightCond.setExpression("2");

        SearchCond searchCondition = SearchCond.getOrCond(SearchCond.getLeafCond(usernameLeafCond),
                SearchCond.getLeafCond(idRightCond));

        List<SyncopeUser> matchingUsers = searchDAO.search(EntitlementUtil.getRoleIds(entitlementDAO.findAll()),
                searchCondition, SubjectType.USER);

        assertNotNull(matchingUsers);
        assertEquals(1, matchingUsers.size());
        assertEquals("rossini", matchingUsers.iterator().next().getUsername());
        assertEquals(1L, matchingUsers.iterator().next().getId().longValue());
    }

    @Test
    public void searchByRolenameAndId() {
        SubjectCond rolenameLeafCond = new SubjectCond(SubjectCond.Type.EQ);
        rolenameLeafCond.setSchema("name");
        rolenameLeafCond.setExpression("root");

        SubjectCond idRightCond = new SubjectCond(SubjectCond.Type.LT);
        idRightCond.setSchema("id");
        idRightCond.setExpression("2");

        SearchCond searchCondition = SearchCond.getAndCond(SearchCond.getLeafCond(rolenameLeafCond),
                SearchCond.getLeafCond(idRightCond));

        assertTrue(searchCondition.isValid());

        List<SyncopeRole> matchingRoles = searchDAO.search(EntitlementUtil.getRoleIds(entitlementDAO.findAll()),
                searchCondition, SubjectType.ROLE);

        assertNotNull(matchingRoles);
        assertEquals(1, matchingRoles.size());
        assertEquals("root", matchingRoles.iterator().next().getName());
        assertEquals(1L, matchingRoles.iterator().next().getId().longValue());
    }

    @Test
    public void searchByUsernameAndFullname() {
        SubjectCond usernameLeafCond = new SubjectCond(SubjectCond.Type.EQ);
        usernameLeafCond.setSchema("username");
        usernameLeafCond.setExpression("rossini");

        AttributeCond idRightCond = new AttributeCond(AttributeCond.Type.LIKE);
        idRightCond.setSchema("fullname");
        idRightCond.setExpression("Giuseppe V%");

        SearchCond searchCondition = SearchCond.getOrCond(SearchCond.getLeafCond(usernameLeafCond),
                SearchCond.getLeafCond(idRightCond));

        List<SyncopeUser> matchingUsers =
                searchDAO.search(EntitlementUtil.getRoleIds(entitlementDAO.findAll()), searchCondition,
                        SubjectType.USER);

        assertNotNull(matchingUsers);
        assertEquals(2, matchingUsers.size());
    }

    @Test
    public void searchByUsernameAndFullnameIgnoreCase() {
        SubjectCond usernameLeafCond = new SubjectCond(SubjectCond.Type.IEQ);
        usernameLeafCond.setSchema("username");
        usernameLeafCond.setExpression("RoSsini");

        AttributeCond idRightCond = new AttributeCond(AttributeCond.Type.ILIKE);
        idRightCond.setSchema("fullname");
        idRightCond.setExpression("gIuseppe v%");

        SearchCond searchCondition = SearchCond.getOrCond(
                SearchCond.getLeafCond(usernameLeafCond),
                SearchCond.getLeafCond(idRightCond));

        List<SyncopeUser> matchingUsers =
                searchDAO.search(EntitlementUtil.getRoleIds(entitlementDAO.findAll()), searchCondition,
                        SubjectType.USER);

        assertNotNull(matchingUsers);
        assertEquals(2, matchingUsers.size());
    }

    @Test
    public void searchById() {
        SubjectCond idLeafCond = new SubjectCond(SubjectCond.Type.LT);
        idLeafCond.setSchema("id");
        idLeafCond.setExpression("2");

        SearchCond searchCondition = SearchCond.getLeafCond(idLeafCond);
        assertTrue(searchCondition.isValid());

        List<SyncopeUser> users =
                searchDAO.search(EntitlementUtil.getRoleIds(entitlementDAO.findAll()), searchCondition,
                        SubjectType.USER);

        assertNotNull(users);
        assertEquals(1, users.size());
        assertEquals(1L, users.iterator().next().getId().longValue());

        idLeafCond = new SubjectCond(SubjectCond.Type.LT);
        idLeafCond.setSchema("id");
        idLeafCond.setExpression("4");

        searchCondition = SearchCond.getNotLeafCond(idLeafCond);
        assertTrue(searchCondition.isValid());

        users = searchDAO.search(EntitlementUtil.getRoleIds(entitlementDAO.findAll()), searchCondition,
                SubjectType.USER);

        assertNotNull(users);
        assertEquals(2, users.size());
        boolean found = false;
        for (SyncopeUser user : users) {
            if (user.getId() == 4) {
                found = true;
            }
        }
        assertTrue(found);
    }

    @Test
    public void userOrderBy() {
        SubjectCond usernameLeafCond = new SubjectCond(SubjectCond.Type.EQ);
        usernameLeafCond.setSchema("username");
        usernameLeafCond.setExpression("rossini");
        AttributeCond idRightCond = new AttributeCond(AttributeCond.Type.LIKE);
        idRightCond.setSchema("fullname");
        idRightCond.setExpression("Giuseppe V%");
        SearchCond searchCondition = SearchCond.getOrCond(
                SearchCond.getLeafCond(usernameLeafCond), SearchCond.getLeafCond(idRightCond));

        List<OrderByClause> orderByClauses = new ArrayList<OrderByClause>();
        OrderByClause orderByClause = new OrderByClause();
        orderByClause.setField("username");
        orderByClause.setDirection(OrderByClause.Direction.DESC);
        orderByClauses.add(orderByClause);
        orderByClause = new OrderByClause();
        orderByClause.setField("fullname");
        orderByClause.setDirection(OrderByClause.Direction.ASC);
        orderByClauses.add(orderByClause);

        List<SyncopeUser> users = searchDAO.search(EntitlementUtil.getRoleIds(entitlementDAO.findAll()),
                searchCondition, orderByClauses,
                SubjectType.USER);
        assertEquals(searchDAO.count(EntitlementUtil.getRoleIds(entitlementDAO.findAll()),
                searchCondition, SubjectType.USER),
                users.size());
    }

    @Test
    public void roleOrderBy() {
        SubjectCond idLeafCond = new SubjectCond(SubjectCond.Type.LIKE);
        idLeafCond.setSchema("name");
        idLeafCond.setExpression("%r");
        SearchCond searchCondition = SearchCond.getLeafCond(idLeafCond);
        assertTrue(searchCondition.isValid());

        OrderByClause orderByClause = new OrderByClause();
        orderByClause.setField("name");

        List<SyncopeRole> roles = searchDAO.search(EntitlementUtil.getRoleIds(entitlementDAO.findAll()),
                searchCondition, Collections.singletonList(orderByClause),
                SubjectType.ROLE);
        assertEquals(searchDAO.count(EntitlementUtil.getRoleIds(entitlementDAO.findAll()),
                searchCondition, SubjectType.ROLE),
                roles.size());
    }

    @Test
    public void issue202() {
        ResourceCond ws2 = new ResourceCond();
        ws2.setResourceName("ws-target-resource-2");

        ResourceCond ws1 = new ResourceCond();
        ws1.setResourceName("ws-target-resource-list-mappings-1");

        SearchCond searchCondition =
                SearchCond.getAndCond(SearchCond.getNotLeafCond(ws2), SearchCond.getNotLeafCond(ws1));
        assertTrue(searchCondition.isValid());

        List<SyncopeUser> users = searchDAO.search(EntitlementUtil.getRoleIds(entitlementDAO.findAll()),
                searchCondition, SubjectType.USER);
        assertNotNull(users);
        assertEquals(2, users.size());
        boolean found = false;
        for (SyncopeUser user : users) {
            if (user.getId() == 4) {
                found = true;
            }
        }
        assertTrue(found);
    }

    @Test
    public void issue242() {
        SubjectCond cond = new SubjectCond(AttributeCond.Type.LIKE);
        cond.setSchema("id");
        cond.setExpression("test%");

        SearchCond searchCondition = SearchCond.getLeafCond(cond);
        assertTrue(searchCondition.isValid());

        List<SyncopeUser> users = searchDAO.search(EntitlementUtil.getRoleIds(entitlementDAO.findAll()),
                searchCondition, SubjectType.USER);
        assertNotNull(users);
        assertTrue(users.isEmpty());
    }

    @Test
    public void issueSYNCOPE46() {
        SubjectCond cond = new SubjectCond(AttributeCond.Type.LIKE);
        cond.setSchema("username");
        cond.setExpression("%ossin%");

        SearchCond searchCondition = SearchCond.getLeafCond(cond);
        assertTrue(searchCondition.isValid());

        List<SyncopeUser> users = searchDAO.search(EntitlementUtil.getRoleIds(entitlementDAO.findAll()),
                searchCondition, SubjectType.USER);
        assertNotNull(users);
        assertEquals(1, users.size());
    }

    @Test
    public void issueSYNCOPE433() {
        AttributeCond isNullCond = new AttributeCond(AttributeCond.Type.ISNULL);
        isNullCond.setSchema("loginDate");

        SubjectCond likeCond = new SubjectCond(AttributeCond.Type.LIKE);
        likeCond.setSchema("username");
        likeCond.setExpression("%ossin%");

        SearchCond searchCond = SearchCond.getOrCond(
                SearchCond.getLeafCond(isNullCond), SearchCond.getLeafCond(likeCond));

        Integer count = searchDAO.count(EntitlementUtil.getRoleIds(entitlementDAO.findAll()), searchCond,
                SubjectType.USER);
        assertNotNull(count);
        assertTrue(count > 0);
    }

    @Test
    public void issueSYNCOPE929() {
        AttributeCond rossiniCond = new AttributeCond(AttributeCond.Type.EQ);
        rossiniCond.setSchema("surname");
        rossiniCond.setExpression("Rossini");

        AttributeCond genderCond = new AttributeCond(AttributeCond.Type.EQ);
        genderCond.setSchema("gender");
        genderCond.setExpression("M");

        SearchCond orCond = SearchCond.getOrCond(
                SearchCond.getLeafCond(rossiniCond), SearchCond.getLeafCond(genderCond));

        AttributeCond belliniCond = new AttributeCond(AttributeCond.Type.EQ);
        belliniCond.setSchema("surname");
        belliniCond.setExpression("Bellini");

        SearchCond searchCond =
                SearchCond.getAndCond(orCond, SearchCond.getLeafCond(belliniCond));

        List<SyncopeUser> users = searchDAO.search(EntitlementUtil.getRoleIds(entitlementDAO.findAll()),
                searchCond, SubjectType.USER);
        assertNotNull(users);
        assertEquals(1, users.size());
    }

    @Test
    public void issueSYNCOPE983() {
        AttributeCond fullnameLeafCond = new AttributeCond(AttributeCond.Type.LIKE);
        fullnameLeafCond.setSchema("surname");
        fullnameLeafCond.setExpression("%o%");

        List<OrderByClause> orderByClauses = new ArrayList<OrderByClause>();
        OrderByClause orderByClause = new OrderByClause();
        orderByClause.setField("surname");
        orderByClause.setDirection(OrderByClause.Direction.ASC);
        orderByClauses.add(orderByClause);
        orderByClause = new OrderByClause();
        orderByClause.setField("username");
        orderByClause.setDirection(OrderByClause.Direction.DESC);
        orderByClauses.add(orderByClause);

        List<SyncopeUser> users = searchDAO.search(
                EntitlementUtil.getRoleIds(entitlementDAO.findAll()),
                SearchCond.getLeafCond(fullnameLeafCond),
                orderByClauses,
                SubjectType.USER);
        assertFalse(users.isEmpty());
    }
}
