/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package org.apache.syncope.core.workflow.user;

import java.util.Map;
import org.apache.syncope.common.mod.UserMod;
import org.apache.syncope.common.to.UserTO;
import org.apache.syncope.core.persistence.beans.user.SyncopeUser;
import org.apache.syncope.core.rest.controller.UnauthorizedRoleException;
import org.apache.syncope.core.workflow.WorkflowAdapter;
import org.apache.syncope.core.workflow.WorkflowException;
import org.apache.syncope.core.workflow.WorkflowResult;

/**
 * Interface for calling underlying workflow implementations.
 */
public interface UserWorkflowAdapter extends WorkflowAdapter {

    /**
     * Create an user.
     *
     * @param userTO user to be created and wether to propagate it as active
     * @param storePassword whether password shall be stored into the internal storage
     * @return user just created
     * @throws UnauthorizedRoleException authorization exception
     * @throws WorkflowException workflow exception
     */
    WorkflowResult<Map.Entry<Long, Boolean>> create(UserTO userTO, boolean storePassword) throws
            UnauthorizedRoleException, WorkflowException;

    /**
     * Create an user, optionally disabling password policy check.
     *
     * @param userTO user to be created and wether to propagate it as active
     * @param disablePwdPolicyCheck disable password policy check?
     * @param storePassword whether password shall be stored into the internal storage
     * @return user just created
     * @throws UnauthorizedRoleException authorization exception
     * @throws WorkflowException workflow exception
     */
    WorkflowResult<Map.Entry<Long, Boolean>> create(UserTO userTO, boolean disablePwdPolicyCheck,
            boolean storePassword) throws UnauthorizedRoleException, WorkflowException;

    /**
     * Create an user, optionally disabling password policy check.
     *
     * @param userTO user to be created and wether to propagate it as active
     * @param disablePwdPolicyCheck disable password policy check?
     * @param enabled specify true/false to force active/supended status
     * @param storePassword whether password shall be stored into the internal storage
     * @return user just created
     * @throws UnauthorizedRoleException authorization exception
     * @throws WorkflowException workflow exception
     */
    WorkflowResult<Map.Entry<Long, Boolean>> create(UserTO userTO, boolean disablePwdPolicyCheck, final Boolean enabled,
            boolean storePassword) throws UnauthorizedRoleException, WorkflowException;

    /**
     * Execute a task on an user.
     *
     * @param userTO user to be subject to task
     * @param taskId to be executed
     * @return user just updated
     * @throws UnauthorizedRoleException authorization exception
     * @throws WorkflowException workflow exception
     */
    WorkflowResult<Long> execute(UserTO userTO, String taskId) throws UnauthorizedRoleException, WorkflowException;

    /**
     * Activate an user.
     *
     * @param userId user to be activated
     * @param token to be verified for activation
     * @return user just updated
     * @throws UnauthorizedRoleException authorization exception
     * @throws WorkflowException workflow exception
     */
    WorkflowResult<Long> activate(Long userId, String token) throws UnauthorizedRoleException, WorkflowException;

    /**
     * Update an user.
     *
     * @param userMod modification set to be performed
     * @return user just updated and propagations to be performed
     * @throws UnauthorizedRoleException authorization exception
     * @throws WorkflowException workflow exception
     */
    WorkflowResult<Map.Entry<UserMod, Boolean>> update(UserMod userMod)
            throws UnauthorizedRoleException, WorkflowException;

    /**
     * Suspend an user.
     *
     * @param userId user to be suspended
     * @return user just suspended
     * @throws UnauthorizedRoleException authorization exception
     * @throws WorkflowException workflow exception
     */
    WorkflowResult<Long> suspend(Long userId) throws UnauthorizedRoleException, WorkflowException;

    /**
     * Suspend an user.
     *
     * @param user to be suspended
     * @return user just suspended
     * @throws UnauthorizedRoleException authorization exception
     * @throws WorkflowException workflow exception
     */
    WorkflowResult<Long> suspend(SyncopeUser user) throws UnauthorizedRoleException, WorkflowException;

    /**
     * Reactivate an user.
     *
     * @param userId user to be reactivated
     * @return user just reactivated
     * @throws UnauthorizedRoleException authorization exception
     * @throws WorkflowException workflow exception
     */
    WorkflowResult<Long> reactivate(Long userId) throws UnauthorizedRoleException, WorkflowException;

    /**
     * Request password reset for an user.
     *
     * @param userId user requesting password reset
     * @throws UnauthorizedRoleException authorization exception
     * @throws WorkflowException workflow exception
     */
    void requestPasswordReset(Long userId) throws UnauthorizedRoleException, WorkflowException;

    /**
     * Confirm password reset for an user.
     *
     * @param userId user confirming password reset
     * @param token security token
     * @param password new password value
     * @throws UnauthorizedRoleException authorization exception
     * @throws WorkflowException workflow exception
     * @return user just updated and propagations to be performed
     */
    WorkflowResult<Map.Entry<UserMod, Boolean>> confirmPasswordReset(Long userId, String token, String password)
            throws UnauthorizedRoleException, WorkflowException;

    /**
     * Delete an user.
     *
     * @param userId user to be deleted
     * @throws UnauthorizedRoleException authorization exception
     * @throws WorkflowException workflow exception
     */
    void delete(Long userId) throws UnauthorizedRoleException, WorkflowException;
}
