/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package org.apache.syncope.core.persistence.validation.entity;

import java.text.ParseException;

import javax.validation.ConstraintValidatorContext;

import org.apache.syncope.common.types.EntityViolationType;
import org.apache.syncope.core.persistence.beans.SchedTask;
import org.apache.syncope.core.quartz.TaskJob;
import org.quartz.CronExpression;

public class SchedTaskValidator extends AbstractValidator<SchedTaskCheck, SchedTask> {

    @Override
    public boolean isValid(final SchedTask object, final ConstraintValidatorContext context) {
        boolean isValid;

        Class<?> jobClass = null;
        try {
            jobClass = Class.forName(object.getJobClassName());
            isValid = TaskJob.class.isAssignableFrom(jobClass);
        } catch (Exception e) {
            LOG.error("Invalid Job class specified", e);
            isValid = false;
        }
        if (jobClass == null || !isValid) {
            isValid = false;

            context.disableDefaultConstraintViolation();
            context.buildConstraintViolationWithTemplate(
                    getTemplate(EntityViolationType.InvalidSchedTask, "Invalid job class name")).
                    addPropertyNode("jobClassName").addConstraintViolation();
        }

        if (isValid && object.getCronExpression() != null) {
            try {
                new CronExpression(object.getCronExpression());
            } catch (ParseException e) {
                LOG.error("Invalid cron expression '" + object.getCronExpression() + "'", e);
                isValid = false;

                context.disableDefaultConstraintViolation();
                context.buildConstraintViolationWithTemplate(
                        getTemplate(EntityViolationType.InvalidSchedTask, "Invalid cron expression")).
                        addPropertyNode("cronExpression").addConstraintViolation();
            }
        }

        return isValid;
    }
}
