/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package org.apache.syncope.core.persistence.beans;

import java.util.ArrayList;
import java.util.List;
import javax.persistence.Basic;
import javax.persistence.ElementCollection;
import javax.persistence.EnumType;
import javax.persistence.Enumerated;
import javax.persistence.FetchType;
import javax.persistence.ManyToOne;
import javax.persistence.MappedSuperclass;
import javax.validation.constraints.Max;
import javax.validation.constraints.Min;
import javax.validation.constraints.NotNull;
import org.apache.syncope.common.types.MatchingRule;
import org.apache.syncope.common.types.UnmatchingRule;
import org.apache.syncope.core.persistence.validation.entity.AbstractSyncTaskCheck;

@MappedSuperclass
@AbstractSyncTaskCheck
public abstract class AbstractSyncTask extends SchedTask {

    private static final long serialVersionUID = -4141057723006682562L;

    /**
     * ExternalResource to which the sync happens.
     */
    @ManyToOne
    private ExternalResource resource;

    @Basic
    @Min(0)
    @Max(1)
    private Integer performCreate;

    @Basic
    @Min(0)
    @Max(1)
    private Integer performUpdate;

    @Basic
    @Min(0)
    @Max(1)
    private Integer performDelete;

    @Basic
    @Min(0)
    @Max(1)
    private Integer syncStatus;

    /**
     * @see UnmatchingRule
     */
    @NotNull
    @Enumerated(EnumType.STRING)
    protected UnmatchingRule unmatchingRule;

    /**
     * @see MatchingRule
     */
    @NotNull
    @Enumerated(EnumType.STRING)
    protected MatchingRule matchingRule;

    @ElementCollection(fetch = FetchType.EAGER)
    private List<String> actionsClassNames = new ArrayList<String>();

    public AbstractSyncTask(final String jobClassName) {
        super();
        super.setJobClassName(jobClassName);
    }

    @Override
    public void setJobClassName(final String jobClassName) {
        // fixed to SyncJob, cannot be changed
    }

    public ExternalResource getResource() {
        return resource;
    }

    public void setResource(ExternalResource resource) {
        this.resource = resource;
    }

    public boolean isPerformCreate() {
        return isBooleanAsInteger(performCreate);
    }

    public void setPerformCreate(final boolean performCreate) {
        this.performCreate = getBooleanAsInteger(performCreate);
    }

    public boolean isPerformUpdate() {
        return isBooleanAsInteger(performUpdate);
    }

    public void setPerformUpdate(final boolean performUpdate) {
        this.performUpdate = getBooleanAsInteger(performUpdate);
    }

    public boolean isPerformDelete() {
        return isBooleanAsInteger(performDelete);
    }

    public void setPerformDelete(boolean performDelete) {
        this.performDelete = getBooleanAsInteger(performDelete);
    }

    public boolean isSyncStatus() {
        return isBooleanAsInteger(syncStatus);
    }

    public void setSyncStatus(final boolean syncStatus) {
        this.syncStatus = getBooleanAsInteger(syncStatus);
    }

    public UnmatchingRule getUnmatchingRule() {
        return this.unmatchingRule;
    }

    public void setUnmatchingRule(final UnmatchingRule unmatchigRule) {
        this.unmatchingRule = unmatchigRule;
    }

    public MatchingRule getMatchingRule() {
        return this.matchingRule;
    }

    public void setMatchingRule(final MatchingRule matchigRule) {
        this.matchingRule = matchigRule;
    }

    public List<String> getActionsClassNames() {
        return actionsClassNames;
    }
}
