/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */

package org.apache.streams.util;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.lang.management.ManagementFactory;
import java.util.Queue;
import java.util.Set;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.TimeUnit;
import javax.management.InstanceAlreadyExistsException;
import javax.management.MBeanRegistrationException;
import javax.management.MBeanServer;
import javax.management.MalformedObjectNameException;
import javax.management.NotCompliantMBeanException;
import javax.management.ObjectName;

/**
 * Common utilities for Streams components.
 */
public class ComponentUtils {
  private static final Logger LOGGER = LoggerFactory.getLogger(ComponentUtils.class);

  /**
   * Certain types of queues will fail to {@link java.util.Queue#offer(Object)} an item due to many factors
   * depending on the type of queue. <code>offerUntilSuccess</code> will not return until the item has been
   * successfully queued onto the desired queue
   * @param entry item to queue
   * @param queue queue to add the entry to
   * @param <T> type
   */
  public static <T> void offerUntilSuccess(T entry, Queue<T> queue) {
    boolean success;
    do {
      success = queue.offer(entry);
      Thread.yield();
    }
    while ( !success );
  }

  /**
   * Certain types of queues will return null when calling {@link java.util.Queue#poll()} due to many factors depending
   * on the type of queue.  <code>pollWhileNotEmpty</code> will poll the queue until an item from the queue is returned
   * or the queue is empty.  If the queue is empty it will return NULL.
   * @param queue queue to read the entry from
   * @param <T> type
   * @return result
   */
  public static <T> T pollWhileNotEmpty(Queue<T> queue) {
    T item = queue.poll();
    while (!queue.isEmpty() && item == null) {
      Thread.yield();
      item = queue.poll();
    }
    return item;
  }

  /**
   * Attempts to safely {@link java.util.concurrent.ExecutorService#shutdown()}
   * and {@link java.util.concurrent.ExecutorService#awaitTermination(long, java.util.concurrent.TimeUnit)}
   * of an {@link java.util.concurrent.ExecutorService}.
   * @param stream service to be shutdown
   * @param initialWait time in seconds to wait for currently running threads to finish execution
   * @param secondaryWait time in seconds to wait for running threads that did not terminate to acknowledge their forced termination
   */
  public static void shutdownExecutor(ExecutorService stream, int initialWait, int secondaryWait) {
    stream.shutdown();
    try {
      if (!stream.awaitTermination(initialWait, TimeUnit.SECONDS)) {
        stream.shutdownNow();
        if (!stream.awaitTermination(secondaryWait, TimeUnit.SECONDS)) {
          LOGGER.error("Executor Service did not terminate");
        }
      }
    } catch (InterruptedException ie) {
      stream.shutdownNow();
      Thread.currentThread().interrupt();
    }
  }

  /**
   * Removes all mbeans registered undered a specific domain.  Made specificly to clean up at unit tests
   * @param domain mbean domain
   */
  public static void removeAllMBeansOfDomain(String domain) throws Exception {
    MBeanServer mbs = ManagementFactory.getPlatformMBeanServer();
    domain = domain.endsWith(":") ? domain : domain + ":";
    ObjectName objectName = new ObjectName(domain + "*");
    Set<ObjectName> mbeanNames = mbs.queryNames(objectName, null);
    for (ObjectName name : mbeanNames) {
      mbs.unregisterMBean(name);
    }
  }

  /**
   * Attempts to register an object with local MBeanServer.  Throws runtime exception on errors.
   * @param name name to register bean with
   * @param mbean mbean to register
   */
  public static <V> void registerLocalMBean(String name, V mbean) {
    try {
      ObjectName objectName = new ObjectName(name);
      MBeanServer mbs = ManagementFactory.getPlatformMBeanServer();
      mbs.registerMBean(mbean, objectName);
    } catch (MalformedObjectNameException | InstanceAlreadyExistsException | MBeanRegistrationException | NotCompliantMBeanException ex) {
      LOGGER.error("Failed to register MXBean : {}", ex);
      throw new RuntimeException(ex);
    }
  }

}
