/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */

package org.apache.streams.sysomos.provider;

import org.apache.streams.sysomos.SysomosException;

import org.joda.time.DateTime;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.net.MalformedURLException;
import java.net.URL;

import static org.apache.streams.sysomos.util.SysomosUtils.SYSOMOS_DATE_FORMATTER;

/**
 * Builds requests for the Sysomos Heartbeat Content API.  This is the preferred method of
 * accessing data from Sysomoos Heartbeat
 */
public class ContentRequestBuilder extends AbstractRequestBuilder implements RequestBuilder {

  private static final Logger LOGGER = LoggerFactory.getLogger(ContentRequestBuilder.class);

  private String baseUrl;
  private String hid;
  private String addedAfter;
  private String addedBefore;
  private String size;
  private String offset;
  private String apiKey;

  /**
   * The max number of items you are allowed to get per request.
   */
  public static final int MAX_ALLOWED_PER_REQUEST = 10000;

  /**
   * Constructs a new ContentRequestBuilder for the specified API key and Sysomos URL.
   * @param baseUrl the base URL for the sysomos API
   * @param apiKey the API key generated by Sysomos for authorization
   */
  protected ContentRequestBuilder(String baseUrl, String apiKey) {
    this.baseUrl = baseUrl;
    this.apiKey = apiKey;
  }

  /**
   * Gets the Request URL based on the local fields.
   * @return a valid URL for the Sysomos API or an exception
   */
  @Override
  public URL getRequestUrl()  {
    StringBuilder url = new StringBuilder();
    url.append(this.baseUrl);
    url.append("v1/heartbeat/content?");
    url.append("apiKey=");
    url.append(this.apiKey);
    url.append("&hid=");
    url.append(this.hid);
    if (size != null) {
      url.append("&size=");
      url.append(this.size);
    }
    if (this.addedAfter != null) {
      url.append("&addedAfter=");
      url.append(this.addedAfter);
    }
    if (this.addedBefore != null) {
      url.append("&addedBefore=");
      url.append(this.addedBefore);
    }
    if (this.offset != null) {
      url.append("&offset=");
      url.append(this.offset);
    }
    String urlString = url.toString();
    LOGGER.debug("Constructed Request URL: {}", urlString);
    try {
      return new URL(urlString);
    } catch (MalformedURLException ex) {
      throw new SysomosException("Malformed Request URL.  Check Request Builder parameters", ex);
    }
  }

  @Override
  public RequestBuilder setHeartBeatId(int hid) {
    return setHeartBeatId(Integer.toString(hid));
  }

  @Override
  public RequestBuilder setHeartBeatId(String hid) {
    this.hid = hid;
    return this;
  }

  @Override
  public RequestBuilder setAddedAfterDate(DateTime afterDate) {
    this.addedAfter = SYSOMOS_DATE_FORMATTER.print(afterDate);
    return this;
  }

  @Override
  public RequestBuilder setAddedBeforeDate(DateTime beforeDate) {
    this.addedBefore = SYSOMOS_DATE_FORMATTER.print(beforeDate);
    return this;
  }

  @Override
  public RequestBuilder setReturnSetSize(long size) {
    this.size = Long.toString(Math.min(size, MAX_ALLOWED_PER_REQUEST));
    return this;
  }

  @Override
  public RequestBuilder setOffset(long offset) {
    this.offset = Long.toString(offset);
    return this;
  }

}
