/*
 * Decompiled with CFR 0.152.
 */
package org.apache.storm.windowing.persistence;

import java.util.AbstractCollection;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Deque;
import java.util.HashSet;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.Map;
import java.util.NoSuchElementException;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.ConcurrentLinkedQueue;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.locks.ReentrantLock;
import java.util.function.Supplier;
import org.apache.storm.shade.com.google.common.collect.ImmutableMap;
import org.apache.storm.state.KeyValueState;
import org.apache.storm.windowing.Event;
import org.apache.storm.windowing.persistence.SimpleWindowPartitionCache;
import org.apache.storm.windowing.persistence.WindowPartitionCache;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class WindowState<T>
extends AbstractCollection<Event<T>> {
    public static final int MAX_PARTITION_EVENTS = 1000;
    public static final int MIN_PARTITIONS = 10;
    private static final Logger LOG = LoggerFactory.getLogger(WindowState.class);
    private static final String PARTITION_IDS_KEY = "pk";
    private final KeyValueState<String, Deque<Long>> partitionIdsState;
    private final KeyValueState<Long, WindowPartition<T>> windowPartitionsState;
    private final KeyValueState<String, Optional<?>> windowSystemState;
    private final ReentrantLock partitionIdsLock = new ReentrantLock(true);
    private final WindowPartitionLock windowPartitionsLock = new WindowPartitionLock();
    private final long maxEventsInMemory;
    private volatile Deque<Long> partitionIds;
    private volatile long latestPartitionId;
    private volatile WindowPartition<T> latestPartition;
    private volatile WindowPartitionCache<Long, WindowPartition<T>> cache;
    private Supplier<Map<String, Optional<?>>> windowSystemStateSupplier;
    private Set<Long> iteratorPins = new HashSet<Long>();

    public WindowState(KeyValueState<Long, WindowPartition<T>> windowPartitionsState, KeyValueState<String, Deque<Long>> partitionIdsState, KeyValueState<String, Optional<?>> windowSystemState, Supplier<Map<String, Optional<?>>> windowSystemStateSupplier, long maxEventsInMemory) {
        this.windowPartitionsState = windowPartitionsState;
        this.partitionIdsState = partitionIdsState;
        this.windowSystemState = windowSystemState;
        this.windowSystemStateSupplier = windowSystemStateSupplier;
        this.maxEventsInMemory = Math.max(10000L, maxEventsInMemory);
        this.init();
    }

    @Override
    public boolean add(Event<T> event) {
        if (this.latestPartition.size() >= 1000) {
            this.cache.unpin(this.latestPartition.getId());
            this.latestPartition = this.getPinnedPartition(this.getNextPartitionId());
        }
        this.latestPartition.add(event);
        return true;
    }

    @Override
    public Iterator<Event<T>> iterator() {
        return new Iterator<Event<T>>(){
            private Iterator<Long> ids = this.getIds();
            private Iterator<Event<T>> current = Collections.emptyIterator();
            private Iterator<Event<T>> removeFrom;
            private WindowPartition<T> curPartition;

            private Iterator<Long> getIds() {
                try {
                    WindowState.this.partitionIdsLock.lock();
                    LOG.debug("Iterator partitionIds: {}", (Object)WindowState.this.partitionIds);
                    Iterator<Long> iterator = new ArrayList(WindowState.this.partitionIds).iterator();
                    return iterator;
                }
                finally {
                    WindowState.this.partitionIdsLock.unlock();
                }
            }

            @Override
            public void remove() {
                if (this.removeFrom == null) {
                    throw new IllegalStateException("No calls to next() since last call to remove()");
                }
                this.removeFrom.remove();
                this.removeFrom = null;
            }

            @Override
            public boolean hasNext() {
                boolean curHasNext = this.current.hasNext();
                while (!curHasNext && this.ids.hasNext()) {
                    if (this.curPartition != null) {
                        this.unpin(this.curPartition.getId());
                    }
                    this.curPartition = WindowState.this.getPinnedPartition(this.ids.next());
                    if (this.curPartition == null) continue;
                    WindowState.this.iteratorPins.add(this.curPartition.getId());
                    this.current = this.curPartition.iterator();
                    curHasNext = this.current.hasNext();
                }
                if (!curHasNext && this.curPartition != null) {
                    this.unpin(this.curPartition.getId());
                    this.curPartition = null;
                }
                return curHasNext;
            }

            @Override
            public Event<T> next() {
                if (!this.hasNext()) {
                    throw new NoSuchElementException();
                }
                this.removeFrom = this.current;
                return this.current.next();
            }

            private void unpin(long id) {
                WindowState.this.cache.unpin(id);
                WindowState.this.iteratorPins.remove(id);
            }
        };
    }

    public void clearIteratorPins() {
        LOG.debug("clearIteratorPins '{}'", this.iteratorPins);
        Iterator<Long> it = this.iteratorPins.iterator();
        while (it.hasNext()) {
            this.cache.unpin(it.next());
            it.remove();
        }
    }

    @Override
    public int size() {
        throw new UnsupportedOperationException();
    }

    public void prepareCommit(long txid) {
        this.flush();
        this.partitionIdsState.prepareCommit(txid);
        this.windowPartitionsState.prepareCommit(txid);
        this.windowSystemState.prepareCommit(txid);
    }

    public void commit(long txid) {
        this.partitionIdsState.commit(txid);
        this.windowPartitionsState.commit(txid);
        this.windowSystemState.commit(txid);
    }

    public void rollback(boolean reInit) {
        this.partitionIdsState.rollback();
        this.windowPartitionsState.rollback();
        this.windowSystemState.rollback();
        if (reInit) {
            this.init();
        }
    }

    private void init() {
        this.initCache();
        this.initPartitions();
    }

    private void initPartitions() {
        this.partitionIds = this.partitionIdsState.get(PARTITION_IDS_KEY, new LinkedList());
        if (this.partitionIds.isEmpty()) {
            this.partitionIds.add(0L);
            this.partitionIdsState.put(PARTITION_IDS_KEY, this.partitionIds);
        }
        this.latestPartitionId = this.partitionIds.peekLast();
        this.latestPartition = this.cache.pinAndGet(this.latestPartitionId);
    }

    private void initCache() {
        long size = this.maxEventsInMemory / 1000L;
        LOG.info("maxEventsInMemory: {}, partition size: {}, number of partitions: {}", new Object[]{this.maxEventsInMemory, 1000, size});
        this.cache = ((SimpleWindowPartitionCache.SimpleWindowPartitionCacheBuilder)((SimpleWindowPartitionCache.SimpleWindowPartitionCacheBuilder)SimpleWindowPartitionCache.newBuilder().maximumSize(size)).removalListener(new WindowPartitionCache.RemovalListener<Long, WindowPartition<T>>(){

            /*
             * WARNING - Removed try catching itself - possible behaviour change.
             */
            @Override
            public void onRemoval(Long pid, WindowPartition<T> p, WindowPartitionCache.RemovalCause removalCause) {
                Objects.requireNonNull(pid, "Null partition id");
                Objects.requireNonNull(p, "Null window partition");
                LOG.debug("onRemoval for id '{}', WindowPartition '{}'", (Object)pid, p);
                try {
                    WindowState.this.windowPartitionsLock.lock(pid);
                    if (p.isEmpty() && pid != WindowState.this.latestPartitionId) {
                        if (removalCause != WindowPartitionCache.RemovalCause.EXPLICIT) {
                            WindowState.this.deletePartition(pid);
                            WindowState.this.windowPartitionsState.delete(pid);
                        }
                    } else if (p.isModified()) {
                        WindowState.this.windowPartitionsState.put(pid, p);
                    } else {
                        LOG.debug("WindowPartition '{}' is not modified", (Object)pid);
                    }
                }
                finally {
                    WindowState.this.windowPartitionsLock.unlock(pid);
                }
            }
        })).build(new WindowPartitionCache.CacheLoader<Long, WindowPartition<T>>(){

            @Override
            public WindowPartition<T> load(Long id) {
                LOG.debug("Load partition: {}", (Object)id);
                try {
                    WindowState.this.windowPartitionsLock.lock(id);
                    WindowPartition windowPartition = WindowState.this.windowPartitionsState.get(id, new WindowPartition(id));
                    return windowPartition;
                }
                finally {
                    WindowState.this.windowPartitionsLock.unlock(id);
                }
            }
        });
    }

    private void deletePartition(long pid) {
        LOG.debug("Delete partition: {}", (Object)pid);
        try {
            this.partitionIdsLock.lock();
            this.partitionIds.remove(pid);
            this.partitionIdsState.put(PARTITION_IDS_KEY, this.partitionIds);
        }
        finally {
            this.partitionIdsLock.unlock();
        }
    }

    private long getNextPartitionId() {
        try {
            this.partitionIdsLock.lock();
            this.partitionIds.add(++this.latestPartitionId);
            this.partitionIdsState.put(PARTITION_IDS_KEY, this.partitionIds);
        }
        finally {
            this.partitionIdsLock.unlock();
        }
        return this.latestPartitionId;
    }

    private WindowPartition<T> getPinnedPartition(long id) {
        return this.cache.pinAndGet(id);
    }

    private void flush() {
        LOG.debug("Flushing modified partitions");
        this.cache.asMap().forEach((? super K pid, ? super V p) -> {
            Long pidToInvalidate = null;
            try {
                this.windowPartitionsLock.lock(pid);
                if (p.isEmpty() && pid != this.latestPartitionId) {
                    LOG.debug("Invalidating empty partition {}", pid);
                    this.deletePartition((long)pid);
                    this.windowPartitionsState.delete((Long)pid);
                    pidToInvalidate = pid;
                } else if (p.isModified()) {
                    LOG.debug("Updating modified partition {}", pid);
                    p.clearModified();
                    this.windowPartitionsState.put((Long)pid, (WindowPartition<T>)p);
                }
            }
            finally {
                this.windowPartitionsLock.unlock(pid);
            }
            if (pidToInvalidate != null) {
                this.cache.invalidate(pidToInvalidate);
            }
        });
        Map<String, Optional<?>> state = this.windowSystemStateSupplier.get();
        for (Map.Entry<String, Optional<?>> entry : state.entrySet()) {
            this.windowSystemState.put(entry.getKey(), entry.getValue());
        }
    }

    public static class WindowPartition<T>
    implements Iterable<Event<T>> {
        private final ConcurrentLinkedQueue<Event<T>> events = new ConcurrentLinkedQueue();
        private final AtomicInteger size = new AtomicInteger();
        private final long id;
        private volatile transient boolean modified;

        public WindowPartition(long id) {
            this.id = id;
        }

        void add(Event<T> event) {
            this.events.add(event);
            this.size.incrementAndGet();
            this.setModified();
        }

        boolean isModified() {
            return this.modified;
        }

        void setModified() {
            if (!this.modified) {
                this.modified = true;
            }
        }

        void clearModified() {
            this.modified = false;
        }

        boolean isEmpty() {
            return this.events.isEmpty();
        }

        @Override
        public Iterator<Event<T>> iterator() {
            return new Iterator<Event<T>>(){
                Iterator<Event<T>> it;
                {
                    this.it = events.iterator();
                }

                @Override
                public boolean hasNext() {
                    return this.it.hasNext();
                }

                @Override
                public Event<T> next() {
                    return this.it.next();
                }

                @Override
                public void remove() {
                    this.it.remove();
                    size.decrementAndGet();
                    this.setModified();
                }
            };
        }

        public int size() {
            return this.size.get();
        }

        public long getId() {
            return this.id;
        }

        public Collection<Event<T>> getEvents() {
            return Collections.unmodifiableCollection(this.events);
        }

        public String toString() {
            return "WindowPartition{id=" + this.id + ", size=" + this.size + '}';
        }
    }

    private static class WindowPartitionLock {
        private final int numLocks = 8;
        private final ImmutableMap<Long, ReentrantLock> locks;

        WindowPartitionLock() {
            ImmutableMap.Builder builder = ImmutableMap.builder();
            for (long i = 0L; i < 8L; ++i) {
                builder.put((Object)i, (Object)new ReentrantLock(true));
            }
            this.locks = builder.build();
        }

        private void lock(long i) {
            ((ReentrantLock)this.locks.get((Object)(i % 8L))).lock();
        }

        private void unlock(long i) {
            ((ReentrantLock)this.locks.get((Object)(i % 8L))).unlock();
        }
    }
}

