/*
 * Decompiled with CFR 0.152.
 */
package org.apache.storm.utils;

import com.codahale.metrics.Meter;
import java.io.BufferedReader;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.util.Map;
import java.util.Timer;
import java.util.TimerTask;
import java.util.concurrent.atomic.AtomicBoolean;
import org.apache.storm.utils.DefaultShellLogHandler;
import org.apache.storm.utils.ShellLogHandler;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public abstract class ShellUtils {
    public static final Logger LOG = LoggerFactory.getLogger(ShellUtils.class);
    public static final OSType osType = ShellUtils.getOSType();
    public static final boolean WINDOWS = osType == OSType.OS_TYPE_WIN;
    public static final boolean SOLARIS = osType == OSType.OS_TYPE_SOLARIS;
    public static final boolean MAC = osType == OSType.OS_TYPE_MAC;
    public static final boolean FREEBSD = osType == OSType.OS_TYPE_FREEBSD;
    public static final boolean LINUX = osType == OSType.OS_TYPE_LINUX;
    public static final boolean OTHER = osType == OSType.OS_TYPE_OTHER;
    public static final Meter numShellExceptions = new Meter();
    public static final String TOKEN_SEPARATOR_REGEX = WINDOWS ? "[|\n\r]" : "[ \t\n\r\f]";
    private final boolean redirectErrorStream;
    protected long timeOutInterval = 0L;
    private long interval;
    private long lastTime;
    private Map<String, String> environment;
    private File dir;
    private Process process;
    private int exitCode;
    private AtomicBoolean timedOut;
    private volatile AtomicBoolean completed;

    public ShellUtils() {
        this(0L);
    }

    public ShellUtils(long interval) {
        this(interval, false);
    }

    public ShellUtils(long interval, boolean redirectErrorStream) {
        this.interval = interval;
        this.lastTime = interval < 0L ? 0L : -interval;
        this.redirectErrorStream = redirectErrorStream;
    }

    private static OSType getOSType() {
        String osName = System.getProperty("os.name");
        if (osName.startsWith("Windows")) {
            return OSType.OS_TYPE_WIN;
        }
        if (osName.contains("SunOS") || osName.contains("Solaris")) {
            return OSType.OS_TYPE_SOLARIS;
        }
        if (osName.contains("Mac")) {
            return OSType.OS_TYPE_MAC;
        }
        if (osName.contains("FreeBSD")) {
            return OSType.OS_TYPE_FREEBSD;
        }
        if (osName.startsWith("Linux")) {
            return OSType.OS_TYPE_LINUX;
        }
        return OSType.OS_TYPE_OTHER;
    }

    public static String[] getGroupsCommand() {
        if (WINDOWS) {
            throw new UnsupportedOperationException("Getting user groups is not supported on Windows");
        }
        return new String[]{"bash", "-c", "groups"};
    }

    public static String[] getGroupsForUserCommand(String user) {
        if (WINDOWS) {
            throw new UnsupportedOperationException("Getting user groups is not supported on Windows");
        }
        return new String[]{"bash", "-c", "id -gn " + user + "&& id -Gn " + user};
    }

    private static void joinThread(Thread t) {
        while (t.isAlive()) {
            try {
                t.join();
            }
            catch (InterruptedException ie) {
                if (LOG.isWarnEnabled()) {
                    LOG.warn("Interrupted while joining on: " + t, (Throwable)ie);
                }
                t.interrupt();
            }
        }
    }

    public static ShellLogHandler getLogHandler(Map<String, Object> topoConf) {
        if (topoConf == null) {
            throw new IllegalArgumentException("Config is required");
        }
        String logHandlerClassName = null;
        if (topoConf.containsKey("topology.multilang.log.handler")) {
            try {
                logHandlerClassName = topoConf.get("topology.multilang.log.handler").toString();
                return (ShellLogHandler)Class.forName(logHandlerClassName).newInstance();
            }
            catch (ClassCastException | ClassNotFoundException | IllegalAccessException | InstantiationException e) {
                throw new RuntimeException("Error loading ShellLogHandler " + logHandlerClassName, e);
            }
        }
        return new DefaultShellLogHandler();
    }

    public int getExitCode() {
        return this.exitCode;
    }

    protected void setEnvironment(Map<String, String> env) {
        this.environment = env;
    }

    protected void setWorkingDirectory(File dir) {
        this.dir = dir;
    }

    protected void run() throws IOException {
        if (this.lastTime + this.interval > System.currentTimeMillis()) {
            return;
        }
        this.exitCode = 0;
        try {
            this.runCommand();
        }
        catch (IOException e) {
            numShellExceptions.mark();
            throw e;
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void runCommand() throws IOException {
        ProcessBuilder builder = new ProcessBuilder(this.getExecString());
        Timer timeOutTimer = null;
        ShellTimeoutTimerTask timeoutTimerTask = null;
        this.timedOut = new AtomicBoolean(false);
        this.completed = new AtomicBoolean(false);
        if (this.environment != null) {
            builder.environment().putAll(this.environment);
        }
        if (this.dir != null) {
            builder.directory(this.dir);
        }
        builder.redirectErrorStream(this.redirectErrorStream);
        this.process = builder.start();
        if (this.timeOutInterval > 0L) {
            timeOutTimer = new Timer("Shell command timeout");
            timeoutTimerTask = new ShellTimeoutTimerTask(this);
            timeOutTimer.schedule((TimerTask)timeoutTimerTask, this.timeOutInterval);
        }
        final BufferedReader errReader = new BufferedReader(new InputStreamReader(this.process.getErrorStream()));
        BufferedReader inReader = new BufferedReader(new InputStreamReader(this.process.getInputStream()));
        final StringBuffer errMsg = new StringBuffer();
        Thread errThread = new Thread(){

            @Override
            public void run() {
                try {
                    String line = errReader.readLine();
                    while (line != null && !this.isInterrupted()) {
                        errMsg.append(line);
                        errMsg.append(System.getProperty("line.separator"));
                        line = errReader.readLine();
                    }
                }
                catch (IOException ioe) {
                    LOG.warn("Error reading the error stream", (Throwable)ioe);
                }
            }
        };
        try {
            errThread.start();
        }
        catch (IllegalStateException illegalStateException) {
            // empty catch block
        }
        try {
            this.parseExecResult(inReader);
            String line = inReader.readLine();
            while (line != null) {
                line = inReader.readLine();
            }
            this.exitCode = this.process.waitFor();
            ShellUtils.joinThread(errThread);
            this.completed.set(true);
            if (this.exitCode != 0) {
                throw new ExitCodeException(this.exitCode, errMsg.toString());
            }
        }
        catch (InterruptedException ie) {
            throw new IOException(ie.toString());
        }
        finally {
            InputStream inputStream;
            if (timeOutTimer != null) {
                timeOutTimer.cancel();
            }
            try {
                InputStream stdout;
                inputStream = stdout = this.process.getInputStream();
                synchronized (inputStream) {
                    inReader.close();
                }
            }
            catch (IOException ioe) {
                LOG.warn("Error while closing the input stream", (Throwable)ioe);
            }
            if (!this.completed.get()) {
                errThread.interrupt();
                ShellUtils.joinThread(errThread);
            }
            try {
                InputStream stderr;
                inputStream = stderr = this.process.getErrorStream();
                synchronized (inputStream) {
                    errReader.close();
                }
            }
            catch (IOException ioe) {
                LOG.warn("Error while closing the error stream", (Throwable)ioe);
            }
            this.process.destroy();
            this.lastTime = System.currentTimeMillis();
        }
    }

    protected abstract String[] getExecString();

    protected abstract void parseExecResult(BufferedReader var1) throws IOException;

    public Process getProcess() {
        return this.process;
    }

    public boolean isTimedOut() {
        return this.timedOut.get();
    }

    private void setTimedOut() {
        this.timedOut.set(true);
    }

    private static class ShellTimeoutTimerTask
    extends TimerTask {
        private ShellUtils shell;

        public ShellTimeoutTimerTask(ShellUtils shell) {
            this.shell = shell;
        }

        @Override
        public void run() {
            block2: {
                Process p = this.shell.getProcess();
                try {
                    p.exitValue();
                }
                catch (Exception e) {
                    if (p == null || this.shell.completed.get()) break block2;
                    this.shell.setTimedOut();
                    p.destroy();
                }
            }
        }
    }

    public static class ShellCommandExecutor
    extends ShellUtils {
        private String[] command;
        private StringBuffer output;

        public ShellCommandExecutor(String[] execString) {
            this(execString, null);
        }

        public ShellCommandExecutor(String[] execString, File dir) {
            this(execString, dir, null);
        }

        public ShellCommandExecutor(String[] execString, File dir, Map<String, String> env) {
            this(execString, dir, env, 0L);
        }

        public ShellCommandExecutor(String[] execString, File dir, Map<String, String> env, long timeout) {
            this.command = (String[])execString.clone();
            if (dir != null) {
                this.setWorkingDirectory(dir);
            }
            if (env != null) {
                this.setEnvironment(env);
            }
            this.timeOutInterval = timeout;
        }

        public void execute() throws IOException {
            this.run();
        }

        @Override
        public String[] getExecString() {
            return this.command;
        }

        @Override
        protected void parseExecResult(BufferedReader lines) throws IOException {
            int read;
            this.output = new StringBuffer();
            char[] buf = new char[512];
            while ((read = lines.read(buf, 0, buf.length)) > 0) {
                this.output.append(buf, 0, read);
            }
        }

        public String getOutput() {
            return this.output == null ? "" : this.output.toString();
        }

        public String toString() {
            String[] args;
            StringBuilder builder = new StringBuilder();
            for (String s : args = this.getExecString()) {
                if (s.indexOf(32) >= 0) {
                    builder.append('\"').append(s).append('\"');
                } else {
                    builder.append(s);
                }
                builder.append(' ');
            }
            return builder.toString();
        }
    }

    public static class ExitCodeException
    extends IOException {
        int exitCode;

        public ExitCodeException(int exitCode, String message) {
            super(message);
            this.exitCode = exitCode;
        }

        public int getExitCode() {
            return this.exitCode;
        }
    }

    public static enum OSType {
        OS_TYPE_LINUX,
        OS_TYPE_WIN,
        OS_TYPE_SOLARIS,
        OS_TYPE_MAC,
        OS_TYPE_FREEBSD,
        OS_TYPE_OTHER;

    }
}

