/*
 * Decompiled with CFR 0.152.
 */
package org.apache.storm.cassandra.bolt;

import com.datastax.driver.core.Statement;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.TimeUnit;
import org.apache.storm.Config;
import org.apache.storm.cassandra.bolt.BaseCassandraBolt;
import org.apache.storm.cassandra.bolt.GroupingBatchBuilder;
import org.apache.storm.cassandra.bolt.PairBatchStatementTuples;
import org.apache.storm.cassandra.bolt.PairStatementTuple;
import org.apache.storm.cassandra.executor.AsyncResultHandler;
import org.apache.storm.cassandra.executor.impl.BatchAsyncResultHandler;
import org.apache.storm.cassandra.query.CQLStatementTupleMapper;
import org.apache.storm.task.OutputCollector;
import org.apache.storm.task.TopologyContext;
import org.apache.storm.tuple.ITuple;
import org.apache.storm.tuple.Tuple;
import org.apache.storm.utils.Time;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class BatchCassandraWriterBolt
extends BaseCassandraBolt<List<Tuple>> {
    private static final Logger LOG = LoggerFactory.getLogger(BatchCassandraWriterBolt.class);
    public static final int DEFAULT_EMIT_FREQUENCY = 2;
    private LinkedBlockingQueue<Tuple> queue;
    private int tickFrequencyInSeconds;
    private long lastModifiedTimesMillis;
    private int batchMaxSize = 1000;
    private String componentID;
    private AsyncResultHandler<List<Tuple>> asyncResultHandler;

    public BatchCassandraWriterBolt(CQLStatementTupleMapper tupleMapper) {
        this(tupleMapper, 2);
    }

    public BatchCassandraWriterBolt(CQLStatementTupleMapper tupleMapper, int tickFrequencyInSeconds) {
        super(tupleMapper);
        this.tickFrequencyInSeconds = tickFrequencyInSeconds;
    }

    @Override
    public void prepare(Map stormConfig, TopologyContext topologyContext, OutputCollector outputCollector) {
        super.prepare(stormConfig, topologyContext, outputCollector);
        this.componentID = topologyContext.getThisComponentId();
        this.queue = new LinkedBlockingQueue(this.batchMaxSize);
        this.lastModifiedTimesMillis = this.now();
    }

    @Override
    protected AsyncResultHandler<List<Tuple>> getAsyncHandler() {
        if (this.asyncResultHandler == null) {
            this.asyncResultHandler = new BatchAsyncResultHandler(this.getResultHandler());
        }
        return this.asyncResultHandler;
    }

    protected void process(Tuple input) {
        if (!this.queue.offer(input)) {
            LOG.info(this.logPrefix() + "The message queue is full, preparing batch statement...");
            this.prepareAndExecuteStatement();
            this.queue.add(input);
        }
    }

    protected void onTickTuple(Tuple tuple) {
        this.prepareAndExecuteStatement();
    }

    public void prepareAndExecuteStatement() {
        int size = this.queue.size();
        if (size > 0) {
            ArrayList<Tuple> inputs = new ArrayList<Tuple>(size);
            this.queue.drainTo(inputs);
            try {
                List<PairStatementTuple> psl = this.buildStatement(inputs);
                int sinceLastModified = this.updateAndGetSecondsSinceLastModified();
                LOG.debug(this.logPrefix() + "Execute cql batches with {} statements after {} seconds", (Object)size, (Object)sinceLastModified);
                this.checkTimeElapsedSinceLastExec(sinceLastModified);
                GroupingBatchBuilder batchBuilder = new GroupingBatchBuilder(this.cassandraConfConfig.getBatchSizeRows(), psl);
                int batchSize = 0;
                for (PairBatchStatementTuples batch : batchBuilder) {
                    LOG.debug(this.logPrefix() + "Writing data to {} in batches of {} rows.", (Object)this.cassandraConfConfig.getKeyspace(), (Object)batch.getInputs().size());
                    this.getAsyncExecutor().execAsync((Statement)batch.getStatement(), batch.getInputs());
                    ++batchSize;
                }
                int pending = this.getAsyncExecutor().getPendingTasksSize();
                if (pending > batchSize) {
                    LOG.warn(this.logPrefix() + "Currently pending tasks is superior to the number of submit batches({}) : {}", (Object)batchSize, (Object)pending);
                }
            }
            catch (Throwable r) {
                LOG.error(this.logPrefix() + "Error(s) occurred while preparing batch statements");
                this.getAsyncHandler().failure(r, inputs);
            }
        }
    }

    private List<PairStatementTuple> buildStatement(List<Tuple> inputs) {
        ArrayList<PairStatementTuple> stmts = new ArrayList<PairStatementTuple>(inputs.size());
        for (Tuple t : inputs) {
            List<Statement> sl = this.getMapper().map(this.stormConfig, this.session, (ITuple)t);
            for (Statement s : sl) {
                stmts.add(new PairStatementTuple(t, s));
            }
        }
        return stmts;
    }

    private void checkTimeElapsedSinceLastExec(int sinceLastModified) {
        if (sinceLastModified > this.tickFrequencyInSeconds) {
            LOG.warn(this.logPrefix() + "The time elapsed since last execution exceeded tick tuple frequency - {} > {} seconds", (Object)sinceLastModified, (Object)this.tickFrequencyInSeconds);
        }
    }

    private String logPrefix() {
        return this.componentID + " - ";
    }

    public BatchCassandraWriterBolt withTickFrequency(long time, TimeUnit unit) {
        this.tickFrequencyInSeconds = (int)unit.toSeconds(time);
        return this;
    }

    public BatchCassandraWriterBolt withQueueSize(int size) {
        this.batchMaxSize = size;
        return this;
    }

    @Override
    public Map<String, Object> getComponentConfiguration() {
        Config conf = new Config();
        conf.put((Object)"topology.tick.tuple.freq.secs", (Object)this.tickFrequencyInSeconds);
        return conf;
    }

    private int updateAndGetSecondsSinceLastModified() {
        long now = this.now();
        int seconds = (int)(now - this.lastModifiedTimesMillis) / 1000;
        this.lastModifiedTimesMillis = now;
        return seconds;
    }

    private long now() {
        return Time.currentTimeMillis();
    }
}

