/*
 * Decompiled with CFR 0.152.
 */
package org.apache.storm.hdfs.spout;

import java.io.IOException;
import java.lang.reflect.Constructor;
import java.net.URI;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Timer;
import java.util.TimerTask;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.atomic.AtomicBoolean;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.FileSystem;
import org.apache.hadoop.fs.Path;
import org.apache.storm.hdfs.common.HdfsUtils;
import org.apache.storm.hdfs.common.security.HdfsSecurityUtil;
import org.apache.storm.hdfs.spout.DirLock;
import org.apache.storm.hdfs.spout.FileLock;
import org.apache.storm.hdfs.spout.FileOffset;
import org.apache.storm.hdfs.spout.FileReader;
import org.apache.storm.hdfs.spout.ParseException;
import org.apache.storm.hdfs.spout.ProgressTracker;
import org.apache.storm.hdfs.spout.SequenceFileReader;
import org.apache.storm.hdfs.spout.TextFileReader;
import org.apache.storm.spout.SpoutOutputCollector;
import org.apache.storm.task.TopologyContext;
import org.apache.storm.topology.OutputFieldsDeclarer;
import org.apache.storm.topology.base.BaseRichSpout;
import org.apache.storm.tuple.Fields;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class HdfsSpout
extends BaseRichSpout {
    private String hdfsUri;
    private String readerType;
    private Fields outputFields;
    private Path sourceDirPath;
    private Path archiveDirPath;
    private Path badFilesDirPath;
    private Path lockDirPath;
    private int commitFrequencyCount = 20000;
    private int commitFrequencySec = 10;
    private int maxOutstanding = 10000;
    private int lockTimeoutSec = 300;
    private boolean clocksInSync = true;
    private String inprogress_suffix = ".inprogress";
    private String ignoreSuffix = ".ignore";
    private static final Logger LOG = LoggerFactory.getLogger(HdfsSpout.class);
    private ProgressTracker tracker = null;
    private FileSystem hdfs;
    private FileReader reader;
    private SpoutOutputCollector collector;
    HashMap<MessageId, List<Object>> inflight = new HashMap();
    LinkedBlockingQueue<HdfsUtils.Pair<MessageId, List<Object>>> retryList = new LinkedBlockingQueue();
    private Configuration hdfsConfig;
    private Map conf = null;
    private FileLock lock;
    private String spoutId = null;
    HdfsUtils.Pair<Path, FileLock.LogEntry> lastExpiredLock = null;
    private long lastExpiredLockTime = 0L;
    private long tupleCounter = 0L;
    private boolean ackEnabled = false;
    private int acksSinceLastCommit = 0;
    private final AtomicBoolean commitTimeElapsed = new AtomicBoolean(false);
    private Timer commitTimer;
    private boolean fileReadCompletely = true;
    private String configKey = "hdfs.config";

    public HdfsSpout withOutputFields(String ... fields) {
        this.outputFields = new Fields(fields);
        return this;
    }

    public HdfsSpout withConfigKey(String configKey) {
        this.configKey = configKey;
        return this;
    }

    public Path getLockDirPath() {
        return this.lockDirPath;
    }

    public SpoutOutputCollector getCollector() {
        return this.collector;
    }

    public void nextTuple() {
        LOG.trace("Next Tuple {}", (Object)this.spoutId);
        if (!this.retryList.isEmpty()) {
            LOG.debug("Sending tuple from retry list");
            HdfsUtils.Pair pair = (HdfsUtils.Pair)this.retryList.remove();
            this.emitData((List)pair.getValue(), (MessageId)pair.getKey());
            return;
        }
        if (this.ackEnabled && this.tracker.size() >= this.maxOutstanding) {
            LOG.warn("Waiting for more ACKs before generating new tuples. Progress tracker size has reached limit {}, SpoutID {}", (Object)this.maxOutstanding, (Object)this.spoutId);
            return;
        }
        while (true) {
            try {
                while (true) {
                    if (this.reader == null) {
                        this.reader = this.pickNextFile();
                        if (this.reader == null) {
                            LOG.debug("Currently no new files to process under : " + this.sourceDirPath);
                            return;
                        }
                        this.fileReadCompletely = false;
                    }
                    if (this.fileReadCompletely) {
                        return;
                    }
                    List<Object> tuple = this.reader.next();
                    if (tuple != null) {
                        this.fileReadCompletely = false;
                        ++this.tupleCounter;
                        MessageId msgId = new MessageId(this.tupleCounter, this.reader.getFilePath(), this.reader.getFileOffset());
                        this.emitData(tuple, msgId);
                        if (!this.ackEnabled) {
                            ++this.acksSinceLastCommit;
                            this.commitProgress(this.reader.getFileOffset());
                        } else {
                            this.commitProgress(this.tracker.getCommitPosition());
                        }
                        return;
                    }
                    this.fileReadCompletely = true;
                    if (this.ackEnabled) continue;
                    this.markFileAsDone(this.reader.getFilePath());
                }
            }
            catch (IOException e) {
                LOG.error("I/O Error processing at file location " + HdfsSpout.getFileProgress(this.reader), (Throwable)e);
                return;
            }
            catch (ParseException e) {
                LOG.error("Parsing error when processing at file location " + HdfsSpout.getFileProgress(this.reader) + ". Skipping remainder of file.", (Throwable)e);
                this.markFileAsBad(this.reader.getFilePath());
                continue;
            }
            break;
        }
    }

    private void commitProgress(FileOffset position) {
        if (position == null) {
            return;
        }
        if (this.lock != null && this.canCommitNow()) {
            try {
                String pos = position.toString();
                this.lock.heartbeat(pos);
                LOG.debug("{} Committed progress. {}", (Object)this.spoutId, (Object)pos);
                this.acksSinceLastCommit = 0;
                this.commitTimeElapsed.set(false);
                this.setupCommitElapseTimer();
            }
            catch (IOException e) {
                LOG.error("Unable to commit progress Will retry later. Spout ID = " + this.spoutId, (Throwable)e);
            }
        }
    }

    private void setupCommitElapseTimer() {
        if (this.commitFrequencySec <= 0) {
            return;
        }
        TimerTask timerTask = new TimerTask(){

            @Override
            public void run() {
                HdfsSpout.this.commitTimeElapsed.set(true);
            }
        };
        this.commitTimer.schedule(timerTask, this.commitFrequencySec * 1000);
    }

    private static String getFileProgress(FileReader reader) {
        return reader.getFilePath() + " " + reader.getFileOffset();
    }

    private void markFileAsDone(Path filePath) {
        try {
            Path newFile = this.renameCompletedFile(this.reader.getFilePath());
            LOG.info("Completed processing {}. Spout Id = {}", (Object)newFile, (Object)this.spoutId);
        }
        catch (IOException e) {
            LOG.error("Unable to archive completed file" + filePath + " Spout ID " + this.spoutId, (Throwable)e);
        }
        this.closeReaderAndResetTrackers();
    }

    private void markFileAsBad(Path file) {
        String fileName = file.toString();
        String fileNameMinusSuffix = fileName.substring(0, fileName.indexOf(this.inprogress_suffix));
        String originalName = new Path(fileNameMinusSuffix).getName();
        Path newFile = new Path(this.badFilesDirPath + "/" + originalName);
        LOG.info("Moving bad file {} to {}. Processed it till offset {}. SpoutID= {}", new Object[]{originalName, newFile, this.tracker.getCommitPosition(), this.spoutId});
        try {
            if (!this.hdfs.rename(file, newFile)) {
                throw new IOException("Move failed for bad file: " + file);
            }
        }
        catch (IOException e) {
            LOG.warn("Error moving bad file: " + file + " to destination " + newFile + " SpoutId =" + this.spoutId, (Throwable)e);
        }
        this.closeReaderAndResetTrackers();
    }

    private void closeReaderAndResetTrackers() {
        this.inflight.clear();
        this.tracker.offsets.clear();
        this.retryList.clear();
        this.reader.close();
        this.reader = null;
        HdfsSpout.releaseLockAndLog(this.lock, this.spoutId);
        this.lock = null;
    }

    private static void releaseLockAndLog(FileLock fLock, String spoutId) {
        try {
            if (fLock != null) {
                fLock.release();
                LOG.debug("Spout {} released FileLock. SpoutId = {}", (Object)fLock.getLockFile(), (Object)spoutId);
            }
        }
        catch (IOException e) {
            LOG.error("Unable to delete lock file : " + fLock.getLockFile() + " SpoutId =" + spoutId, (Throwable)e);
        }
    }

    protected void emitData(List<Object> tuple, MessageId id) {
        LOG.trace("Emitting - {}", (Object)id);
        this.collector.emit(tuple, (Object)id);
        this.inflight.put(id, tuple);
    }

    public void open(Map conf, TopologyContext context, SpoutOutputCollector collector) {
        Object ackers;
        Map map;
        LOG.info("Opening HDFS Spout");
        this.conf = conf;
        this.commitTimer = new Timer();
        this.tracker = new ProgressTracker();
        this.hdfsConfig = new Configuration();
        this.collector = collector;
        this.hdfsConfig = new Configuration();
        this.tupleCounter = 0L;
        if (!conf.containsKey("hdfsspout.hdfs")) {
            throw new RuntimeException("hdfsspout.hdfs setting is required");
        }
        this.hdfsUri = conf.get("hdfsspout.hdfs").toString();
        try {
            this.hdfs = FileSystem.get((URI)URI.create(this.hdfsUri), (Configuration)this.hdfsConfig);
        }
        catch (IOException e) {
            LOG.error("Unable to instantiate file system", (Throwable)e);
            throw new RuntimeException("Unable to instantiate file system", e);
        }
        if (conf.containsKey(this.configKey) && (map = (Map)conf.get(this.configKey)) != null) {
            for (String keyName : map.keySet()) {
                LOG.info("HDFS Config override : {} = {} ", (Object)keyName, (Object)String.valueOf(map.get(keyName)));
                this.hdfsConfig.set(keyName, String.valueOf(map.get(keyName)));
            }
            try {
                HdfsSecurityUtil.login(conf, this.hdfsConfig);
            }
            catch (IOException e) {
                LOG.error("HDFS Login failed ", (Throwable)e);
                throw new RuntimeException(e);
            }
        }
        if (conf.containsKey("hdfsspout.reader.type")) {
            this.readerType = conf.get("hdfsspout.reader.type").toString();
            HdfsSpout.checkValidReader(this.readerType);
        }
        if (!conf.containsKey("hdfsspout.source.dir")) {
            LOG.error("hdfsspout.source.dir setting is required");
            throw new RuntimeException("hdfsspout.source.dir setting is required");
        }
        this.sourceDirPath = new Path(conf.get("hdfsspout.source.dir").toString());
        if (!conf.containsKey("hdfsspout.archive.dir")) {
            LOG.error("hdfsspout.archive.dir setting is required");
            throw new RuntimeException("hdfsspout.archive.dir setting is required");
        }
        this.archiveDirPath = new Path(conf.get("hdfsspout.archive.dir").toString());
        HdfsSpout.validateOrMakeDir(this.hdfs, this.archiveDirPath, "Archive");
        if (!conf.containsKey("hdfsspout.badfiles.dir")) {
            LOG.error("hdfsspout.badfiles.dir setting is required");
            throw new RuntimeException("hdfsspout.badfiles.dir setting is required");
        }
        this.badFilesDirPath = new Path(conf.get("hdfsspout.badfiles.dir").toString());
        HdfsSpout.validateOrMakeDir(this.hdfs, this.badFilesDirPath, "bad files");
        if (conf.containsKey("hdfsspout.ignore.suffix")) {
            this.ignoreSuffix = conf.get("hdfsspout.ignore.suffix").toString();
        }
        String lockDir = !conf.containsKey("hdfsspout.lock.dir") ? this.getDefaultLockDir(this.sourceDirPath) : conf.get("hdfsspout.lock.dir").toString();
        this.lockDirPath = new Path(lockDir);
        HdfsSpout.validateOrMakeDir(this.hdfs, this.lockDirPath, "locks");
        if (conf.get("hdfsspout.lock.timeout.sec") != null) {
            this.lockTimeoutSec = Integer.parseInt(conf.get("hdfsspout.lock.timeout.sec").toString());
        }
        if ((ackers = conf.get("topology.acker.executors")) != null) {
            int ackerCount = Integer.parseInt(ackers.toString());
            this.ackEnabled = ackerCount > 0;
            LOG.debug("ACKer count = {}", (Object)ackerCount);
        } else {
            this.ackEnabled = true;
            LOG.debug("ACK count not explicitly set on topology.");
        }
        LOG.info("ACK mode is {}", (Object)(this.ackEnabled ? "enabled" : "disabled"));
        if (conf.get("hdfsspout.commit.count") != null) {
            this.commitFrequencyCount = Integer.parseInt(conf.get("hdfsspout.commit.count").toString());
        }
        if (conf.get("hdfsspout.commit.sec") != null) {
            this.commitFrequencySec = Integer.parseInt(conf.get("hdfsspout.commit.sec").toString());
            if (this.commitFrequencySec <= 0) {
                throw new RuntimeException("hdfsspout.commit.sec setting must be greater than 0");
            }
        }
        if (conf.get("hdfsspout.max.outstanding") != null) {
            this.maxOutstanding = Integer.parseInt(conf.get("hdfsspout.max.outstanding").toString());
        }
        if (conf.get("hdfsspout.clocks.insync") != null) {
            this.clocksInSync = Boolean.parseBoolean(conf.get("hdfsspout.clocks.insync").toString());
        }
        this.spoutId = context.getThisComponentId();
        this.setupCommitElapseTimer();
    }

    private static void validateOrMakeDir(FileSystem fs, Path dir, String dirDescription) {
        try {
            if (fs.exists(dir)) {
                if (!fs.isDirectory(dir)) {
                    LOG.error(dirDescription + " directory is a file, not a dir. " + dir);
                    throw new RuntimeException(dirDescription + " directory is a file, not a dir. " + dir);
                }
            } else if (!fs.mkdirs(dir)) {
                LOG.error("Unable to create " + dirDescription + " directory " + dir);
                throw new RuntimeException("Unable to create " + dirDescription + " directory " + dir);
            }
        }
        catch (IOException e) {
            LOG.error("Unable to create " + dirDescription + " directory " + dir, (Throwable)e);
            throw new RuntimeException("Unable to create " + dirDescription + " directory " + dir, e);
        }
    }

    private String getDefaultLockDir(Path sourceDirPath) {
        return sourceDirPath.toString() + "/" + ".lock";
    }

    private static void checkValidReader(String readerType) {
        if (readerType.equalsIgnoreCase("text") || readerType.equalsIgnoreCase("seq")) {
            return;
        }
        try {
            Class<?> classType = Class.forName(readerType);
            classType.getConstructor(FileSystem.class, Path.class, Map.class);
            return;
        }
        catch (ClassNotFoundException e) {
            LOG.error(readerType + " not found in classpath.", (Throwable)e);
            throw new IllegalArgumentException(readerType + " not found in classpath.", e);
        }
        catch (NoSuchMethodException e) {
            LOG.error(readerType + " is missing the expected constructor for Readers.", (Throwable)e);
            throw new IllegalArgumentException(readerType + " is missing the expected constuctor for Readers.");
        }
    }

    public void ack(Object msgId) {
        LOG.trace("Ack received for msg {} on spout {}", msgId, (Object)this.spoutId);
        if (!this.ackEnabled) {
            return;
        }
        MessageId id = (MessageId)msgId;
        this.inflight.remove(id);
        ++this.acksSinceLastCommit;
        this.tracker.recordAckedOffset(id.offset);
        this.commitProgress(this.tracker.getCommitPosition());
        if (this.fileReadCompletely && this.inflight.isEmpty()) {
            this.markFileAsDone(this.reader.getFilePath());
            this.reader = null;
        }
        super.ack(msgId);
    }

    private boolean canCommitNow() {
        if (this.commitFrequencyCount > 0 && this.acksSinceLastCommit >= this.commitFrequencyCount) {
            return true;
        }
        return this.commitTimeElapsed.get();
    }

    public void fail(Object msgId) {
        LOG.trace("Fail received for msg id {} on spout {}", msgId, (Object)this.spoutId);
        super.fail(msgId);
        if (this.ackEnabled) {
            HdfsUtils.Pair item = HdfsUtils.Pair.of(msgId, this.inflight.remove(msgId));
            this.retryList.add(item);
        }
    }

    private FileReader pickNextFile() {
        try {
            this.lock = this.getOldestExpiredLock();
            if (this.lock != null) {
                LOG.debug("Spout {} now took over ownership of abandoned FileLock {}", (Object)this.spoutId, (Object)this.lock.getLockFile());
                Path file = this.getFileForLockFile(this.lock.getLockFile(), this.sourceDirPath);
                String resumeFromOffset = this.lock.getLastLogEntry().fileOffset;
                LOG.info("Resuming processing of abandoned file : {}", (Object)file);
                return this.createFileReader(file, resumeFromOffset);
            }
            ArrayList<Path> listing = HdfsUtils.listFilesByModificationTime(this.hdfs, this.sourceDirPath, 0L);
            for (Path file : listing) {
                if (file.getName().endsWith(this.inprogress_suffix) || file.getName().endsWith(this.ignoreSuffix)) continue;
                this.lock = FileLock.tryLock(this.hdfs, file, this.lockDirPath, this.spoutId);
                if (this.lock == null) {
                    LOG.debug("Unable to get FileLock for {}, so skipping it.", (Object)file);
                    continue;
                }
                try {
                    Path newFile = this.renameToInProgressFile(file);
                    FileReader result = this.createFileReader(newFile);
                    LOG.info("Processing : {} ", (Object)file);
                    return result;
                }
                catch (Exception e) {
                    LOG.error("Skipping file " + file, (Throwable)e);
                    HdfsSpout.releaseLockAndLog(this.lock, this.spoutId);
                }
            }
            return null;
        }
        catch (IOException e) {
            LOG.error("Unable to select next file for consumption " + this.sourceDirPath, (Throwable)e);
            return null;
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private FileLock getOldestExpiredLock() throws IOException {
        DirLock dirlock = DirLock.tryLock(this.hdfs, this.lockDirPath);
        if (dirlock == null) {
            dirlock = DirLock.takeOwnershipIfStale(this.hdfs, this.lockDirPath, this.lockTimeoutSec);
            if (dirlock == null) {
                LOG.debug("Spout {} could not take over ownership of DirLock for {}", (Object)this.spoutId, (Object)this.lockDirPath);
                return null;
            }
            LOG.debug("Spout {} now took over ownership of abandoned DirLock for {}", (Object)this.spoutId, (Object)this.lockDirPath);
        } else {
            LOG.debug("Spout {} now owns DirLock for {}", (Object)this.spoutId, (Object)this.lockDirPath);
        }
        try {
            if (this.clocksInSync) {
                FileLock fileLock = FileLock.acquireOldestExpiredLock(this.hdfs, this.lockDirPath, this.lockTimeoutSec, this.spoutId);
                return fileLock;
            }
            if (this.lastExpiredLock == null) {
                this.lastExpiredLock = FileLock.locateOldestExpiredLock(this.hdfs, this.lockDirPath, this.lockTimeoutSec);
                this.lastExpiredLockTime = System.currentTimeMillis();
                FileLock fileLock = null;
                return fileLock;
            }
            if (this.hasExpired(this.lastExpiredLockTime)) {
                FileLock fileLock = null;
                return fileLock;
            }
            FileLock.LogEntry lastEntry = FileLock.getLastEntry(this.hdfs, this.lastExpiredLock.getKey());
            if (lastEntry.equals(this.lastExpiredLock.getValue())) {
                FileLock result = FileLock.takeOwnership(this.hdfs, this.lastExpiredLock.getKey(), lastEntry, this.spoutId);
                this.lastExpiredLock = null;
                FileLock fileLock = result;
                return fileLock;
            }
            this.lastExpiredLock = null;
            FileLock fileLock = null;
            return fileLock;
        }
        finally {
            dirlock.release();
            LOG.debug("Released DirLock {}, SpoutID {} ", (Object)dirlock.getLockFile(), (Object)this.spoutId);
        }
    }

    private boolean hasExpired(long lastModifyTime) {
        return System.currentTimeMillis() - lastModifyTime < (long)(this.lockTimeoutSec * 1000);
    }

    private FileReader createFileReader(Path file) throws IOException {
        if (this.readerType.equalsIgnoreCase("seq")) {
            return new SequenceFileReader(this.hdfs, file, this.conf);
        }
        if (this.readerType.equalsIgnoreCase("text")) {
            return new TextFileReader(this.hdfs, file, this.conf);
        }
        try {
            Class<?> clsType = Class.forName(this.readerType);
            Constructor<?> constructor = clsType.getConstructor(FileSystem.class, Path.class, Map.class);
            return (FileReader)constructor.newInstance(this.hdfs, file, this.conf);
        }
        catch (Exception e) {
            LOG.error(e.getMessage(), (Throwable)e);
            throw new RuntimeException("Unable to instantiate " + this.readerType + " reader", e);
        }
    }

    private FileReader createFileReader(Path file, String offset) throws IOException {
        if (this.readerType.equalsIgnoreCase("seq")) {
            return new SequenceFileReader(this.hdfs, file, this.conf, offset);
        }
        if (this.readerType.equalsIgnoreCase("text")) {
            return new TextFileReader(this.hdfs, file, this.conf, offset);
        }
        try {
            Class<?> clsType = Class.forName(this.readerType);
            Constructor<?> constructor = clsType.getConstructor(FileSystem.class, Path.class, Map.class, String.class);
            return (FileReader)constructor.newInstance(this.hdfs, file, this.conf, offset);
        }
        catch (Exception e) {
            LOG.error(e.getMessage(), (Throwable)e);
            throw new RuntimeException("Unable to instantiate " + this.readerType, e);
        }
    }

    private Path renameToInProgressFile(Path file) throws IOException {
        Path newFile = new Path(file.toString() + this.inprogress_suffix);
        try {
            if (this.hdfs.rename(file, newFile)) {
                return newFile;
            }
            throw new RenameException(file, newFile);
        }
        catch (IOException e) {
            throw new RenameException(file, newFile, e);
        }
    }

    private Path getFileForLockFile(Path lockFile, Path sourceDirPath) throws IOException {
        String lockFileName = lockFile.getName();
        Path dataFile = new Path(sourceDirPath + "/" + lockFileName + this.inprogress_suffix);
        if (this.hdfs.exists(dataFile)) {
            return dataFile;
        }
        dataFile = new Path(sourceDirPath + "/" + lockFileName);
        if (this.hdfs.exists(dataFile)) {
            return dataFile;
        }
        return null;
    }

    private Path renameCompletedFile(Path file) throws IOException {
        String fileName = file.toString();
        String fileNameMinusSuffix = fileName.substring(0, fileName.indexOf(this.inprogress_suffix));
        String newName = new Path(fileNameMinusSuffix).getName();
        Path newFile = new Path(this.archiveDirPath + "/" + newName);
        LOG.info("Completed consuming file {}", (Object)fileNameMinusSuffix);
        if (!this.hdfs.rename(file, newFile)) {
            throw new IOException("Rename failed for file: " + file);
        }
        LOG.debug("Renamed file {} to {} ", (Object)file, (Object)newFile);
        return newFile;
    }

    public void declareOutputFields(OutputFieldsDeclarer declarer) {
        declarer.declare(this.outputFields);
    }

    private static class RenameException
    extends IOException {
        public final Path oldFile;
        public final Path newFile;

        public RenameException(Path oldFile, Path newFile) {
            super("Rename of " + oldFile + " to " + newFile + " failed");
            this.oldFile = oldFile;
            this.newFile = newFile;
        }

        public RenameException(Path oldFile, Path newFile, IOException cause) {
            super("Rename of " + oldFile + " to " + newFile + " failed", cause);
            this.oldFile = oldFile;
            this.newFile = newFile;
        }
    }

    static class MessageId
    implements Comparable<MessageId> {
        public long msgNumber;
        public String fullPath;
        public FileOffset offset;

        public MessageId(long msgNumber, Path fullPath, FileOffset offset) {
            this.msgNumber = msgNumber;
            this.fullPath = fullPath.toString();
            this.offset = offset;
        }

        public String toString() {
            return "{'" + this.fullPath + "':" + this.offset + "}";
        }

        @Override
        public int compareTo(MessageId rhs) {
            if (this.msgNumber < rhs.msgNumber) {
                return -1;
            }
            if (this.msgNumber > rhs.msgNumber) {
                return 1;
            }
            return 0;
        }
    }
}

