/*
 * Decompiled with CFR 0.152.
 */
package org.apache.storm.jdbc.common;

import com.google.common.base.Function;
import com.google.common.base.Joiner;
import com.google.common.collect.Collections2;
import com.google.common.collect.Lists;
import java.sql.Connection;
import java.sql.DatabaseMetaData;
import java.sql.Date;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.ResultSetMetaData;
import java.sql.SQLException;
import java.sql.Statement;
import java.sql.Time;
import java.sql.Timestamp;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.List;
import org.apache.commons.lang.StringUtils;
import org.apache.storm.jdbc.common.Column;
import org.apache.storm.jdbc.common.ConnectionProvider;
import org.apache.storm.jdbc.common.Util;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class JdbcClient {
    private static final Logger LOG = LoggerFactory.getLogger(JdbcClient.class);
    private ConnectionProvider connectionProvider;
    private int queryTimeoutSecs;

    public JdbcClient(ConnectionProvider connectionProvider, int queryTimeoutSecs) {
        this.connectionProvider = connectionProvider;
        this.queryTimeoutSecs = queryTimeoutSecs;
    }

    public void insert(String tableName, List<List<Column>> columnLists) {
        String query = this.constructInsertQuery(tableName, columnLists);
        this.executeInsertQuery(query, columnLists);
    }

    public void executeInsertQuery(String query, List<List<Column>> columnLists) {
        Connection connection = null;
        try {
            connection = this.connectionProvider.getConnection();
            boolean autoCommit = connection.getAutoCommit();
            if (autoCommit) {
                connection.setAutoCommit(false);
            }
            LOG.debug("Executing query {}", (Object)query);
            PreparedStatement preparedStatement = connection.prepareStatement(query);
            if (this.queryTimeoutSecs > 0) {
                preparedStatement.setQueryTimeout(this.queryTimeoutSecs);
            }
            for (List<Column> columnList : columnLists) {
                this.setPreparedStatementParams(preparedStatement, columnList);
                preparedStatement.addBatch();
            }
            int[] results = preparedStatement.executeBatch();
            if (Arrays.asList(new int[][]{results}).contains(-3)) {
                connection.rollback();
                throw new RuntimeException("failed at least one sql statement in the batch, operation rolled back.");
            }
            try {
                connection.commit();
            }
            catch (SQLException e) {
                throw new RuntimeException("Failed to commit insert query " + query, e);
            }
        }
        catch (SQLException e) {
            throw new RuntimeException("Failed to execute insert query " + query, e);
        }
        finally {
            this.closeConnection(connection);
        }
    }

    private String constructInsertQuery(String tableName, List<List<Column>> columnLists) {
        StringBuilder sb = new StringBuilder();
        sb.append("Insert into ").append(tableName).append(" (");
        Collection columnNames = Collections2.transform((Collection)columnLists.get(0), (Function)new Function<Column, String>(){

            public String apply(Column input) {
                return input.getColumnName();
            }
        });
        String columns = Joiner.on((String)",").join((Iterable)columnNames);
        sb.append(columns).append(") values ( ");
        String placeHolders = StringUtils.chop((String)StringUtils.repeat((String)"?,", (int)columnNames.size()));
        sb.append(placeHolders).append(")");
        return sb.toString();
    }

    public List<List<Column>> select(String sqlQuery, List<Column> queryParams) {
        Connection connection = null;
        try {
            connection = this.connectionProvider.getConnection();
            PreparedStatement preparedStatement = connection.prepareStatement(sqlQuery);
            if (this.queryTimeoutSecs > 0) {
                preparedStatement.setQueryTimeout(this.queryTimeoutSecs);
            }
            this.setPreparedStatementParams(preparedStatement, queryParams);
            ResultSet resultSet = preparedStatement.executeQuery();
            ArrayList rows = Lists.newArrayList();
            while (resultSet.next()) {
                ResultSetMetaData metaData = resultSet.getMetaData();
                int columnCount = metaData.getColumnCount();
                ArrayList row = Lists.newArrayList();
                for (int i = 1; i <= columnCount; ++i) {
                    String columnLabel = metaData.getColumnLabel(i);
                    int columnType = metaData.getColumnType(i);
                    Class columnJavaType = Util.getJavaType(columnType);
                    if (columnJavaType.equals(String.class)) {
                        row.add(new Column<String>(columnLabel, resultSet.getString(columnLabel), columnType));
                        continue;
                    }
                    if (columnJavaType.equals(Integer.class)) {
                        row.add(new Column<Integer>(columnLabel, resultSet.getInt(columnLabel), columnType));
                        continue;
                    }
                    if (columnJavaType.equals(Double.class)) {
                        row.add(new Column<Double>(columnLabel, resultSet.getDouble(columnLabel), columnType));
                        continue;
                    }
                    if (columnJavaType.equals(Float.class)) {
                        row.add(new Column<Float>(columnLabel, Float.valueOf(resultSet.getFloat(columnLabel)), columnType));
                        continue;
                    }
                    if (columnJavaType.equals(Short.class)) {
                        row.add(new Column<Short>(columnLabel, resultSet.getShort(columnLabel), columnType));
                        continue;
                    }
                    if (columnJavaType.equals(Boolean.class)) {
                        row.add(new Column<Boolean>(columnLabel, resultSet.getBoolean(columnLabel), columnType));
                        continue;
                    }
                    if (columnJavaType.equals(byte[].class)) {
                        row.add(new Column<byte[]>(columnLabel, resultSet.getBytes(columnLabel), columnType));
                        continue;
                    }
                    if (columnJavaType.equals(Long.class)) {
                        row.add(new Column<Long>(columnLabel, resultSet.getLong(columnLabel), columnType));
                        continue;
                    }
                    if (columnJavaType.equals(Date.class)) {
                        row.add(new Column<Date>(columnLabel, resultSet.getDate(columnLabel), columnType));
                        continue;
                    }
                    if (columnJavaType.equals(Time.class)) {
                        row.add(new Column<Time>(columnLabel, resultSet.getTime(columnLabel), columnType));
                        continue;
                    }
                    if (columnJavaType.equals(Timestamp.class)) {
                        row.add(new Column<Timestamp>(columnLabel, resultSet.getTimestamp(columnLabel), columnType));
                        continue;
                    }
                    throw new RuntimeException("type =  " + columnType + " for column " + columnLabel + " not supported.");
                }
                rows.add(row);
            }
            ArrayList arrayList = rows;
            return arrayList;
        }
        catch (SQLException e) {
            throw new RuntimeException("Failed to execute select query " + sqlQuery, e);
        }
        finally {
            this.closeConnection(connection);
        }
    }

    public List<Column> getColumnSchema(String tableName) {
        Connection connection = null;
        ArrayList<Column> columns = new ArrayList<Column>();
        try {
            connection = this.connectionProvider.getConnection();
            DatabaseMetaData metaData = connection.getMetaData();
            ResultSet resultSet = metaData.getColumns(null, null, tableName, null);
            while (resultSet.next()) {
                columns.add(new Column(resultSet.getString("COLUMN_NAME"), resultSet.getInt("DATA_TYPE")));
            }
            ArrayList<Column> arrayList = columns;
            return arrayList;
        }
        catch (SQLException e) {
            throw new RuntimeException("Failed to get schema for table " + tableName, e);
        }
        finally {
            this.closeConnection(connection);
        }
    }

    public void executeSql(String sql) {
        Connection connection = null;
        try {
            connection = this.connectionProvider.getConnection();
            Statement statement = connection.createStatement();
            statement.execute(sql);
        }
        catch (SQLException e) {
            throw new RuntimeException("Failed to execute SQL", e);
        }
        finally {
            this.closeConnection(connection);
        }
    }

    private void setPreparedStatementParams(PreparedStatement preparedStatement, List<Column> columnList) throws SQLException {
        int index = 1;
        for (Column column : columnList) {
            Class columnJavaType = Util.getJavaType(column.getSqlType());
            if (column.getVal() == null) {
                preparedStatement.setNull(index, column.getSqlType());
            } else if (columnJavaType.equals(String.class)) {
                preparedStatement.setString(index, (String)column.getVal());
            } else if (columnJavaType.equals(Integer.class)) {
                preparedStatement.setInt(index, (Integer)column.getVal());
            } else if (columnJavaType.equals(Double.class)) {
                preparedStatement.setDouble(index, (Double)column.getVal());
            } else if (columnJavaType.equals(Float.class)) {
                preparedStatement.setFloat(index, ((Float)column.getVal()).floatValue());
            } else if (columnJavaType.equals(Short.class)) {
                preparedStatement.setShort(index, (Short)column.getVal());
            } else if (columnJavaType.equals(Boolean.class)) {
                preparedStatement.setBoolean(index, (Boolean)column.getVal());
            } else if (columnJavaType.equals(byte[].class)) {
                preparedStatement.setBytes(index, (byte[])column.getVal());
            } else if (columnJavaType.equals(Long.class)) {
                preparedStatement.setLong(index, (Long)column.getVal());
            } else if (columnJavaType.equals(Date.class)) {
                preparedStatement.setDate(index, (Date)column.getVal());
            } else if (columnJavaType.equals(Time.class)) {
                preparedStatement.setTime(index, (Time)column.getVal());
            } else if (columnJavaType.equals(Timestamp.class)) {
                preparedStatement.setTimestamp(index, (Timestamp)column.getVal());
            } else {
                throw new RuntimeException("Unknown type of value " + column.getVal() + " for column " + column.getColumnName());
            }
            ++index;
        }
    }

    private void closeConnection(Connection connection) {
        if (connection != null) {
            try {
                connection.close();
            }
            catch (SQLException e) {
                throw new RuntimeException("Failed to close connection", e);
            }
        }
    }
}

