-- Licensed to the Apache Software Foundation (ASF) under one or more
-- contributor license agreements. See the NOTICE file distributed with
-- this work for additional information regarding copyright ownership.


-- Create a temporary database on PostgreSQL for testing geometries and rasters.
-- The "postgis_raster" extension must be installed before to execute this test.

SET search_path TO public;

CREATE TABLE features."SpatialData" (
    "filename" VARCHAR(20) NOT NULL,
    "geometry" GEOMETRY,
    "image"    RASTER,

    CONSTRAINT "PK_SpatialData" PRIMARY KEY ("filename")
);


--
-- Rasters with dummy sample values. Those data are duplicated in two ways:
-- The same rasters are created in Java code by the `TestRaster` class, and
-- the same bytes are stored in test files with ".wkb" extension.
--
INSERT INTO features."SpatialData" ("filename", "image") VALUES
  ('raster-ushort.wkb', ('0100000200000000000000F43F000000000000044000000000000054C00000000000004EC0000000'
   || '00000000000000000000000000E6100000030004000600006F0079008300D300DD00E700370141014B019B01A501AF01060'
   || '00070007A008400D400DE00E800380142014C019C01A601B001')::raster);


--
-- Geometries with arbitrary coordinate values.
--
INSERT INTO features."SpatialData" ("filename", "geometry") VALUES
  ('point-prj',   ST_GeomFromText('POINT(2 3)', 3395)),
  ('linestring',  ST_GeomFromText('LINESTRING(-71.160281 42.258729,-71.160837 42.259113,-71.161144 42.25932)', 4326)),
  ('polygon-prj', ST_GeomFromText('POLYGON((0 0,0 1,1 1,1 0,0 0))', 3395)),
  ('polygon',     ST_GeomFromText('POLYGON'
   || '((-71.1776585052917 42.3902909739571,-71.1776820268866 42.3903701743239,-71.1776063012595 42.3903825660754,'
   ||   '-71.1775826583081 42.3903033653531,-71.1776585052917 42.3902909739571))', 4326)),
  ('multi-linestring', ST_GeomFromText('MULTILINESTRING'
   || '((-71.160281 42.258729,-71.160837 42.259113,-71.161144 42.25932),'
   ||  '(-71.1031627617667 42.3152960829043,-71.102923838298 42.3149156848307))', 4326)),
  ('multi-polygon', ST_GeomFromText('MULTIPOLYGON('
   || '((-71.1031880899493 42.3152774590236,-71.1031627617667 42.3152960829043,-71.102923838298 42.3149156848307,'
   ||   '-71.1023097974109 42.3151969047397,-71.1019285062273 42.3147384934248,-71.102505233663 42.3144722937587,'
   ||   '-71.1027748747100 42.3141658254797,-71.1031139451630 42.3142739188902,-71.103248764160 42.3140248998700,'
   ||   '-71.1033002961013 42.3140393340215,-71.1033488797549 42.3139495090772,-71.103396240451 42.3138632439557,'
   ||   '-71.1041521907712 42.3141153348029,-71.1041411411543 42.3141545014533,-71.104128779591 42.3142114839058,'
   ||   '-71.1041188134329 42.3142693656241,-71.1041112482575 42.3143272556118,-71.104107284573 42.3143851580048,'
   ||   '-71.1041057218871 42.3144430686681,-71.1041065602059 42.3145009876017,-71.104109799536 42.3145589148055,'
   ||   '-71.1041166403905 42.3146168544148,-71.1041258822717 42.3146748022936,-71.104137530758 42.3147318674446,'
   ||   '-71.1041492906949 42.3147711126569,-71.1041598612795 42.3148085717390,-71.104251501387 42.3151287620809,'
   ||   '-71.1041173835118 42.3150739481917,-71.1040809891419 42.3151344119048,-71.104043867891 42.3151191367447,'
   ||   '-71.1040194562988 42.3151832057859,-71.1038734225584 42.3151140942995,-71.103844693824 42.3151006300338,'
   ||   '-71.1038315271889 42.3150943475350,-71.1037393329282 42.3150548249850,-71.103544755557 42.3152608696313,'
   ||   '-71.1033436658644 42.3151648370544,-71.1032580383161 42.3152269126061,-71.103223066939 42.3152517403219,'
   ||   '-71.1031880899493 42.3152774590236)),'
   || '((-71.1043632495873 42.3151131085460,-71.1043583974082 42.3151211109857,-71.1043443253471 42.315067601583,'
   ||   '-71.1043850704575 42.3150793250568,-71.1043632495873 42.315113108546)))', 4326));


--
-- Geometries with WKT representation in one column and WKB in another column.
-- Used for parsing the same geometry in two ways and comparing the results.
--
CREATE VIEW features."Geometries" AS
  SELECT ST_AsText(geometry) AS "WKT", geometry AS "WKB", ST_SRID(geometry) AS "SRID"
  FROM   features."SpatialData"
  WHERE  geometry IS NOT NULL;
