/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.sis.openoffice;

import com.sun.star.uno.XInterface;
import com.sun.star.lang.IllegalArgumentException;


/**
 * Services from the {@link org.apache.sis.referencing} package to be exported to Apache
 * <a href="http://www.openoffice.org">OpenOffice</a> or LibreOffice.
 *
 * <p>This interface is derived from the {@code XReferencing.idl} file by first compiling with the {@code javamaker}
 * tool provided in OpenOffice SDK, then disassembling the output using the {@code javap} tool provided in Java SDK.
 * This source file exists only for convenience: for avoiding that IDEs report errors, and opportunistically for
 * Javadoc purpose. But the {@code XReferencing.class} file compiled from this source file <strong>must</strong>
 * be overwritten by the {@code XReferencing.class} file generated by {@code javamaker}.
 * See {@code application/sis-openoffice/src/main/unopkg/build-instruction.html} for more information.</p>
 *
 * @author  Martin Desruisseaux (IRD, Geomatys)
 * @version 0.8
 * @since   0.8
 * @module
 */
public interface XReferencing extends XInterface {
    /**
     * Returns the identified object name from an authority code.
     *
     * @param  codeOrPath  the code allocated by an authority, or the path to a file.
     * @return the object name.
     */
    String getName(String codeOrPath);

    /**
     * Returns the scope of an identified object.
     *
     * @param  codeOrPath  the code allocated by an authority, or the path to a file.
     * @return the object scope.
     */
    String getScope(String codeOrPath);

    /**
     * Returns the domain of validity as country or region names for an identified object.
     *
     * @param  codeOrPath  the code allocated by an authority, or the path to a file.
     * @return the domain of validity.
     */
    String getDomainOfValidity(String codeOrPath);

    /**
     * Returns the domain of validity as a geographic bounding box for an identified object.
     * This method returns a 2×2 matrix:
     * the first row contains the latitude and longitude of upper left corner,
     * and the second row contains the latitude and longitude of bottom right corner.
     * Units are degrees.
     *
     * @param  codeOrPath  the code allocated by an authority, or the path to a file.
     * @return the object bounding box.
     */
    double[][] getGeographicArea(String codeOrPath);

    /**
     * Returns the accuracy of a transformation between two coordinate reference systems.
     *
     * @param  sourceCRS       the authority code for the source coordinate reference system.
     * @param  targetCRS       the authority code for the target coordinate reference system.
     * @param  areaOfInterest  an optional bounding box of source coordinates to transform.
     * @return the operation accuracy.
     * @throws IllegalArgumentException if {@code points} is not a {@code double[][]} value or void.
     */
    double getAccuracy(String sourceCRS, String targetCRS, Object areaOfInterest) throws IllegalArgumentException;

    /**
     * Returns the axis name and units for the specified dimension in a coordinate reference system or coordinate system.
     * This method returns a short axis name as used in Well Known Text (WKT) format, for example <cite>"Latitude"</cite>
     * instead of <cite>"Geodetic latitude"</cite>.
     *
     * @param  codeOrPath  the code allocated by an authority, or the path to a file.
     * @param  dimension   the dimension (1, 2, …).
     * @return the name of the requested axis.
     */
    String getAxis(String codeOrPath, int dimension);

    /**
     * Transforms coordinates from the specified source CRS to the specified target CRS.
     *
     * @param  sourceCRS  the authority code for the source coordinate reference system.
     * @param  targetCRS  the authority code for the target coordinate reference system.
     * @param  points     the coordinates to transform.
     * @return the transformed coordinates.
     */
    double[][] transformPoints(String sourceCRS, String targetCRS, double[][] points);

    /**
     * Transforms an envelope from the specified source CRS to the specified target CRS.
     *
     * @param  sourceCRS  the authority code for the source coordinate reference system.
     * @param  targetCRS  the authority code for the target coordinate reference system.
     * @param  envelope   points inside the envelope to transform.
     * @return the transformed envelope.
     */
    double[][] transformEnvelope(String sourceCRS, String targetCRS, double[][] envelope);

    /**
     * Converts text in degrees-minutes-seconds to an angle in decimal degrees.
     * See {@link org.apache.sis.measure.AngleFormat} for pattern description.
     *
     * @param  text     the text to be converted to an angle.
     * @param  pattern  an optional text that describes the format (example: "D°MM.m'").
     * @param  locale   the convention to use (e.g. decimal separator symbol).
     * @return the angle parsed as a number.
     * @throws IllegalArgumentException if {@code pattern} is not a string value or void.
     */
    double[][] parseAngle(String[][] text, Object pattern, Object locale) throws IllegalArgumentException;

    /**
     * Converts an angle to text according to a given format. This method uses the pattern
     * described by {@link org.apache.sis.measure.AngleFormat} with the following extension:
     *
     * <ul>
     *   <li>If the pattern ends with E or W, then the angle is formatted as a longitude.</li>
     *   <li>If the pattern ends with N or S, then the angle is formatted as a latitude.</li>
     * </ul>
     *
     * @param  value    the angle value (in decimal degrees) to be converted.
     * @param  pattern  an optional text that describes the format (example: "D°MM.m'").
     * @param  locale   the convention to use (e.g. decimal separator symbol).
     * @return the angle formatted as a string.
     * @throws IllegalArgumentException if {@code pattern} is not a string value or void.
     */
    String[][] formatAngle(double[][] value, Object pattern, Object locale) throws IllegalArgumentException;
}
