/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.sis.internal.unopkg;

import java.io.File;
import java.io.IOException;

import org.apache.maven.plugin.AbstractMojo;
import org.apache.maven.plugin.MojoExecutionException;
import org.apache.maven.plugins.annotations.LifecyclePhase;
import org.apache.maven.plugins.annotations.Mojo;
import org.apache.maven.plugins.annotations.Parameter;
import org.codehaus.plexus.util.FileUtils;


/**
 * Compiles Java interfaces from OpenOffice IDL files.
 *
 * <p>In an ideal world, this plugin would execute {@code idlc} on the {@code *.idl} files,
 * then {@code regmerge} on the generated {@code *.urd} files,
 * then {@code javamaker} on the generated {@code *.rdb} files.
 * However, since the above mentioned tools are native and would require a manual installation
 * on every developer machine, current version just copies a pre-compiled class file.
 * This copy must occurs after the compilation phase (in order to overwrite the files generated
 * by {@code javac}), which is why the usual Maven resources mechanism doesn't fit.
 *
 * @author  Martin Desruisseaux (IRD, Geomatys)
 * @version 0.8
 * @since   0.8
 * @module
 */
@Mojo(name = "javamaker", defaultPhase = LifecyclePhase.PROCESS_CLASSES)
public final class JavaMaker extends AbstractMojo {
    /**
     * Base directory of the module to compile.
     */
    @Parameter(property="basedir", required=true, readonly=true)
    private String baseDirectory;

    /**
     * Directory where the output Java files will be located.
     */
    @Parameter(property="project.build.outputDirectory", required=true, readonly=true)
    private String outputDirectory;

    /**
     * Invoked by reflection for creating the MOJO.
     */
    public JavaMaker() {
    }

    /**
     * Copies the {@code .class} files generated by OpenOffice.
     *
     * @throws MojoExecutionException if the plugin execution failed.
     */
    @Override
    public void execute() throws MojoExecutionException {
        final int n;
        try {
            n = copyClasses(new File(baseDirectory, UnoPkg.SOURCE_DIRECTORY), new File(outputDirectory));
        } catch (IOException e) {
            throw new MojoExecutionException("Failed to copy *.class files.", e);
        }
        getLog().info("[geotk-unopkg] Copied " + n + " pre-compiled class files.");
    }

    /**
     * Copies {@code *.class} files from source directory to output directory.
     * The output directory shall already exists. It should be the case if all
     * sources files have been compiled before this method is invoked.
     *
     * @return the number of files copied.
     */
    private static int copyClasses(final File sourceDirectory,
                                   final File outputDirectory) throws IOException
    {
        int n = 0;
        final String[] filenames = sourceDirectory.list();
        for (final String filename : filenames) {
            final File file = new File(sourceDirectory, filename);
            if (file.isFile()) {
                if (filename.endsWith(".class") || filename.endsWith(".CLASS")) {
                    FileUtils.copyFileToDirectory(file, outputDirectory);
                    n++;
                }
            } else if (file.isDirectory()) {
                n += copyClasses(file, new File(outputDirectory, filename));
            }
        }
        return n;
    }
}
