/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership. The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License. You may obtain a copy of the License at
 * 
 *      http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package net.jini.event;

import java.rmi.RemoteException;

import net.jini.core.event.RemoteEventListener;
import net.jini.core.lease.Lease;


/**
 * The <code>MailboxRegistration</code> defines the interface through which
 * a client manages its registration and its notification processing. 
 * Event mailbox clients use this interface to:
 * <UL>
 * <LI>Manage the lease for this registration
 * <LI>Obtain a <code>RemoteEventListener</code> reference that
 *     can be registered with event generators. This listener will
 *     then store any received events for this registration.
 * <LI>Enable or disable the delivery of any stored events for this
 *     registration.
 * </UL>
 *
 * @author Sun Microsystems, Inc.
 *
 * @since 1.1
 */
public interface MailboxRegistration {

    /**
     * Returns the <code>Lease</code> object associated
     * with this registration. The client can renew or 
     * cancel the registration with the mailbox service
     * through this lease object.
     *
     * @return The lease object associated with this registration
     */
    public Lease getLease();

    /**
     * Returns the <code>RemoteEventListener</code> associated
     * with this registration. This listener can then be submitted
     * as the <code>RemoteEventListener</code> argument to an
     * event generator's registration method(s).
     *
     * @return The <code>RemoteEventListener</code> associated
     *         with this registration.
     */
    public RemoteEventListener getListener();

    /**
     * Initiates delivery of stored notifications to the supplied
     * <code>target</code> listener, if any. If a target listener
     * already exists, then it will be replaced with the specified
     * target listener. Passing <code>null</code> as the
     * <code>target</code> parameter has the same effect as calling
     * the <code>disableDelivery</code> method.
     *
     * @param target The listener to be notified of stored events, if any.
     *
     * @throws IllegalArgumentException if the supplied <code>target</code> 
     *         parameter is a listener object that was generated by the
     *         mailbox service itself.
     *
     * @throws java.rmi.RemoteException if there is
     *  a communication failure between the client and the service.
     */
    public void enableDelivery(RemoteEventListener target)
		    throws RemoteException;

    /**
     * Ceases delivery of stored notifications to the existing
     * target listener, if any. It is acceptable to call this
     * method even if no target listener is currently enabled.
     *
     * @throws java.rmi.RemoteException if there is
     *  a communication failure between the client and the service.
     */
    public void disableDelivery() throws RemoteException;
}
