/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership. The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License. You may obtain a copy of the License at
 * 
 *      http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.river.test.spec.renewalservice;

import java.util.logging.Level;

// net.jini
import net.jini.core.event.EventRegistration;
import net.jini.core.lease.Lease;
import net.jini.lease.LeaseRenewalManager;
import net.jini.lease.LeaseRenewalService;
import net.jini.lease.LeaseRenewalSet;

// 
import org.apache.river.qa.harness.TestException;

// org.apache.river.qa
import org.apache.river.qa.harness.QATestEnvironment;
import org.apache.river.qa.harness.Test;
import org.apache.river.test.share.RememberingRemoteListener;

import net.jini.export.Exporter;
import net.jini.config.ConfigurationException;



/**
 * Assert the following:
 *<OL>
 *<LI>An event registration and the set from which it is generated share
 *    the same lease</LI>
 *<LI>An event registration ID is unique with respect to all other active
 *    registrations generated by the same LRS. (weak test because only
 *    40 cases are tested. It would be impractical to test all cases
 *    exhaustively.)</LI>
 *<LI>Assert that when a registration is replaced with a new one that
 *    the same event ID is used.</LI>

 */
public class EventRegistrationTest extends AbstractLeaseRenewalServiceTest {
    
    /**
     * the number of renewal sets used in this test
     */
    public static final int NUMBER_OF_RENEWAL_SETS = 20;

    /**
     * an array to hold references to the renewal sets 
     */
    LeaseRenewalSet[] renewalSet = new LeaseRenewalSet[NUMBER_OF_RENEWAL_SETS];

    /**
     * an array to hold references to the remote listeners
     */
    RememberingRemoteListener[] listener = 
	new RememberingRemoteListener[NUMBER_OF_RENEWAL_SETS];

    /**
     * an array to hold references to the eventRegistrations 
     */
    EventRegistration[] failReg = 
	new EventRegistration[NUMBER_OF_RENEWAL_SETS];
    EventRegistration[] warnReg = 
	new EventRegistration[NUMBER_OF_RENEWAL_SETS];

    /**
     *  The LeaseRenewalManager used for LRS impls that grant only short leases
     */
    private LeaseRenewalManager lrm = null;

    /**
     * Sets up the testing environment.
     */
    public Test construct(org.apache.river.qa.harness.QAConfig sysConfig) throws Exception {

       // mandatory call to parent
       super.construct(sysConfig);
	
       // Announce where we are in the test
       logger.log(Level.FINE, "EventRegistrationTest: In setup() method.");

       // create lease renewal manager for wider use across implementations
       lrm = new LeaseRenewalManager(sysConfig.getConfiguration());
       return this;
    }


    /**
     * Assert the following:
     *<OL>
     *<LI>An event registration and the set from which it is generated share
     *    the same lease</LI>
     *<LI>An event registration ID is unique with respect to all other active
     *    registrations generated by the same LRS. (weak test because only
     *    40 cases are tested. It would be impractical to test all cases
     *    exhaustively.)</LI>
     *<LI>Assert that when a registration is replaced with a new one that
     *    the same event ID is used.</LI>
     */
    public void run() throws Exception {

	// Announce where we are in the test
	logger.log(Level.FINE, "EventRegistrationTest: In run() method.");

	// grab the ever popular LRS
	LeaseRenewalService lrs = getLRS();

	// create all required renewal sets
	long renewSetDur = Lease.FOREVER;
	for (int i = 0; i < NUMBER_OF_RENEWAL_SETS; ++i) {
	    logger.log(Level.FINE, "Creating renewal set #" + i + 
			      " with lease duration of Lease.FOREVER.");
	    renewalSet[i] = prepareSet(lrs.createLeaseRenewalSet(renewSetDur));
	    lrm.renewFor(prepareLease(renewalSet[i].getRenewalSetLease()), renewSetDur,
			 null);
	}

	// create all required RemoteListeners
	for (int i = 0; i < NUMBER_OF_RENEWAL_SETS; ++i) {
	    logger.log(Level.FINE, "Creating remote listener #" + i);
	    listener[i] = new RememberingRemoteListener(getExporter());
	}

	// register first listener for ExpirationWarningEvents
	EventRegistration evReg = 
	    renewalSet[0].setExpirationWarningListener(listener[0], 1, 
						       null);
	evReg = prepareRegistration(evReg);
	
	/* ASSERTION #1
	   The set lease and the registration lease are the same. */
	Lease lease01 = prepareLease(renewalSet[0].getRenewalSetLease());
	Lease lease02 = evReg.getLease();
	if (lease01.equals(lease02) == false) {
	    String message = "The renewal set lease and the warning " +
			     "event registration lease are not the same.";
	    throw new TestException(message);
	}

	// register for RenewalFailureEvents
	evReg = renewalSet[0].setRenewalFailureListener(listener[0], null);
	evReg = prepareRegistration(evReg);
	
	/* ASSERTION #1 cont'd
	   The set lease and the registration lease are the same. */
	Lease lease03 = evReg.getLease();
	if (lease01.equals(lease03) == false) {
	    String message = "The renewal set lease and the failure" +
		"event registration lease are not the same.";
	    throw new TestException(message);
	}

	logger.log(Level.FINE, "Assertion #1 passed.");

	// register all twenty remote listeners
	for (int i = 0; i < NUMBER_OF_RENEWAL_SETS; ++i) {
	    evReg = 
		renewalSet[i].setExpirationWarningListener(listener[i], 1, 
							   null);
	    evReg = prepareRegistration(evReg);
	    warnReg[i] = evReg;
	    evReg = 
		renewalSet[i].setRenewalFailureListener(listener[i], null);
	    evReg = prepareRegistration(evReg);
	    failReg[i] = evReg;
	}

	/* ASSERTION #2
	   The registrations are unique and event ids are valid across
	   all active registrations. */
	for (int i = 0; i < NUMBER_OF_RENEWAL_SETS; ++i) {

	    if (rstUtil.isValidExpWarnEventReg(warnReg[i], 
				       renewalSet[i]) == false) {
		String message = "Warning event registration # " + i +
		    " is invalid." + rstUtil.getFailureReason();
		throw new TestException(message);
	    }

	    if (rstUtil.isValidRenewFailEventReg(failReg[i], 
					 renewalSet[i]) == false) {
		String message = "Failure event registration # " + i +
		    " is invalid." + rstUtil.getFailureReason();
		throw new TestException(message);
	    }

	    // create yet another RemoteEventListener
	    Exporter exp = null;
	    RememberingRemoteListener newListener = 
		new RememberingRemoteListener(getExporter());
	    evReg = 
		renewalSet[0].setExpirationWarningListener(newListener,
							   1, null);
	    evReg = prepareRegistration(evReg);

	    /* ASSERTION #3
	       When a listener is replaced the event registration has
	       the same event id */
	    if (! (evReg.getID() == warnReg[0].getID())) {
		String message = "When a warning registration is " +
		    "replaced, the event id is not preserved.";
		throw new TestException(message);
	    }
	    
	    // and just for grins check the lease as well
	    if (evReg.getLease().equals(warnReg[0].getLease()) == false) {
		String message = "When a warning registration is " +
		    "replaced, the lease is not preserved.";
		throw new TestException(message);
	    }
	    
	    /* ASSERTION #3 (cont'd for renewal failure events)
	       When a listener is replaced the event registration has
	       the same event id */
	    evReg = 
		renewalSet[0].setRenewalFailureListener(newListener,
							null);
	    evReg = prepareRegistration(evReg);
	    if (! (evReg.getID() == failReg[0].getID())) {
		String message = "When a failure registration is " +
		    "replaced, the event id is not preserved.";
		throw new TestException(message);
	    }
	    
	    // and just for grins check the lease as well
	    if (evReg.getLease().equals(failReg[0].getLease()) == false) {
		String message = "When a failure registration is " +
		    "replaced, the lease is not preserved.";
		throw new TestException(message);
	    }

	}

	logger.log(Level.FINE, "Assertion #2 passed.");

	logger.log(Level.FINE, "Assertion #3 passed.");
    }
} // EventRegistrationTest
