/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership. The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License. You may obtain a copy of the License at
 * 
 *      http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.river.test.spec.joinmanager;

import java.util.logging.Level;

import org.apache.river.qa.harness.TestException;
import org.apache.river.test.share.LookupServices;

/**
 * This class verifies that the <code>JoinManager</code> utility class
 * operates in a manner consistent with the specification. In particular,
 * this class verifies that when <code>modifyAttributes</code> is invoked
 * to change each element of the initial set of attributes to an 
 * attribute with a new value, the join manager will not only propagates
 * the appropriate set of attributes to each lookup service with which the
 * test service is registered, but will also propagate those attributes to 
 * each new lookup service that may be started after the service's attributes
 * have been modified.
 * 
 */
public class ModifyAttributesPropNew extends ModifyAttributesProp {

    /** Executes the current test by doing the following:
     * <p>
     *   Changes the initial set of attributes to the new set, and verifies
     *   that the join manager propagates the appropriate attributes
     *   to each lookup service with which the associated test service is
     *   currently registered, and to each new lookup service with which
     *   the test service may be registered in the future.
     */
    public void run() throws Exception {
        boolean oldUseFastTimeout = useFastTimeout;
        useFastTimeout = false;//needs max discovery time for lookup start
        /* Change attributes & verify propagation to current lookup services */
        super.run();

        /* Stagger-start additional lookup services */
        logger.log(Level.FINE, "starting "+getnAddLookupServices()
                          +" additional lookup service(s) ...");
        LookupServices lookups = getLookupServices();
        Thread lookupsThread = lookups.staggeredStartThread(lookups.curLookupListSize());
//             new StaggeredStartThread(getLookupsStarted().size(), getAllLookupsToStart());
        lookupsThread.start();
	try {
            mainListener.clearAllEventInfo();
            mainListener.setLookupsToDiscover(getAddLookupsToStart(),
                                             toGroupsArray(getAddLookupsToStart()));
            waitForDiscovery(mainListener);
        } finally {
            /* If an exception occurred before the thread finished starting
             * all lookups, then we need to tell the thread to stop.
             *
             * If waitForDiscovery() somehow completed successfully, but the
             * thread is still running - creating lookups - then we still need
             * to tell the thread to stop so that it doesn't continue running
             * into the next test.
             */
            lookupsThread.interrupt();
            useFastTimeout = oldUseFastTimeout;//reset for next test
        }
        /* Verify attribute propagation to new lookup services */
        logger.log(Level.FINE, "verifying all attributes were "
                          +"propagated to each new lookup service ...");
        /* Wait for added lookups before beginning verification */
        verifyPropagation(expectedAttrs, getnSecsJoin());
        logger.log(Level.FINE, "join manager successfully propagated "
                          +"all attributes to the new lookup service(s)");
    }//end run

} //end class ModifyAttributesPropNew


