/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership. The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License. You may obtain a copy of the License at
 * 
 *      http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.river.test.impl.outrigger.matching;

// java.util
import java.util.Vector;
import java.util.List;
import java.util.LinkedList;
import java.util.Iterator;
import java.util.Enumeration;

// net.jini
import net.jini.core.event.RemoteEvent;
import net.jini.core.event.RemoteEventListener;
import net.jini.core.event.EventRegistration;
import net.jini.core.event.UnknownEventException;
import net.jini.core.entry.Entry;

// java.rmi
import java.rmi.MarshalledObject;
import java.io.IOException;
import java.rmi.RemoteException;

import java.io.Serializable;
import java.io.ObjectStreamException;

// java(x).security
import javax.security.auth.login.LoginContext;
import javax.security.auth.Subject;
import javax.security.auth.login.LoginException;
import java.security.PrivilegedExceptionAction;
import java.security.PrivilegedActionException;

import net.jini.config.Configuration;
import net.jini.config.ConfigurationException;
import net.jini.export.Exporter;
import net.jini.security.TrustVerifier;
import net.jini.security.proxytrust.ServerProxyTrust;

import org.apache.river.proxy.BasicProxyTrustVerifier;

import org.apache.river.qa.harness.QAConfig;
import java.rmi.server.ExportException;
import java.security.AccessControlContext;
import java.security.AccessController;
import java.security.PrivilegedAction;
import java.util.logging.Level;
import java.util.logging.Logger;

public class MonitoredSpaceListener
    implements RemoteEventListener, ServerProxyTrust, Serializable
{

    /* the proxy */
    private Object proxy;

    /*
     * The template of the writes this tests is supposed to match
     * Also used to synchronize access to expected & mayNotHaveHappened
     */
    final private Template notifyTmpl;

    // handBack object that was registered with the client's notify
    final private MarshalledObject handBack;

    // The client who really registered for this event
    final private RemoteEventListener client;

    // The record of events we have received
    final private Vector receptRecord = new Vector();

    // List of RemoteEvents that arived before we were initialized
    private List waitList = null;

    // Event Registration that was generated by the actual registration
    private EventRegistration registration = null;

    // The higest sequence number recived
    private long highestSeqNum = -1;

    // Number of events we are expecting
    private long expected = 0;

    /*
     * Number of events that we are expecting, but may not get because
     * the write threw a RemoteException
     */
    private long mayNotHaveHappened = 0;

    /*
     * A loggin context identical to that used by the real space
     */
    private LoginContext context;

    /*
     * A number of booleans that track invarents that all of the event
     * should maintain, if the invarent is not mainted the value is set to true
     */
    private boolean wrongID = false; // event with the wrong id
    private boolean wrongHandBack = false; // event with the wrong
                                           // Registration object
    private boolean wrongSource = false; // event with the wrong source
    private boolean early = false; // event before seq num in registation
    
    private final Exporter exporter;
    private final AccessControlContext acContext;

    /**
     * Create a <code>MonitoredSpaceListener</code> that will pass the
     * events it recives to the passed client client.
     * @param tmpl      The template that is listener is being registered
     *                  with
     * @param client    The listener events will be forwarded to
     * @param regObject The object that was registered by the client
     *                  with the notify.
     * @exception Throws IllegalAccessException or IOException if
     * <code>tmpl</code> is not a legal JavaSpace entry.
     */
    public MonitoredSpaceListener(Configuration c, Entry tmpl, RemoteEventListener client,
            MarshalledObject regObject)
            throws IllegalAccessException, IOException 
    {
	try {
	    Exporter exporter = QAConfig.getDefaultExporter();
	    if (c instanceof org.apache.river.qa.harness.QAConfiguration) {
		exporter =
		(Exporter) c.getEntry("test", "outriggerListenerExporter", Exporter.class);
	        context = (LoginContext) c.getEntry("test", 
	    					    "spaceLoginContext",
	    					    LoginContext.class);
	    }
            acContext = AccessController.getContext();
            this.exporter = exporter;
	    if (context != null) {
		context.login();
	    }
	} catch (ConfigurationException e) {
	    throw new RuntimeException("Bad configuration", e);
	} catch (LoginException e) {
	    throw new RuntimeException("Login failed", e);
	}
        notifyTmpl = new Template(tmpl);
        this.client = client;
        handBack = regObject;
    }
    
    public synchronized void export() throws ExportException {
        try {
            proxy = AccessController.doPrivileged(new PrivilegedExceptionAction(){

                @Override
                public Object run() throws ExportException {
                    return exporter.export(MonitoredSpaceListener.this);
                }
                
            }, acContext);
        } catch (PrivilegedActionException ex) {
            throw (ExportException) ex.getException();
        }
    }

    public synchronized Object writeReplace() throws ObjectStreamException {
        return proxy;
    }

    public synchronized TrustVerifier getProxyVerifier() {
	return new BasicProxyTrustVerifier(proxy);
    }

    /**
     * Remove large gaps in seq nums inserted by Outrigger on restart.
     * Outrigger inserts large gaps into the sequence numbers upon a
     * restart. The purpose of these large gaps (increments) is to
     * ensure no duplicate sequence numbers will be issued by notifications.
     * These gaps, however, only confuse the things we're trying to test
     * here so we mask them off.
     *
     * <b>NB: THIS IS OUTRIGGER SPECIFIC!</b>
     */
    private long maskRestarts(long seqNum) {
        long mi = Integer.MAX_VALUE;
        long restarts = seqNum / mi;
        long maskedSeqNum = seqNum - (restarts * mi);
        return maskedSeqNum;
    }

    // Methods to deal with recept record
    private void addToReceptRecord(RemoteEvent theEvent) {
        final int slot =
            (int)(((maskRestarts(theEvent.getSequenceNumber())) -
                   (maskRestarts(registration.getSequenceNumber()))) - 1);


        if (slot < 0) {
            early = true;
            return;
        }

        // Do we have room for this event?
        if (slot >= receptRecord.size()) {
            receptRecord.setSize(slot + 1);
        }

        // Do we already have this event on record?
        final Object rec = receptRecord.get(slot);

        if (rec == null) {

            // No add it
            receptRecord.set(slot, new Integer(1));
        } else {

            // Yes, add one to it
            receptRecord.set(slot, new Integer(((Integer) rec).intValue() + 1));
        }
    }

    private boolean duplicates() {
        for (Enumeration i = receptRecord.elements(); i.hasMoreElements();) {
            final Object rec = i.nextElement();

            if (rec != null && ((Integer) rec).intValue() > 1) {
                return true;
            }
        }
        return false;
    }

    // Function to do the real work of a notify
    private void processNotify(RemoteEvent theEvent) {

        // Check invariants
        if (theEvent.getID() != registration.getID()) {
            wrongID = true;
        }

        if (!theEvent.getSource().equals(registration.getSource())) {
            wrongSource = true;
        }

        if (handBack != null
                && !handBack.equals(theEvent.getRegistrationObject())) {
            wrongHandBack = true;
        }

        if (maskRestarts(theEvent.getSequenceNumber()) > highestSeqNum) {
            highestSeqNum = maskRestarts(theEvent.getSequenceNumber());
        }
        addToReceptRecord(theEvent);
    }

    /**
     * Processes the remote call from the space. Before calling the real
     * client, the remote call thread assumes the identity of the
     * space.
     */
    public void notify(RemoteEvent theEvent)
            throws UnknownEventException, java.rmi.RemoteException {

        // Do we have the event registraion info yet?
        synchronized (this) {
            if (registration == null) {

                // No queue it up
                if (waitList == null) {
                    waitList = new LinkedList();
                }
                waitList.add(theEvent);
            } else {

                // Process now
                processNotify(theEvent);
            }
        }
	
        // Ether-way, tell the client now
        if (client != null) {
	    if (context == null) {
		client.notify(theEvent);
	    } else {
		try {
		    final RemoteEvent ev = theEvent;
		    Subject.doAs(context.getSubject(),
				 new PrivilegedExceptionAction() {
					 public Object run() throws UnknownEventException, RemoteException {
					     client.notify(ev);
					     return null;
					 }
				     });
		} catch (PrivilegedActionException e) {
		    Throwable t = e.getException();
		    if (t instanceof RemoteException) {
			throw (RemoteException) t;
		    }
		    if (t instanceof UnknownEventException) {
			throw (UnknownEventException) t;
		    }
		}
	    }
	}
    }

    /**
     * Completes registration process.
     * @param registration The <code>EventRegistration</code> object
     *                     retured by the notify call this was passed to.
     * @return <code>EventRegistration</code> that should be retured
     * to the caller of <code>JavaSpace.notify()</code>.
     */
    synchronized EventRegistration complete(EventRegistration registration) {
        this.registration = registration;

        // Clear the waitList
        if (waitList != null) {
            for (Iterator i = waitList.iterator(); i.hasNext();) {
                processNotify((RemoteEvent) i.next());
            }
            waitList = null;
        }
        return registration;
    }

    /**
     * Lets listener know if it should be expecting an event
     * corresponding to this <code>Entry</code>.  Checks to see if the
     * <code>Entry</code> matches the template that was registered
     * with the listener.
     * @exception Throws IllegalAccessException or IOException if
     * <code>entry</code> is not a legal JavaSpace entry.
     */
    void expect(Entry entry, boolean remoteException)
            throws IllegalAccessException, IOException {
        if (notifyTmpl.doesMatch(entry)) {

            /*
             * Using notifyTmpl to syncronize access to expected &
             * mayNotHaveHappened
             */
            synchronized (notifyTmpl) {
                expected++;

                if (remoteException) {
                    mayNotHaveHappened++;
                }
            }
        }
    }

    /**
     * Returns a string describing any errors that the monitor has
     * detected.
     * @param ignoreDuplicates If <code>true</code> does not consider
     *                         duplicate events an error.
     * @return A string describing any detected errors, or
     * <code>null</code> if no errors have been detected.
     */
    String getErrors(boolean ignoreDuplicates) {
        String rslt = "";
        long curExpected;
        long curMayNotHaveHappened;
        synchronized (notifyTmpl) {
            curExpected = expected;
            curMayNotHaveHappened = mayNotHaveHappened;
        }

        if (highestSeqNum == -1 && curExpected > 0) {
            rslt = rslt + "Was exspecting at least one event and recived none";
        } else {
            final long got = (highestSeqNum == -1) ?
                0:highestSeqNum - (
                        maskRestarts(registration.getSequenceNumber()));

            final long missing = curExpected - got;

            if (missing > 0) {

                // could it be explained by RemoteExceptions on write?
                if (missing <= curMayNotHaveHappened) {
                    rslt = rslt + "Did not get " + missing + " event "
                            + "notification, but " + curMayNotHaveHappened
                            + " writes may not have happened\n";
                } else {
                    rslt = rslt + missing
                            + " event notifications are missing\n";
                }
            }

            if (missing < 0) {
                rslt = rslt + "Received " + (-1 * missing)
                        + " unexpected events";
            }
        }

        if (wrongID) {
            rslt = rslt + "Received an event with the wrong ID\n";
        }

        if (wrongHandBack) {
            rslt = rslt + "Received an event with the wrong hand back object\n";
        }

        if (wrongSource) {
            rslt = rslt + "Received an event with the wrong source\n";
        }

        if (early) {
            rslt = rslt + "Received an event with a lower sequence "
                    + "number than listed in the event registration\n";
        }

        if (!ignoreDuplicates && duplicates()) {
            rslt = rslt + "Received duplicate events\n";
        }

        if (rslt.equals("")) {
            return null;
        }
        return rslt;
    }
}
