/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership. The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License. You may obtain a copy of the License at
 * 
 *      http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

/**
 * A program which generates the ConstraintsInterface.java source
 * based on a specific method naming convention
 */
package org.apache.river.test.impl.end2end.generators;

import java.rmi.Remote;
import java.rmi.RemoteException;

/**
 * The naming convention for generating interface methods is:
 *
 *  the first character of the method name is one of:
 *    o - return value is an simple object
 *    w - return value is an object with a writeObject method
 *    p - return value is a primitive
 *    v - return value is a void
 *
 *  there follows a string of characters, one per argument:
 *    o - parameter is an simple object
 *    r - parameter is an object with a readObject method
 *    p - parameter is a primitive
 *
 *  then follows a string of tokens describing the server constraints
 *
 *    Auth   => client authentication
 *    Noauth => no server authenticaion
 *    Conf   => confidentiality
 *    Noconf => no confidentiality
 *    Integ  => integrity
 *    CMinP  => ClientMinPrincipal
 *    CMinTp => ClientMinPrincipalType
 *    Alt1   => ConstraintAlternatives with clientDSA1/clientRSA1 principals
 *    Alt2   => ConstraintAlternatives with clientDSA2/clientRSA2 principals
 *
 *  The Noint variant is not generated
 *  as the provider always performs integrity checking, so if
 *  the NoInt constraint were applied by the server the export
 *  would fail.
 *
 *  several objects should implement this interface, each object
 *  exported with a different set of principals to exercise the
 *  authentication code.
 *
 *  the interface generated by this code contains the following
 *  combinations:
 *
 *  for the Auth and Noauth solo cases, all combinations of parameters
 *  and return values are generated. Otherwise, all possible combinations
 *  of constraints are generated, passing no arguments and returning void.
 *  Another exception is that the AuthNoauth combination is excluded, becuase
 *  jsse does not support client authentication without server authenication
 *  Another exception is that the NoauthNoconf combination is exclued, because
 *  the only ciphersuites which support Noconf are those which provide RSA
 *  authentication.
 *
 *  An interface method named vBogus is also created. A permission to
 *  execute this remote method is not included in the permissions file,
 *  so this call should fail.
 *
 *  Several interface methods named multi are also created overloaded with
 *  various calling parameters. This is used to test the wild card
 *  parameters field in method constraints.
 */

public class InterfaceGenerator {

    public static void main(String[] args) {

//      String[] retStrings = {"o","w","p","v"};
//  	String[] paramStrings = {"","o","r","p"};

	//XXX primitives don't marshall properly
  	String[] retStrings = {"o","w","v"};
  	String[] paramStrings = {"","o","r"};

	String[] constraintStrings = {
                                      "Auth",
                                      "Noauth",
				      "Conf",
				      "Noconf",
				      "Integ",
				      "CMinP",
				      "CMinTp",
				      "Alt1",
				      "Alt2"
				     };
        /*
	 * the BADMIN1MASK/BADMIN2MASK combinations must be
	 * discarded because the the combined constraints
	 * results in the creation of ClientMinPrincipal
	 * constraints containing multiple principals. The
	 * JSSE provider only supports ClientMinPrincipals
	 * with single principals.
	 */
        int BADAUTHMASK = 3;
        int BADCONFMASK = 0xc;
        int BADAUTHCONFMASK = 0xa;
        int BADALTMASK = 0x180;
        int BADMIN1MASK = 0xa0 ; // CMinP conflicts with Alt1
        int BADMIN2MASK = 0x120 ; // CMinP conflicts with Alt2
	String[] authStrings = {"Auth","Noauth"};

	print("package org.apache.river.test.impl.end2end.e2etest;");
	print("import java.rmi.Remote;");
	print("import java.rmi.RemoteException;");
	print("");
        print("/**");
	print(" * The interface which defines remote methods to be executed");
	print(" * as part of the test.");
	print(" *");
	print(" * WARNING: THIS INTERFACE IS GENERATED AUTOMATICALLY AND ");
	print(" *          SHOULD NOT BE EDITED.");
	print(" */");
	print("");
	print("public interface ConstraintsInterface extends Remote {");

	/* first, handle the Auth/Noauth special case */
	for (int i=0; i<authStrings.length; i++) {
	    for (int j=0; j<retStrings.length; j++) {
		for (int k=0; k<paramStrings.length; k++) {
		    for (int n=k; n<paramStrings.length; n++) {
			StringBuffer sb = new StringBuffer(retStrings[j]);
			sb.append(paramStrings[k]);
			sb.append(paramStrings[n]);
			sb.append(authStrings[i]);
			print(getInterfaceMethod(sb.toString()));
		    }
		}
	    }
	}

	/* now handle the combinations of constraints */
        for (int i=0; i<(1<<(constraintStrings.length)); i++) {
            int count = i;
            if (i == 1) continue; // avoid duplicate vAuth method
            if (i == 2) continue; // avoid duplicate vNoauth method
            if ((count & BADAUTHMASK) == BADAUTHMASK) continue;
            if ((count & BADCONFMASK) == BADCONFMASK) continue;
            if ((count & BADAUTHCONFMASK) == BADAUTHCONFMASK) continue;
            if ((count & BADALTMASK) == BADALTMASK) continue;
            if ((count & BADMIN1MASK) == BADMIN1MASK) continue;
            if ((count & BADMIN2MASK) == BADMIN2MASK) continue;
	    StringBuffer sb = new StringBuffer("v");
            for (int j=0; j<constraintStrings.length; j++) {
		if ((count & 1) != 0) sb.append(constraintStrings[j]);
		count >>= 1;
	    }
	    print(getInterfaceMethod(sb.toString()));
	}

	/*
         * add a method which should fail a permission check
	 * when the server imposes ClientAuthentication.YES, since
	 * the security file generator will not add a permission to
	 * invoke this method
	 */
        print(getInterfaceMethod("vBogus"));
        printMultiMethods();

        print("}");
    }

    private static String getInterfaceMethod(String methodName) {
	StringBuffer sb = new StringBuffer();
	sb.append("\n");
	sb.append("    public ");
	if (methodName.startsWith("v")) sb.append("void ");
	if (methodName.startsWith("w")) sb.append("WriterObject ");
	if (methodName.startsWith("p")) sb.append("int ");
	if (methodName.startsWith("o")) sb.append("PlainObject ");
	sb.append(methodName + "(");
	String trunc = methodName.substring(1);
        if (trunc.length() > 0) {
            String args1= "";
            String args2 = "";
	    if (trunc.startsWith("o")) args1 = "PlainObject obj1";
	    if (trunc.startsWith("r")) args1 = "ReaderObject obj1";
	    if (trunc.startsWith("p")) args1 = "int i1";
	    trunc = trunc.substring(1);
	    if (trunc.length() > 0 && args1.length() != 0) {
		if (trunc.startsWith("o")) args2 = "PlainObject obj2";
		if (trunc.startsWith("r")) args2 = "ReaderObject obj2";
		if (trunc.startsWith("p")) args2 = "int i2";
	    }
            String comma = "";
            if (args1.length() > 0 && args2.length() > 0)comma = ",";
            sb.append(args1 + comma + args2);
	}
	sb.append(") throws RemoteException;");
        return sb.toString();
    }

    private static void printMultiMethods() {
        String[] args = {"",
                         "int i",
			 "int i, ReaderObject obj1",
			 "PlainObject obj1",
			 "PlainObject obj1, ReaderObject obj2"};
	for (int i=0; i < args.length; i++) {
	    StringBuffer sb = new StringBuffer();
	    sb.append("\n");
	    sb.append("    public ");
	    sb.append("void ");
	    sb.append("multi(");
	    sb.append(args[i]);
	    sb.append(") throws RemoteException;");
	    print(sb.toString());
        }
    }

    private static void print(String s) {
        System.out.println(s);
    }
}
