/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership. The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License. You may obtain a copy of the License at
 * 
 *      http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
/* @test 
 * @bug 4408449
 * 
 * @summary Make sure that the preferred class provider can tell when
 * the preferred list definitely does not exist in the resources of a
 * class loader
 *
 * @author Laird Dornin
 *
 * @library ../../../../../../testlibrary
 * @build CheckDefinite
 * @build TestLibrary TestParams HTTPD
 *
 * @run main/othervm/policy=security.policy/timeout=100 -DtrustProxy=true CheckDefinite
 */

import java.net.URL;

import java.rmi.server.RMIClassLoader;
import java.io.IOException;
import java.io.File;
import java.net.MalformedURLException;
import java.net.URI;
import java.net.URISyntaxException;
import java.util.logging.Level;
import java.util.logging.Logger;

/**
 * Ensure that exceptions thrown when searching for a preferred list
 * that is definitely not found do not propagate to user code.
 *
 * Also verifies that exceptions generated by preferred list searches
 * with indefinite results do get thrown to user code.  There are 7
 * cases with decriptions provided below.
 */
public class CheckDefinite {
    static {
	// set the logging configuration file
	System.setProperty("java.util.logging.config.file",
			   TestParams.testSrc + "/../../logging.properties");
    }

    private static String SRC_URL = "file:";
    private static String BASE = "";
    static {
	try {
	    if (!TestParams.testSrc.startsWith(".")) {
		SRC_URL = (new URI("file://" +
		    (TestParams.testSrc + File.separator).
		    replace(File.separatorChar, '/')).toString());

		BASE = TestParams.testSrc;
	    }
	} catch (URISyntaxException e) {
	    e.printStackTrace();
	}
    }

    /* set up urls for class load requests */
    private static String fileCbWithClass = SRC_URL + "cbArchive/";
    private static String httpCbNoPreferredWithJar =
	"http://localhost:9381/nopreferred.jar";
    private static String httpCbWithNoPreferredWithZip =
	"http://localhost:9381/nopreferred.zip";
    private static String httpCbNonexistentJar  =
	"http://localhost:9381/nonexistent.jar";
    private static String httpCbWrongPort =
	"http://localhost:9091/nopreferred.jar";

    public static void main(String[] args) {
	HTTPD codeServer = null;
	try {

	    /*  run http codeserver */
	    codeServer = new HTTPD(9381, BASE + File.separator + "cbArchive");
  	    Thread.sleep(8 * 1000);

	    TestLibrary.suggestSecurityManager(null);

	    /*
	     * loadClass on a class loader with no preferred list
	     * and first url that is a file url.
	     */
	    System.err.println("1. load class from file cb: " +
			       fileCbWithClass);
	    Class c = RMIClassLoader.loadClass(fileCbWithClass, "Foo");
	    
	    /*
	     * loadClass on a class loader with no preferred list
	     * and first url that is an http url. a jar file can be
	     * found at the first url.
	     */
	    System.err.println("2. load class from jar/http cb: " +
			       httpCbNoPreferredWithJar);
	    c = RMIClassLoader.loadClass(httpCbNoPreferredWithJar, "Foo");

	    /*
             * loadClass on a class loader with no preferred list
	     * and first url that is an http url. a zip file can be
	     * found at the first url.
	     */
	    System.err.println("3. load class from zip/http cb: " +
			       httpCbWithNoPreferredWithZip);
	    c = RMIClassLoader.loadClass(httpCbWithNoPreferredWithZip, "Foo");

	    /*
	     * loadClass on a class loader where the directory of the
	     * jar file referenced in the loader's first url can be
	     * found, but the jar file itself can not be found.  the
	     * first url is an http url.
	     */
	    String cbTest4 = httpCbNonexistentJar + " " +
		httpCbNoPreferredWithJar;
	    System.err.println("4. load class from path with first url " +
			       "nonexistent: " + cbTest4);
	    c = RMIClassLoader.loadClass(cbTest4, "Foo");

	    /*
	     * loadClass on a class loader whose first URL references
	     * a directory on a non-existent machine.
	     */
	    case5();

	    /*
	     * loadClass on a class loader whose first URL references
	     * a jar file on a non-existent machine.
	     */
	    case6();	    

	    /*
	     * loadClass on a class loader whose first URL references
	     * a valid machine on a port where no program is
	     * listening.
	     */
	    case7();
	    
	    System.err.println("Test passed: exceptions appropriately " +
			       "thrown for preferred list searches");
	} catch (ClassNotFoundException e) {

	    // REMIND: put this kind of handling into TestLibrary.bomb()
	    Exception detail = (Exception) e.getException();
	    if (detail != null) {
		detail.printStackTrace();
	    } else {
		System.err.println("exception detail was null");
	    }
	    TestLibrary.bomb("unexpected exception", e);
	} catch (Exception e) {
	    if (e instanceof RuntimeException) {
		throw (RuntimeException) e;
	    }
	    TestLibrary.bomb("unexpected exception", e);
	} finally {
	    if (codeServer != null) {
		try {
		    codeServer.stop();
		} catch (IOException e) {
		}
	    }
	}
    }

    /*
     * loadClass on a class loader whose first URL references a
     * directory on a non-existent machine.
     *
     * Thrown ClassNotFoundException should contain an IOException
     * IOException
     */
    private static void case5() throws Exception {
	Exception ex = null;
	try {
	    String mdneDir = "http://machineDoesNotExist:7896/";
	    System.err.println("5. load class from nonexistent " +
			       "machine cb: " + mdneDir);
	    Class c = RMIClassLoader.loadClass(mdneDir, "Foo");
	} catch (ClassNotFoundException e) {
	    ex = (Exception) e.getException();
	    if (ex == null) {
		ex = e;
	    }
	}
	if (ex == null) {
	    TestLibrary.bomb("exception not thrown for url " +
			     "indefinitely present PREFERRED.LIST");
	}
	System.err.println("Exception message: " + ex.getMessage());
    }

    /*
     * loadClass on a class loader whose first URL references a jar
     * file on a non-existent machine.
     *
     * thrown ClassNotFoundException should contain an IOException
     */
    private static void case6() throws Exception {
	Exception ex = null;
	try {
	    String mdneJar = "http://machineDoesNotExist:7896/ajar.jar";
	    System.err.println("6. load class from nonexistent " +
			       "machine cb: " + mdneJar);
	    Class c = RMIClassLoader.loadClass(mdneJar, "Foo");
	} catch (ClassNotFoundException e) {
	    ex = (IOException) e.getException();
	    if (ex == null) {
		ex = e;
	    }
	}
	if (ex == null) {
	    TestLibrary.bomb("exception not thrown for ambiguously " +
			     "present PREFERRED.LIST");
	}
	System.err.println("Exception message: " + ex.getMessage());
   
    }

    /*
     * loadClass on a class loader whose first URL references a valid
     * machine on a port where no program is listening.
     *
     * thrown ClassNotFoundException should contain an IOException
     */
    private static void case7() throws Exception {
	Exception ex = null;
	try {
	    System.err.println("7. attempt to load from a " +
			       "port on which a machine is not listening:"+
			       httpCbWrongPort);
	    Class c = RMIClassLoader.loadClass(httpCbWrongPort, "Foo");
	} catch (ClassNotFoundException e) {
	    ex = (IOException) e.getException();
	    if (ex == null) {
		ex = e;
	    }
	}
	if (ex == null) {
	    TestLibrary.bomb("exception not thrown for ambiguously " +
			     "present PREFERRED.LIST");
	}
	System.err.println("Exception message: " + ex.getMessage());
    }
}
