/*
 * Decompiled with CFR 0.152.
 */
package org.apache.qpid.server.transport;

import java.io.IOException;
import java.security.Principal;
import java.security.cert.Certificate;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Iterator;
import java.util.List;
import java.util.ListIterator;
import javax.net.ssl.SSLEngine;
import javax.net.ssl.SSLEngineResult;
import javax.net.ssl.SSLException;
import javax.net.ssl.SSLPeerUnverifiedException;
import javax.net.ssl.SSLSession;
import org.apache.qpid.bytebuffer.QpidByteBuffer;
import org.apache.qpid.server.model.port.AmqpPort;
import org.apache.qpid.server.transport.NonBlockingConnection;
import org.apache.qpid.server.transport.NonBlockingConnectionDelegate;
import org.apache.qpid.server.util.ServerScopedRuntimeException;
import org.apache.qpid.transport.network.security.ssl.SSLUtil;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class NonBlockingConnectionTLSDelegate
implements NonBlockingConnectionDelegate {
    private static final Logger LOGGER = LoggerFactory.getLogger(NonBlockingConnectionTLSDelegate.class);
    private final SSLEngine _sslEngine;
    private final NonBlockingConnection _parent;
    private final int _networkBufferSize;
    private SSLEngineResult _status;
    private final List<QpidByteBuffer> _encryptedOutput = new ArrayList<QpidByteBuffer>();
    private Principal _principal;
    private Certificate _peerCertificate;
    private boolean _principalChecked;
    private QpidByteBuffer _netInputBuffer;
    private QpidByteBuffer _netOutputBuffer;
    private QpidByteBuffer _applicationBuffer;

    public NonBlockingConnectionTLSDelegate(NonBlockingConnection parent, AmqpPort port) {
        this._parent = parent;
        this._sslEngine = this.createSSLEngine(port);
        this._networkBufferSize = port.getNetworkBufferSize();
        int tlsPacketBufferSize = this._sslEngine.getSession().getPacketBufferSize();
        if (tlsPacketBufferSize > this._networkBufferSize) {
            throw new ServerScopedRuntimeException("TLS implementation packet buffer size (" + tlsPacketBufferSize + ") is greater then broker network buffer size (" + this._networkBufferSize + ")");
        }
        this._netInputBuffer = QpidByteBuffer.allocateDirect((int)this._networkBufferSize);
        this._applicationBuffer = QpidByteBuffer.allocateDirect((int)this._networkBufferSize);
        this._netOutputBuffer = QpidByteBuffer.allocateDirect((int)this._networkBufferSize);
    }

    @Override
    public boolean readyForRead() {
        return this._sslEngine.getHandshakeStatus() != SSLEngineResult.HandshakeStatus.NEED_WRAP;
    }

    @Override
    public boolean processData() throws IOException {
        boolean tasksRun;
        int oldNetBufferPos;
        this._netInputBuffer.flip();
        boolean readData = false;
        do {
            int oldAppBufPos = this._applicationBuffer.position();
            oldNetBufferPos = this._netInputBuffer.position();
            this._status = this._netInputBuffer.decryptSSL(this._sslEngine, this._applicationBuffer);
            if (this._status.getStatus() == SSLEngineResult.Status.CLOSED) {
                int remaining = this._netInputBuffer.remaining();
                this._netInputBuffer.position(this._netInputBuffer.limit());
                LOGGER.debug("SSLEngine closed, discarded {} byte(s)", (Object)remaining);
            }
            tasksRun = this.runSSLEngineTasks(this._status);
            this._applicationBuffer.flip();
            if (this._applicationBuffer.position() > oldAppBufPos) {
                readData = true;
            }
            this._parent.processAmqpData(this._applicationBuffer);
            this.restoreApplicationBufferForWrite();
        } while (this._netInputBuffer.hasRemaining() && this._netInputBuffer.position() > oldNetBufferPos || tasksRun);
        if (this._netInputBuffer.hasRemaining()) {
            this._netInputBuffer.compact();
        } else {
            this._netInputBuffer.clear();
        }
        return readData;
    }

    @Override
    public boolean doWrite(Collection<QpidByteBuffer> bufferArray) throws IOException {
        int bufCount = bufferArray.size();
        this.wrapBufferArray(bufferArray);
        boolean bufsSent = true;
        Iterator<QpidByteBuffer> itr = bufferArray.iterator();
        int bufIndex = 0;
        while (itr.hasNext() && bufsSent && bufIndex++ < bufCount) {
            QpidByteBuffer buf = itr.next();
            bufsSent = !buf.hasRemaining();
        }
        if (!this._encryptedOutput.isEmpty()) {
            QpidByteBuffer buf;
            this._parent.writeToTransport(this._encryptedOutput);
            ListIterator<QpidByteBuffer> iter = this._encryptedOutput.listIterator();
            while (iter.hasNext() && (buf = iter.next()).remaining() == 0) {
                buf.dispose();
                iter.remove();
            }
        }
        return bufsSent && this._encryptedOutput.isEmpty();
    }

    protected void restoreApplicationBufferForWrite() {
        QpidByteBuffer oldApplicationBuffer = this._applicationBuffer;
        int unprocessedDataLength = this._applicationBuffer.remaining();
        this._applicationBuffer.limit(this._applicationBuffer.capacity());
        this._applicationBuffer = this._applicationBuffer.slice();
        this._applicationBuffer.limit(unprocessedDataLength);
        oldApplicationBuffer.dispose();
        if (this._applicationBuffer.limit() <= this._applicationBuffer.capacity() - this._sslEngine.getSession().getApplicationBufferSize()) {
            this._applicationBuffer.position(this._applicationBuffer.limit());
            this._applicationBuffer.limit(this._applicationBuffer.capacity());
        } else {
            int newBufSize;
            QpidByteBuffer currentBuffer = this._applicationBuffer;
            if (currentBuffer.capacity() < this._networkBufferSize) {
                newBufSize = this._networkBufferSize;
            } else {
                newBufSize = currentBuffer.capacity() + this._networkBufferSize;
                this._parent.reportUnexpectedByteBufferSizeUsage();
            }
            this._applicationBuffer = QpidByteBuffer.allocateDirect((int)newBufSize);
            this._applicationBuffer.put(currentBuffer);
            currentBuffer.dispose();
        }
    }

    private void wrapBufferArray(Collection<QpidByteBuffer> bufferArray) throws SSLException {
        boolean encrypted;
        do {
            if (this._sslEngine.getHandshakeStatus() != SSLEngineResult.HandshakeStatus.NEED_UNWRAP) {
                if (this._netOutputBuffer.remaining() < this._sslEngine.getSession().getPacketBufferSize()) {
                    if (this._netOutputBuffer.position() != 0) {
                        this._netOutputBuffer.flip();
                        this._encryptedOutput.add(this._netOutputBuffer);
                    } else {
                        this._netOutputBuffer.dispose();
                    }
                    this._netOutputBuffer = QpidByteBuffer.allocateDirect((int)this._networkBufferSize);
                }
                this._status = QpidByteBuffer.encryptSSL((SSLEngine)this._sslEngine, bufferArray, (QpidByteBuffer)this._netOutputBuffer);
                encrypted = this._status.bytesProduced() > 0;
                this.runSSLEngineTasks(this._status);
                if (!encrypted || this._netOutputBuffer.remaining() >= this._sslEngine.getSession().getPacketBufferSize()) continue;
                this._netOutputBuffer.flip();
                this._encryptedOutput.add(this._netOutputBuffer);
                this._netOutputBuffer = QpidByteBuffer.allocateDirect((int)this._networkBufferSize);
                continue;
            }
            encrypted = false;
        } while (encrypted && this._sslEngine.getHandshakeStatus() != SSLEngineResult.HandshakeStatus.NEED_UNWRAP);
        if (this._netOutputBuffer.position() != 0) {
            QpidByteBuffer outputBuffer = this._netOutputBuffer;
            this._netOutputBuffer = this._netOutputBuffer.slice();
            outputBuffer.flip();
            this._encryptedOutput.add(outputBuffer);
        }
    }

    private boolean runSSLEngineTasks(SSLEngineResult status) {
        if (status.getHandshakeStatus() == SSLEngineResult.HandshakeStatus.NEED_TASK) {
            Runnable task;
            while ((task = this._sslEngine.getDelegatedTask()) != null) {
                task.run();
            }
            return true;
        }
        return false;
    }

    @Override
    public Principal getPeerPrincipal() {
        this.checkPeerPrincipal();
        return this._principal;
    }

    @Override
    public Certificate getPeerCertificate() {
        this.checkPeerPrincipal();
        return this._peerCertificate;
    }

    @Override
    public boolean needsWork() {
        return this._sslEngine.getHandshakeStatus() != SSLEngineResult.HandshakeStatus.NEED_UNWRAP;
    }

    private synchronized void checkPeerPrincipal() {
        if (!this._principalChecked) {
            try {
                this._principal = this._sslEngine.getSession().getPeerPrincipal();
                Certificate[] peerCertificates = this._sslEngine.getSession().getPeerCertificates();
                if (peerCertificates != null && peerCertificates.length > 0) {
                    this._peerCertificate = peerCertificates[0];
                }
            }
            catch (SSLPeerUnverifiedException e) {
                this._principal = null;
                this._peerCertificate = null;
            }
            this._principalChecked = true;
        }
    }

    private SSLEngine createSSLEngine(AmqpPort<?> port) {
        SSLEngine sslEngine = port.getSSLContext().createSSLEngine();
        sslEngine.setUseClientMode(false);
        SSLUtil.updateEnabledTlsProtocols((SSLEngine)sslEngine, port.getTlsProtocolWhiteList(), port.getTlsProtocolBlackList());
        SSLUtil.updateEnabledCipherSuites((SSLEngine)sslEngine, port.getTlsCipherSuiteWhiteList(), port.getTlsCipherSuiteBlackList());
        if (port.getTlsCipherSuiteWhiteList() != null && !port.getTlsCipherSuiteWhiteList().isEmpty()) {
            SSLUtil.useCipherOrderIfPossible((SSLEngine)sslEngine);
        }
        if (port.getNeedClientAuth()) {
            sslEngine.setNeedClientAuth(true);
        } else if (port.getWantClientAuth()) {
            sslEngine.setWantClientAuth(true);
        }
        return sslEngine;
    }

    @Override
    public QpidByteBuffer getNetInputBuffer() {
        return this._netInputBuffer;
    }

    @Override
    public void shutdownInput() {
        if (this._netInputBuffer != null) {
            this._netInputBuffer.dispose();
            this._netInputBuffer = null;
        }
        if (this._applicationBuffer != null) {
            this._applicationBuffer.dispose();
            this._applicationBuffer = null;
        }
    }

    @Override
    public void shutdownOutput() {
        if (this._netOutputBuffer != null) {
            this._netOutputBuffer.dispose();
            this._netOutputBuffer = null;
        }
        try {
            this._sslEngine.closeOutbound();
            this._sslEngine.closeInbound();
        }
        catch (SSLException e) {
            LOGGER.debug("Exception when closing SSLEngine", (Throwable)e);
        }
    }

    @Override
    public String getTransportInfo() {
        SSLSession session = this._sslEngine.getSession();
        return session.getProtocol() + " ; " + session.getCipherSuite();
    }
}

