package antexts.taskdef.file;

import org.apache.tools.ant.Task;
import org.apache.tools.ant.BuildException;

import java.io.File;
/**
 * Task definition for the basename task. This works similar to the basename
 * command. basename strips off the leading part of a path name, leaving only the
 * final component of the name, which is assumed to be the file name.
 *
 * Functionally this task behaves the same way as the original basename task. But
 * this task overwrites the value of property if it already exists.
 * 
 * <pre>
 * Usage:
 *   <code>
 *     &lt;basename path="/root/path/somefile" property="props"
 *               [suffix="suffix"]/&gt;
 *   </code>
 *
 * Attributes:
 *      path     =  The path string to process
 *      property =  Name of a property to return the basename in.
 *      suffix   =  (Optional) if specified and the remaining portion of name contains
 *                  a suffix which matches suffix, it is removed.
 * </pre>
 */
public class BasenameTask extends Task
{
    private String _path;
    private String _property;
    private String _suffix;

    private void validate()
    {
        if ((_path == null) || (_path.trim().length() == 0))
            throw new BuildException("Must specify the path attribute");
        if ((_property == null) || (_property.trim().length() == 0))
            throw new BuildException("Must specify the property to set");
    }

    public void setPath(String path)
    {
        _path = path;
    }

    public String getPath()
    {
        return _path;
    }

    public void setProperty(String property)
    {
        _property = property;
    }

    public String getProperty()
    {
        return _property;
    }

    public void setSuffix(String suffix)
    {
        _suffix = suffix;
    }

    public String getSuffix(String suffix)
    {
        return _suffix;
    }

    public void execute()
            throws BuildException
    {
        validate();
        File file = new File(_path);
        String basename = file.getName();

        if ((_suffix != null) && (_suffix.trim().length() > 0))
        {
            int in = basename.lastIndexOf(_suffix.trim());
            if (in > 0)
                basename = basename.substring(0, in);
        }
        getProject().setProperty(getProperty(), basename);
    }
}
