package antexts.taskdef.logic;

import org.apache.tools.ant.taskdefs.Sequential;
import org.apache.tools.ant.types.Path;
import org.apache.tools.ant.BuildException;
import org.apache.tools.ant.Task;
import org.apache.tools.ant.taskdefs.Echo;

import java.util.ArrayList;
import java.util.StringTokenizer;

/**
 * Foreach loop task. This task iterates over a list of tokens from a string, or a set of files
 * specified in <path> or both if both are present. For each iteration it loops over any valid
 * Ant tasks that are specified in a <code><loop></code> tag. The iterations run in sequence.
 * Parallel execution is not supported. If both list and path are specified, then elements from
 * list are first processed.
 *
 * <pre>
 * Usage:
 *   <code>
 *     &lt;foreach list="some string" param="property-name"
 *               [trim="true|false"]
 *               [delimiter="delimiter string"]&gt;
 *             &lt;path&gt;...&lt;/path&gt;
 *             &lt;loop&gt;
 *                 ... ant tasks ...
 *              &lt;/loop&gt;
 *     &lt;/foreach&gt;
 *   </code>
 *
 * Attributes:
 *      list      =  The string to split up
 *      param     =  Name of a property to return the current token in.
 *      trim      =  (Optional) if specified any leading and trailing whitespaces are removed
 *      delimiter =  (Optional) The delimiter for the list string. Defaults to ','
 *
 * Valid Nested elements
 *     Path       =  Ant path like elements
 *     loop       =  Container for Ant tasks           
 * </pre>
 */
public class ForEachLoopTask extends Task
{
    private static String defaultDelim = ",";

    private String _list;
    private String _param;
    private String _delimiter = defaultDelim;
    private boolean _trim = false;
    private Path _path;
    private Sequential _loopTasks;

    public void setList(String list)
    {
        _list = list;
    }

    public String getList()
    {
        return _list;
    }

    public void setParam(String param)
    {
        _param = param;
    }

    public String getParam()
    {
        return _param;
    }

    public void setDelimiter(String delimiter)
    {
        _delimiter = delimiter;
    }

    public String getDelimiter()
    {
        return _delimiter;
    }

    public void setTrim(boolean trim)
    {
        _trim = trim;
    }

    public boolean getTrim()
    {
        return _trim;
    }

    public Path createPath() {
        if (_path == null) {
            _path = new Path(getProject());
        }
        return _path;
    }

    public void addLoop(Sequential loop)
    {
        if (_loopTasks != null) {
            throw new BuildException("Only one <loop> child can be present inside Foreachloop");
        }
        _loopTasks = loop;
    }

    public void validate()
    {
        if (((_list == null) || (_list.trim().length() == 0))
                        && (_path == null))
        {
            throw new BuildException("Must speciy either 'list' or nested path");
        }

        if (_param == null)
            throw new BuildException("Must specify the param attribute");
    }


    public void execute()
            throws BuildException
    {
        validate();

        ArrayList elements = new ArrayList();

        if (_list != null)
        {
            StringTokenizer tokens = new StringTokenizer(_list, _delimiter);
            while (tokens.hasMoreElements())
            {
                String token = tokens.nextToken();
                if (_trim)
                {
                    if (token.trim().length() > 0)
                        elements.add(token.trim());
                }
                else
                    elements.add(token);
            }
        }

        // Now process the Path
        if (_path != null)
        {
            String[] files = _path.list();

            for (int i = 0; i < files.length; i++)
                elements.add(files[i]);
        }

        int lc = elements.size();

        for (int i = 0; i < lc; i++)
        {
            getProject().setProperty(_param, (String) elements.get(i));
            _loopTasks.execute();
        }
    }

}
