/*
Licensed to the Apache Software Foundation (ASF) under one
or more contributor license agreements.  See the NOTICE file
distributed with this work for additional information
regarding copyright ownership.  The ASF licenses this file
to you under the Apache License, Version 2.0 (the
"License"); you may not use this file except in compliance
with the License.  You may obtain a copy of the License at

  http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing,
software distributed under the License is distributed on an
"AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
KIND, either express or implied.  See the License for the
specific language governing permissions and limitations
under the License.
*/
package org.apache.plc4x.java.isotp.protocol.model.types;

import java.util.HashMap;
import java.util.Map;

/**
 * Generally based upon RFC 905, but there were some inconsistencies in the spec:
 * - RESIDUAL_ERROR_RATE was defined twice with different codes
 * - The code 0x89 was defined for two types.
 * This version was extracted from information generated by the PcapGenerator.
 */
public enum ParameterCode {
    ATN_EXTENDED_CHECKSUM_32BIT((byte) 0x08),
    ATN_EXTENDED_CHECKSUM_16BIT((byte) 0x09),
    ACKNOWLEDGE_TIME((byte) 0x85),
    RESIDUAL_ERROR_RATE((byte) 0x86),
    PRIORITY((byte) 0x87),
    TRANSIT_DELAY((byte) 0x88),
    THROUGHPUT((byte) 0x89),
    SEQUENCE_NUMBER((byte) 0x8A),
    REASSIGNMENT((byte) 0x8B),
    FLOW_CONTROL((byte) 0x8C),
    TPDU_SIZE((byte) 0xC0),
    CALLING_TSAP((byte) 0xC1),
    CALLED_TSAP((byte) 0xC2),
    CHECKSUM((byte) 0xC3),
    VERSION((byte) 0xC4),
    SECURITY_PARAMETERS((byte) 0xC5),
    ADDITIONAL_OPTION_SELECTION((byte) 0xC6),
    ALTERNATIVE_PROTOCOL_CLASS((byte) 0xC7),
    DISCONNECT_ADDITIONAL_INFORMATION((byte) 0xE0),
    PREFERRED_MAX_PDU_SIZE((byte) 0xF0),
    INACTIVITY_TIMER((byte) 0xF2);

    private static final Map<Byte, ParameterCode> map;
    static {
        map = new HashMap<>();
        for (ParameterCode parameterCode : ParameterCode.values()) {
            map.put(parameterCode.code, parameterCode);
        }
    }

    private final byte code;

    ParameterCode(byte code) {
        this.code = code;
    }

    public byte getCode() {
        return code;
    }

    public static ParameterCode valueOf(byte code) {
        return map.get(code);
    }

}
