//Copyright (c) 2008, California Institute of Technology.
//ALL RIGHTS RESERVED. U.S. Government sponsorship acknowledged.
//
//$Id$

package org.apache.oodt.cas.workflow.policy;

//OODT imports
import org.apache.oodt.cas.workflow.structs.WorkflowCondition;

//JDK imports
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Vector;

/**
 * @author mattmann
 * @version $Revision$
 * 
 * <p>
 * A Jsp utility that processes a {@link PolicyAwareWorkflowTask} that is given
 * in the form of a HttpServletRequestWrapper#getParameterMap().
 * </p>.
 */
public class TaskRequestFactory implements TaskRequestProcessorMetKeys {

    private TaskRequestFactory() throws InstantiationException {
        throw new InstantiationException("Don't construct factories!");
    }

    /**
     * Processes a given {@link PolicyAwareWorkflowTask} and unmarshalls it out
     * of the given {@link Map}.
     * 
     * @param paramMap
     *            A parameter {@link Map}, generated by a call to
     *            {@link HttpServletRequestWrapper#getParameterMap()}.
     * @return An unmarshalled, {@link PolicyAwareWorkflowTask}.
     */
    public static PolicyAwareWorkflowTask getTask(Map paramMap) {
        PolicyAwareWorkflowTask task = new PolicyAwareWorkflowTask();
        task.setTaskName(getFirst((String[]) paramMap.get(TASK_NAME)));
        task.setTaskId(getFirst((String[]) paramMap.get(UPDATE_TASK_ID)));
        task.setPolicyDirPath(getFirst((String[]) paramMap
                .get(TASK_POLICY_DIR_PATH)));
        task.setTaskInstanceClassName(getFirst((String[]) paramMap
                .get(TASK_CLASS_NAME)));
        int reqMetFieldCount = Integer.parseInt(getFirst((String[]) paramMap
                .get(REQ_MET_FIELDS_COUNT)));
        if (reqMetFieldCount > 0) {
            List metFields = new Vector();

            for (int i = 0; i < reqMetFieldCount; i++) {
                metFields.add(getFirst((String[]) paramMap
                        .get(REQ_MET_FIELD_TAGBASE + (i))));
            }

            task.setRequiredMetFields(metFields);
        } else
            task.setRequiredMetFields(Collections.EMPTY_LIST);

        String[] selectedCondIds = (String[]) paramMap.get(TASK_CONDITION_IDS);

        if (selectedCondIds != null && selectedCondIds.length > 0) {
            List conditions = new Vector();
            for (int i = 0; i < selectedCondIds.length; i++) {
                WorkflowCondition cond = new WorkflowCondition();
                cond.setConditionId(selectedCondIds[i]);
                conditions.add(cond);
            }

            task.setConditions(conditions);
        } else
            task.setConditions(Collections.EMPTY_LIST);

        int numConfigProperties = Integer.parseInt(getFirst((String[]) paramMap
                .get(TASK_CONFIG_FIELD_COUNT)));
        if (numConfigProperties > 0) {
            EnvSavingConfiguration config = new EnvSavingConfiguration();
            for (int i = 0; i < numConfigProperties; i++) {
                String propName = getFirst((String[]) paramMap
                        .get(TASK_CONFIG_PROPNAME_BASE + (i)));
                String propVal = getFirst((String[]) paramMap
                        .get(TASK_CONFIG_PROPVAL_BASE + (i)));
                config.addConfigProperty(propName, propVal);
            }

            task.setTaskConfig(config);
        }

        return task;
    }

    /**
     * Merges the <code>newTask</code> {@link PolicyAwareWorkflowTask} into
     * the <code>orig</code> {@link PolicyAwareWorkflowTask}, taking care not
     * to overwrite the {@link EnvSavingConfiguration}, and only its
     * {@link EnvSavingConfiguration#getProperties()}.
     * 
     * @param newTask
     *            The new {@link PolicyAwareWorkflowTask}, whose values will
     *            overwrite those of <code>orig</code>.
     * @param orig
     *            The destination for the new values provided by
     *            <code>newTask</code>.
     */
    public static void merge(PolicyAwareWorkflowTask newTask,
            PolicyAwareWorkflowTask orig) {
        orig.setTaskId(newTask.getTaskId());
        orig.setTaskName(newTask.getTaskName());
        orig.setTaskInstanceClassName(newTask.getTaskInstanceClassName());
        orig.setPolicyDirPath(newTask.getPolicyDirPath());
        orig.setConditions(newTask.getConditions());
        orig.setRequiredMetFields(newTask.getRequiredMetFields());
        // only copy through properties, don't reset config, because
        // we need to save the orig envReplace info
        orig.getTaskConfig().getProperties().clear();
        orig.getTaskConfig().getProperties().putAll(
                newTask.getTaskConfig().getProperties());
    }

    private static String getFirst(String[] strArray) {
        if (strArray != null && strArray.length >= 1) {
            return strArray[0];
        } else
            return null;
    }

}
