/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.oodt.cas.filemgr.catalog.solr;

import java.util.List;

import org.apache.oodt.cas.filemgr.structs.Product;
import org.apache.oodt.cas.filemgr.structs.Reference;
import org.apache.oodt.cas.filemgr.structs.exceptions.CatalogException;
import org.apache.oodt.cas.metadata.Metadata;

/**
 * API for serializing CAS objects (Products, Metadata and References) into documents to be indexed by Solr,
 * and vice-versa deserializing Solr response documents into CAS objects.
 * 
 * Implementing classes are responsible for converting the CAS fields
 * into corresponding Solr fields as defined in their specific Solr schema.xml, 
 * and vice-versa for transforming the Solr fields into corresponding CAS fields.
 * 
 * @author Luca Cinquini
 *
 */
public interface ProductSerializer {
	
	/**
	 * Method to transform a CAS Product into one or more documents to be indexed by Solr.
	 * 
	 * @param product : the product to be indexed
	 * @param create: true to create a new product, false to update an existing one.
	 * @return
	 */
	List<String> serialize(Product product, boolean create);
	
	/**
	 * Method to transform a CAS Metadata object into one or more Solr "update" documents
	 * (i.e. documents containing attributes of an existing product).
	 * 
	 * @param productId : the product unique identifier
	 * @param metadata : the CAS metadata hash table
	 * @param replace : true to override the existing metadata values, false to add new values to the existing ones.
	 *                  Existing metadata fields not contained in the hash are left unchanged.
	 *                  Note: the special value Parameters.NULL is used to indicate that the corresponding key must be removed
	 *                  (as the Metadata object cannot store null or empty lists).
	 * @return
	 */
	List<String> serialize(String productId, Metadata metadata, boolean replace);
	
	/**
	 * Method to transform a CAS Product References object into one or more Solr "update" documents
	 * (i.e. documents containing attributes of an existing product).
	 * 
	 * @param productId
	 * @param rootReference : the product root reference, may be null
	 * @param references : list of product references, may be empty
	 * @param replace : true to replace the existing references, false to add new references to the existing ones
	 * @return
	 */
	List<String> serialize(String productId, Reference rootReference, List<Reference> references, boolean replace);
	
	/**
	 * Method to parse a full XML response document into a list of Product, References and Metadata objects.
	 * 
	 * @param xml : the Solr XML document to parse (input)
	 * @return
	 * @throws Exception
	 */
	QueryResponse deserialize(String xml) throws CatalogException;
	
	/**
	 * Mime type of the documents generated by the specific implementation (XML, JSON etc.)
	 * @return
	 */
	String getMimeType();

}