/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package org.apache.ode.bpel.engine;

import java.util.Map;
import java.util.Properties;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;

import javax.transaction.TransactionManager;

import junit.framework.TestCase;

import org.apache.geronimo.transaction.manager.GeronimoTransactionManager;
import org.apache.ode.bpel.dao.MessageExchangeDAO;
import org.apache.ode.bpel.engine.MyRoleMessageExchangeImpl.ResponseCallback;
import org.apache.ode.scheduler.simple.SimpleScheduler;
import org.jmock.Expectations;
import org.jmock.Mockery;

public class MyRoleMessageExchangeImplTest extends TestCase {
    Mockery context = new Mockery();
    MessageExchangeDAO mexDao;
    
    private TestMyRoleMessageExchangeImpl myRoleMexImpl;
    Contexts contexts;
    BpelEngineImpl engine;
    TransactionManager _txm;
    
    public void testResponseReceived() throws Exception {
        context.checking(new Expectations() {{
            exactly(3).of(mexDao).getCorrelationId(); will(returnValue("corrId"));
        }});
        
        final boolean[] responded = new boolean[1];
        myRoleMexImpl.callbacks().put("corrId", new ResponseCallback() {
            synchronized boolean responseReceived() {
                responded[0] = true;
                return true;
            }

            synchronized void waitResponse(long timeout) {
            }
        });
        
        _txm.begin();
        myRoleMexImpl.responseReceived();
        _txm.rollback();
        
        _txm.begin();
        myRoleMexImpl.responseReceived();
        _txm.rollback();
        
        _txm.begin();
        myRoleMexImpl.responseReceived();
        _txm.commit();
        
        assertTrue(responded[0]);
        context.assertIsSatisfied();
    }
    
    public void testResponseTimeout() throws Exception {
        context.checking(new Expectations() {{
            atLeast(1).of(mexDao).getCorrelationId(); will(returnValue("corrId"));
        }});

        myRoleMexImpl.callbacks().put("corrId", new MyRoleMessageExchangeImpl.ResponseCallback());

        _txm.begin();
        myRoleMexImpl.responseReceived();
        _txm.rollback();

        try {
            new MyRoleMessageExchangeImpl.ResponseFuture("corrId").get(10, TimeUnit.MILLISECONDS);
            fail("Should throw a TimeoutException!!");
        } catch( TimeoutException te ) {}
        context.assertIsSatisfied();
    }
    
    protected void setUp() throws Exception {
        _txm = new GeronimoTransactionManager();
        
        mexDao = context.mock(MessageExchangeDAO.class);
        SimpleScheduler scheduler = new SimpleScheduler("node", null, new Properties());
        scheduler.setTransactionManager(_txm);
        
        contexts = new Contexts();
        contexts.scheduler = scheduler;
        engine = new BpelEngineImpl(contexts);

        myRoleMexImpl = new TestMyRoleMessageExchangeImpl();
        context.assertIsSatisfied();
    }

    class TestMyRoleMessageExchangeImpl extends MyRoleMessageExchangeImpl {
        public TestMyRoleMessageExchangeImpl() {
            super(null, engine, mexDao);
        }
        
        public Map<String, ResponseCallback> callbacks() {
            return _waitingCallbacks;
        }
    }
}
