/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package org.netbeans.modules.cloud.oracle.adm;

import com.oracle.bmc.adm.ApplicationDependencyManagementClient;
import com.oracle.bmc.adm.model.KnowledgeBase;
import com.oracle.bmc.adm.requests.GetKnowledgeBaseRequest;
import com.oracle.bmc.adm.responses.GetKnowledgeBaseResponse;
import com.oracle.bmc.model.BmcException;
import java.util.Collection;
import java.util.Date;
import java.util.concurrent.CompletableFuture;
import org.netbeans.api.project.Project;
import org.netbeans.api.project.ProjectUtils;
import org.netbeans.modules.cloud.oracle.OCIManager;
import org.netbeans.modules.cloud.oracle.items.OCID;
import org.netbeans.spi.project.LookupProvider.Registration.ProjectType;
import org.netbeans.spi.project.ProjectServiceProvider;
import org.openide.util.NbBundle;
import org.openide.util.RequestProcessor;

/**
 * This is a pseudo API that allows an integration LSP module to 
 * configure and work with the vulnerability feature. Allows to temporarily configure a knowledge
 * base for a given project. The setting is kept just in memory.
 * 
 * @author sdedic
 */
@ProjectServiceProvider(service = ProjectVulnerability.class, projectTypes = {
    @ProjectType(id = "org-netbeans-modules-maven"),
    @ProjectType(id = "org-netbeans-modules-gradle")
})
public final class ProjectVulnerability {
    private final RequestProcessor AUDIT_PROCESSOR = new RequestProcessor(ProjectVulnerability.class.getName(), 4);
    private final RequestProcessor CALL_PROCESSOR = new RequestProcessor(ProjectVulnerability.class);
    
    private final Project project;
    
    // @GuardedBy(this)
    private KnowledgeBaseItem knowledgeBaseItem = null;

    public ProjectVulnerability(Project project) {
        this.project = project;
    }
    
    public KnowledgeBaseItem getProjectKnowledgeBase() {
        synchronized (this) {
            return knowledgeBaseItem;
        }
    }
    
    @NbBundle.Messages({
        "# {0} - project name",
        "MSG_CreatingAuditFailed=Creating Vulnerablity audit for project {0} failed.",
    })
    public CompletableFuture<String> runProjectAudit(KnowledgeBaseItem item, boolean force, boolean runIfNoReport) {
        if (item != null) {
            setProjectKnowledgeBase(item);
        }
        CompletableFuture<String> result = new CompletableFuture<>();
        AUDIT_PROCESSOR.post(() -> {
            try {
                result.complete(VulnerabilityWorker.getInstance().findVulnerability(project, force, runIfNoReport));
            } catch (ThreadDeath x) {
                throw x;
            } catch (AuditException ex) {
                final String projectDisplayName = ProjectUtils.getInformation(project).getDisplayName();
                ErrorUtils.processError(ex, Bundle.MSG_CreatingAuditFailed(projectDisplayName));
                result.completeExceptionally(ex);
            } catch (Exception | Error e) {
                result.completeExceptionally(e);
            }
        });
        // PENDING: handle CF.cancel() by cancelling the WorkRequest
        return result;
    }
    
    public CompletableFuture<KnowledgeBaseItem> findKnowledgeBase(String compartmentId, String knowledgeBaseId) {
        CompletableFuture<KnowledgeBaseItem> result = new CompletableFuture<>();
        CALL_PROCESSOR.post(() -> {
            try ( ApplicationDependencyManagementClient client 
                    = new ApplicationDependencyManagementClient(OCIManager.getDefault().getConfigProvider())) {

                GetKnowledgeBaseRequest request = GetKnowledgeBaseRequest.builder()
                        .knowledgeBaseId(knowledgeBaseId).
                        build();

                GetKnowledgeBaseResponse response = client.getKnowledgeBase(request);
                KnowledgeBase p = response.getKnowledgeBase();
                result.complete(new KnowledgeBaseItemProxy(
                        OCID.of(p.getId(), "KnowledgeBase"), // NOI18N 
                        p.getCompartmentId(),
                        p.getDisplayName(), p.getTimeUpdated())
                );
            } catch (ThreadDeath x) {
                throw x;
            } catch (BmcException ex) {
                result.completeExceptionally(new AuditException(ex.getStatusCode(), ex.getOpcRequestId(), ex.getMessage(), ex));
            } catch (Exception | Error e) {
                result.completeExceptionally(e);
            }
        });
        return result;
    }
    
    
    /**
     * This special item just fires a refresh in any "real" KnowledgeBaseItems that may have been created
     * + presented in the IDE UI.
     */
    static class KnowledgeBaseItemProxy extends KnowledgeBaseItem {
        public KnowledgeBaseItemProxy(OCID id, String compartmentId, String displayName, Date timeUpdated) {
            super(id, compartmentId, displayName, timeUpdated);
        }

        @Override
        void registerItem() {
            // do not register self.
        }

        @Override
        public void refresh() {
            super.refresh();
            Collection<KnowledgeBaseItem> delegates = findKnownInstances(getKey());
            for (KnowledgeBaseItem kbi : delegates) {
                kbi.refresh();
            }
        }
    }
    
    public KnowledgeBaseItem setProjectKnowledgeBase(KnowledgeBaseItem item) {
        synchronized (this) {
            return knowledgeBaseItem = item;
        }
    }
}
