/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package org.netbeans.modules.gradle.customizer;

import javax.swing.DefaultComboBoxModel;
import org.netbeans.api.project.Project;
import org.netbeans.modules.gradle.ProjectTrust;
import org.netbeans.modules.gradle.api.GradleBaseProject;
import org.openide.util.NbBundle;
import org.openide.util.NbBundle.Messages;

import static org.netbeans.modules.gradle.customizer.GradleExecutionPanel.TrustLevel.*;

/**
 *
 * @author lkishalmi
 */
@Messages({
    "GRADLE_TRUST_MSG=<html><p>Executing Gradle can be potentially un-safe as it "
            + "allows arbitrary code execution.</p><p></p>"
            + "<p>By trusting this project, and with that all its subprojects, "
            + "you entitle NetBeans to invoke Gradle to load project details "
            + "without further confirmation.</p><p></p>"
            + "<p>Invoking any build related actions, would mark this project "
            + "automatically trusted.</p>",
})
public class GradleExecutionPanel extends javax.swing.JPanel {
    
    enum TrustLevel {
        PERMANENT,
        TEMPORARY,
        NONE;
        
        @Override
        public String toString() {
            return NbBundle.getMessage(GradleExecutionPanel.class, "GradleExecutionPanel.cbTrustLevel." + name()); // NOI18N
        }
        
    }

    Project project;

    /**
     * Creates new form GradleExecutionPanel
     */
    public GradleExecutionPanel() {
        initComponents();
        lbTrustTerms.setText(Bundle.GRADLE_TRUST_MSG());
    }

    public GradleExecutionPanel(Project project) {
        this();
        this.project = project;
        GradleBaseProject gbp = GradleBaseProject.get(project);
        if (gbp != null) {
            lbReadOnly.setVisible(!gbp.isRoot());
            lbTrustLevel.setEnabled(gbp.isRoot());
            cbTrustLevel.setEnabled(gbp.isRoot());
            lbTrustTerms.setEnabled(gbp.isRoot());
            
            cbTrustLevel.setModel(new DefaultComboBoxModel<>(TrustLevel.values()));

            if (ProjectTrust.getDefault().isTrustedPermanently(project)) {
                cbTrustLevel.setSelectedItem(PERMANENT);
            } else if (ProjectTrust.getDefault().isTrusted(project)) {
                cbTrustLevel.setSelectedItem(TEMPORARY);
            } else {
                cbTrustLevel.setSelectedItem(NONE);
            }
        }
    }

    /**
     * This method is called from within the constructor to initialize the form.
     * WARNING: Do NOT modify this code. The content of this method is always
     * regenerated by the Form Editor.
     */
    @SuppressWarnings("unchecked")
    // <editor-fold defaultstate="collapsed" desc="Generated Code">//GEN-BEGIN:initComponents
    private void initComponents() {

        lbTrustTerms = new javax.swing.JLabel();
        lbReadOnly = new javax.swing.JLabel();
        cbTrustLevel = new javax.swing.JComboBox<>();
        lbTrustLevel = new javax.swing.JLabel();

        lbTrustTerms.setVerticalAlignment(javax.swing.SwingConstants.TOP);

        lbReadOnly.setIcon(new javax.swing.ImageIcon(getClass().getResource("/org/netbeans/modules/gradle/resources/info.png"))); // NOI18N
        org.openide.awt.Mnemonics.setLocalizedText(lbReadOnly, org.openide.util.NbBundle.getMessage(GradleExecutionPanel.class, "GradleExecutionPanel.lbReadOnly.text")); // NOI18N

        org.openide.awt.Mnemonics.setLocalizedText(lbTrustLevel, org.openide.util.NbBundle.getMessage(GradleExecutionPanel.class, "GradleExecutionPanel.lbTrustLevel.text")); // NOI18N

        javax.swing.GroupLayout layout = new javax.swing.GroupLayout(this);
        this.setLayout(layout);
        layout.setHorizontalGroup(
            layout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
            .addGroup(layout.createSequentialGroup()
                .addContainerGap()
                .addGroup(layout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
                    .addGroup(layout.createSequentialGroup()
                        .addComponent(lbReadOnly)
                        .addGap(0, 0, Short.MAX_VALUE))
                    .addGroup(layout.createSequentialGroup()
                        .addGap(21, 21, 21)
                        .addGroup(layout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
                            .addGroup(layout.createSequentialGroup()
                                .addComponent(lbTrustLevel)
                                .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.UNRELATED)
                                .addComponent(cbTrustLevel, 0, javax.swing.GroupLayout.DEFAULT_SIZE, Short.MAX_VALUE))
                            .addComponent(lbTrustTerms, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, Short.MAX_VALUE))))
                .addContainerGap())
        );
        layout.setVerticalGroup(
            layout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
            .addGroup(layout.createSequentialGroup()
                .addGap(5, 5, 5)
                .addGroup(layout.createParallelGroup(javax.swing.GroupLayout.Alignment.BASELINE)
                    .addComponent(cbTrustLevel, javax.swing.GroupLayout.PREFERRED_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.PREFERRED_SIZE)
                    .addComponent(lbTrustLevel))
                .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.RELATED)
                .addComponent(lbTrustTerms, javax.swing.GroupLayout.PREFERRED_SIZE, 234, javax.swing.GroupLayout.PREFERRED_SIZE)
                .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.RELATED, 8, Short.MAX_VALUE)
                .addComponent(lbReadOnly)
                .addContainerGap())
        );
    }// </editor-fold>//GEN-END:initComponents

    // Variables declaration - do not modify//GEN-BEGIN:variables
    private javax.swing.JComboBox<TrustLevel> cbTrustLevel;
    private javax.swing.JLabel lbReadOnly;
    private javax.swing.JLabel lbTrustLevel;
    private javax.swing.JLabel lbTrustTerms;
    // End of variables declaration//GEN-END:variables

    void save() {
        if (project != null) {
            TrustLevel v = (TrustLevel)cbTrustLevel.getSelectedItem();
            if (v == null) {
                v = NONE;
            }
            switch (v) {
                case NONE:
                    ProjectTrust.getDefault().distrustProject(project);
                    break;
                    
                case PERMANENT:
                    ProjectTrust.getDefault().trustProject(project, true);
                    break;
                    
                case TEMPORARY:
                    if (ProjectTrust.getDefault().isTrustedPermanently(project)) {
                        ProjectTrust.getDefault().distrustProject(project);
                    }
                    ProjectTrust.getDefault().trustProject(project, false);
                    break;
            }
        }
    }
}
