package org.apache.maven.report.projectinfo;

/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */

import java.io.File;
import java.io.IOException;
import java.util.Locale;

import org.apache.maven.plugin.LegacySupport;
import org.apache.maven.plugin.testing.AbstractMojoTestCase;
import org.apache.maven.plugin.testing.ArtifactStubFactory;
import org.apache.maven.plugin.testing.stubs.MavenProjectStub;
import org.apache.maven.project.DefaultProjectBuildingRequest;
import org.apache.maven.project.MavenProject;
import org.apache.maven.project.ProjectBuilder;
import org.apache.maven.project.ProjectBuildingRequest;
import org.apache.maven.report.projectinfo.stubs.DependencyArtifactStubFactory;
import org.codehaus.plexus.i18n.I18N;
import org.codehaus.plexus.util.StringUtils;
import org.eclipse.aether.internal.impl.SimpleLocalRepositoryManagerFactory;
import org.eclipse.aether.DefaultRepositorySystemSession;
import org.eclipse.aether.repository.LocalRepository;

/**
 * Abstract class to test reports generation with <a href="http://www.httpunit.org/">HTTPUnit</a> framework.
 *
 * @author Edwin Punzalan
 * @author <a href="mailto:vincent.siveton@gmail.com">Vincent Siveton</a>
 * @version $Id$
 */
public abstract class AbstractProjectInfoTestCase
    extends AbstractMojoTestCase
{
    private ArtifactStubFactory artifactStubFactory;
    
    /**
     * The default locale is English.
     */
    protected static final Locale DEFAULT_LOCALE = Locale.ENGLISH;

    /**
     * The current project to be test.
     */
    private MavenProject testMavenProject;

    /**
     * The I18N plexus component.
     */
    private I18N i18n;

    @Override
    protected void setUp()
        throws Exception
    {
        // required for mojo lookups to work
        super.setUp();

        i18n = getContainer().lookup( I18N.class );
        setVariableValueToObject( i18n, "defaultBundleName", "project-info-reports" );

        artifactStubFactory = new DependencyArtifactStubFactory( getTestFile( "target" ), true, false );
        artifactStubFactory.getWorkingDir().mkdirs();
        
        // Set the default Locale
        Locale.setDefault( DEFAULT_LOCALE );
    }

    @Override
    protected void tearDown()
        throws Exception
    {
        super.tearDown();
    }

    /**
     * Gets a trimmed String for the given key from the resource bundle defined by Plexus.
     *
     * @param key the key for the desired string
     * @return the string for the given key
     */
    protected String getString( String key )
    {
        if ( StringUtils.isEmpty( key ) )
        {
            throw new IllegalArgumentException( "The key cannot be empty" );
        }

        return i18n.getString( key, Locale.getDefault() ).trim();
    }

    /**
     * Gets a fully qualified title as generated by Doxia 1.6
     *
     * @param name the name to prepare
     * @param title the title to prepare
     * @return the prepared title as per Doxia 1.6
     * @since 2.8
     */
    protected String prepareTitle( String name, String title )
    {
        if ( StringUtils.isEmpty( name ) )
        {
            throw new IllegalArgumentException( "The name cannot be empty" );
        }

        if ( StringUtils.isEmpty( title ) )
        {
            throw new IllegalArgumentException( "The title cannot be empty" );
        }

        return String.format( "%s \u2013 %s", name, title );
    }

    /**
     * Get the current Maven project
     *
     * @return the maven project
     */
    protected MavenProject getTestMavenProject()
    {
        return testMavenProject;
    }

    /**
     * Get the generated report as file in the test maven project.
     *
     * @param name the name of the report.
     * @return the generated report as file
     * @throws IOException if the return file doesnt exist
     */
    protected File getGeneratedReport( String name )
        throws IOException
    {
        String outputDirectory = getBasedir() + "/target/test-harness/" + getTestMavenProject().getArtifactId();

        File report = new File( outputDirectory, name );
        if ( !report.exists() )
        {
            throw new IOException( "File not found. Attempted :" + report );
        }

        return report;
    }

    /**
     * Generate the report and return the generated file
     *
     * @param goal the mojo goal.
     * @param pluginXml the name of the xml file in "src/test/resources/plugin-configs/".
     * @return the generated HTML file
     * @throws Exception if any
     */
    protected File generateReport( String goal, String pluginXml )
        throws Exception
    {
        File pluginXmlFile = new File( getBasedir(), "src/test/resources/plugin-configs/" + pluginXml );
        AbstractProjectInfoReport mojo  = createReportMojo( goal, pluginXmlFile );
        return generateReport( mojo, pluginXmlFile );
    }

    protected AbstractProjectInfoReport createReportMojo( String goal, File pluginXmlFile )
        throws Exception
    {
        AbstractProjectInfoReport mojo = (AbstractProjectInfoReport) lookupMojo( goal, pluginXmlFile );
        assertNotNull( "Mojo found.", mojo );
        
        LegacySupport legacySupport = lookup( LegacySupport.class );
        legacySupport.setSession( newMavenSession( new MavenProjectStub() ) );
        DefaultRepositorySystemSession repoSession =
            (DefaultRepositorySystemSession) legacySupport.getRepositorySession();
        repoSession.setLocalRepositoryManager( new SimpleLocalRepositoryManagerFactory().newInstance( repoSession, new LocalRepository( artifactStubFactory.getWorkingDir() ) ) );

        setVariableValueToObject( mojo, "session", legacySupport.getSession() );
        setVariableValueToObject( mojo, "remoteRepositories", mojo.getProject().getRemoteArtifactRepositories() );
        setVariableValueToObject( mojo, "pluginRepositories", mojo.getProject().getPluginArtifactRepositories() );
        return mojo;
    }

    protected File generateReport( AbstractProjectInfoReport mojo, File pluginXmlFile )
        throws Exception
    {
        mojo.execute();

        ProjectBuilder builder = lookup( ProjectBuilder.class );
        
        ProjectBuildingRequest buildingRequest = new DefaultProjectBuildingRequest();
        buildingRequest.setRepositorySession( lookup( LegacySupport.class ).getRepositorySession() );

        assertNotNull( "Local repository", mojo.localRepository );
        testMavenProject = builder.build( pluginXmlFile, buildingRequest ).getProject();

        File outputDir = mojo.getReportOutputDirectory();
        String filename = mojo.getOutputName() + ".html";

        return new File( outputDir, filename );
    }


}
