package org.apache.maven.plugins.ear.it;

/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *  http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */

import java.io.File;
import java.io.FileInputStream;
import java.util.Properties;
import java.util.jar.JarFile;
import java.util.jar.Manifest;

import org.apache.maven.shared.utils.io.IOUtil;
import org.codehaus.plexus.util.FileUtils;
import org.codehaus.plexus.util.ReaderFactory;

/**
 * @author <a href="snicoll@apache.org">Stephane Nicoll</a>
 */
public class EarMojoIT
    extends AbstractEarPluginIT
{

    /**
     * Builds an EAR with a single EJB and no configuration.
     * @throws Exception in case of an error.
     */
    public void testProject001()
        throws Exception
    {
        doTestProject( "project-001", new String[] { "eartest-ejb-sample-one-1.0.jar" } );
    }

    /**
     * Builds an EAR with a customized artifact location and a customized artifact name.
     * @throws Exception in case of an error.
     */
    public void testProject002()
        throws Exception
    {
        doTestProject( "project-002",
                       new String[] { "APP-INF/lib/eartest-ejb-sample-one-1.0.jar", "ejb-sample-two.jar" } );
    }

    /**
     * Builds an EAR with a default bundle directory for <tt>java</tt> modules.
     * @throws Exception in case of an error.
     */
    public void testProject003()
        throws Exception
    {
        doTestProject( "project-003", new String[] { "eartest-ejb-sample-one-1.0.jar",
            "APP-INF/lib/eartest-jar-sample-one-1.0.jar", "APP-INF/lib/eartest-jar-sample-two-1.0.jar" } );
    }

    /**
     * Builds an EAR with a default bundle directory for _java_ modules and a custom location overriding the default.
     * @throws Exception in case of an error.
     */
    public void testProject004()
        throws Exception
    {
        doTestProject( "project-004", new String[] { "eartest-ejb-sample-one-1.0.jar", "eartest-jar-sample-one-1.0.jar",
            "APP-INF/lib/eartest-jar-sample-two-1.0.jar" } );
    }

    /**
     * Builds an EAR with a custom URI.
     * @throws Exception in case of an error.
     */
    public void testProject005()
        throws Exception
    {
        doTestProject( "project-005", new String[] { "eartest-ejb-sample-one-1.0.jar", "libs/another-name.jar" } );
    }

    /**
     * Builds an EAR with an excluded module.
     * @throws Exception in case of an error.
     */
    public void testProject006()
        throws Exception
    {
        doTestProject( "project-006",
                       new String[] { "eartest-ejb-sample-one-1.0.jar", "eartest-jar-sample-two-1.0.jar" } );
    }

    /**
     * Builds an EAR with a classified artifact and no extra configuration.
     * @throws Exception in case of an error.
     */
    public void testProject007()
        throws Exception
    {
        doTestProject( "project-007", new String[] { "eartest-ejb-sample-one-1.0-classified.jar" } );
    }

    /**
     * Builds an EAR with deployment descriptor configuration for J2EE 1.3.
     * @throws Exception in case of an error.
     */
    public void testProject008()
        throws Exception
    {
        doTestProject( "project-008", new String[] { "eartest-ejb-sample-one-1.0.jar" } );
    }

    /**
     * Builds an EAR with deployment descriptor configuration for J2EE 1.4.
     * @throws Exception in case of an error.
     */
    public void testProject009()
        throws Exception
    {
        doTestProject( "project-009", new String[] { "eartest-ejb-sample-one-1.0.jar" } );
    }

    /**
     * Builds an EAR with deployment descriptor configuration for Java EE 5.
     * @throws Exception in case of an error.
     */
    public void testProject010()
        throws Exception
    {
        doTestProject( "project-010", new String[] { "eartest-ejb-sample-one-1.0.jar" } );
    }

    /**
     * Builds an EAR and make sure that deployment descriptor default settings are applied.
     * @throws Exception in case of an error.
     */
    public void testProject011()
        throws Exception
    {
        doTestProject( "project-011", new String[] { "eartest-ejb-sample-one-1.0.jar" } );
    }

    /**
     * Builds an EAR and make sure that EAR resources are bundled within the EAR.
     * @throws Exception in case of an error.
     */
    public void testProject012()
        throws Exception
    {
        doTestProject( "project-012", new String[] { "README.txt", "LICENSE.txt", "eartest-ejb-sample-one-1.0.jar" } );
    }

    /**
     * Builds an EAR and make sure that EAR resources in a customized resources directory are bundled within the EAR.
     * @throws Exception in case of an error.
     */
    public void testProject013()
        throws Exception
    {
        doTestProject( "project-013", new String[] { "README.txt", "LICENSE.txt", "eartest-ejb-sample-one-1.0.jar" } );
    }

    /**
     * Builds an EAR and make sure that EAR resources are bundled within the EAR using includes and excludes.
     * @throws Exception in case of an error.
     */
    public void testProject014()
        throws Exception
    {
        doTestProject( "project-014", new String[] { "LICENSE.txt", "eartest-ejb-sample-one-1.0.jar" } );
    }

    /**
     * Builds an EAR and make sure that default manifest is taken into account.
     * @throws Exception in case of an error.
     */
    public void testProject015()
        throws Exception
    {
        final File baseDir = doTestProject( "project-015", new String[] { "eartest-ejb-sample-one-1.0.jar" } );
        final File expectedManifest = new File( baseDir, "src/main/application/META-INF/MANIFEST.MF" );
        final File actualManifest = new File( getEarDirectory( baseDir, "project-015" ), "META-INF/MANIFEST.MF" );
        assertTrue( "Manifest was not copied", actualManifest.exists() );
        assertTrue( FileUtils.contentEquals( expectedManifest, actualManifest ) );
    }

    /**
     * Builds an EAR and make sure that custom manifest is taken into account.
     * @throws Exception in case of an error.
     */
    public void testProject016()
        throws Exception
    {
        final File baseDir = doTestProject( "project-016", new String[] { "eartest-ejb-sample-one-1.0.jar" } );

        final File targetFolder = new File( baseDir, "target" );
        final File createdEarFile = new File( targetFolder, "maven-ear-plugin-test-project-016-99.0.ear" );

        final File sourceManifestFile = new File( baseDir, "src/main/ear/MANIFEST.MF" );

        JarFile jarFile = new JarFile( createdEarFile );
        Manifest manifestFromCreatedEARFile = jarFile.getManifest();
        jarFile.close();

        Manifest sourceManifest = new Manifest( new FileInputStream( sourceManifestFile ) );

        assertTrue( "There are differences in the manifest.", sourceManifest.equals( manifestFromCreatedEARFile ) );
    }

    /**
     * Builds an EAR and make sure that custom application.xml is taken into account.
     * @throws Exception in case of an error.
     */
    public void testProject017()
        throws Exception
    {
        doTestProject( "project-017", new String[] { "eartest-ejb-sample-one-1.0.jar" } );
    }

    /**
     * Builds an EAR with a custom final name.
     * @throws Exception in case of an error.
     */
    public void testProject018()
        throws Exception
    {
        final File baseDir = executeMojo( "project-018", new Properties() );
        final File expectedFile = new File( baseDir, "target/my-custom-file.ear" );
        assertTrue( "EAR archive not found", expectedFile.exists() );
    }

    /**
     * Builds an EAR with unpacked archives using the unpackTypes.
     * @throws Exception in case of an error.
     */
    public void testProject019()
        throws Exception
    {
        doTestProject( "project-019", new String[] { "eartest-ejb-sample-one-1.0.jar", "eartest-sar-sample-one-1.0.sar",
            "eartest-jar-sample-one-1.0.jar" }, new boolean[] { false, true, true } );
    }

    /**
     * Builds an EAR with unpacked archives using the unpack module attribute.
     * @throws Exception in case of an error.
     */
    public void testProject020()
        throws Exception
    {
        doTestProject( "project-020", new String[] { "eartest-ejb-sample-one-1.0.jar", "eartest-sar-sample-one-1.0.sar",
            "eartest-jar-sample-one-1.0.jar" }, new boolean[] { true, false, false } );
    }

    /**
     * Builds an EAR with unpacked archives using both unpackTypes and the unpack module attribute.
     * @throws Exception in case of an error.
     */
    public void testProject021()
        throws Exception
    {
        doTestProject( "project-021",
                       new String[] { "eartest-ejb-sample-one-1.0.jar", "eartest-ejb-sample-two-1.0.jar",
                           "eartest-sar-sample-one-1.0.sar", "eartest-jar-sample-one-1.0.jar",
                           "eartest-jar-sample-two-1.0.jar" },
                       new boolean[] { false, true, false, false, true } );
    }

    /**
     * Builds an EAR with a classifier.
     * @throws Exception in case of an error.
     */
    public void testProject022()
        throws Exception
    {
        final File baseDir = executeMojo( "project-022", new Properties() );
        final File expectedFile = new File( baseDir, "target/maven-ear-plugin-test-project-022-99.0-myclassifier.ear" );
        assertTrue( "EAR archive not found", expectedFile.exists() );
    }

    /**
     * Builds an EAR and make sure that a single classified dependency is detected without specifying the classifier.
     * @throws Exception in case of an error.
     */
    public void testProject023()
        throws Exception
    {
        doTestProject( "project-023",
                       new String[] { "eartest-ejb-sample-one-1.0-classified.jar", "eartest-ejb-sample-two-1.0.jar" },
                       new boolean[] { true, false } );
    }

    /**
     * Builds an EAR and make sure that a single classified dependency is detected when specifying the classifier.
     * @throws Exception in case of an error.
     */
    public void testProject024()
        throws Exception
    {
        doTestProject( "project-024",
                       new String[] { "eartest-ejb-sample-one-1.0-classified.jar", "eartest-ejb-sample-two-1.0.jar" },
                       new boolean[] { true, false } );
    }

    /**
     * Builds an EAR and make sure that a classified dependency with mutiple candidates is detected when specifying the
     * classifier.
     * @throws Exception in case of an error.
     */
    public void testProject025()
        throws Exception
    {
        doTestProject( "project-025",
                       new String[] { "eartest-ejb-sample-one-1.0-classified.jar", "eartest-ejb-sample-one-1.0.jar" },
                       new boolean[] { true, false } );
    }

    /**
     * Builds an EAR and make sure that the build fails if a unclassifed module configuration with mutiple candidates is
     * specified.
     * @throws Exception in case of an error.
     */
    public void testProject026()
        throws Exception
    {
        final File baseDir = executeMojo( "project-026", new Properties(), false );
        // Stupido, checks that the ear archive is not there
        assertFalse( "Execution should have failed", getEarArchive( baseDir, "project-026" ).exists() );
    }

    /**
     * Builds an EAR and make sure that provided dependencies are not included in the EAR.
     * @throws Exception in case of an error.
     */
    public void testProject027()
        throws Exception
    {
        doTestProject( "project-027", new String[] { "eartest-ejb-sample-one-1.0.jar" } );
    }

    /**
     * Builds an EAR and make sure that test dependencies are not included in the EAR.
     * @throws Exception in case of an error.
     */
    public void testProject028()
        throws Exception
    {
        doTestProject( "project-028", new String[] { "eartest-ejb-sample-one-1.0.jar" } );
    }

    /**
     * Builds an EAR and make sure that system dependencies are not included in the EAR.
     * @throws Exception in case of an error.
     */
    public void testProject029()
        throws Exception
    {
        doTestProject( "project-029", new String[] { "eartest-ejb-sample-one-1.0.jar" } );
    }

    /**
     * Builds an EAR and make sure that ejb-client dependencies are detected and not added by default in the generated
     * application.xml.
     * @throws Exception in case of an error.
     */
    public void testProject030()
        throws Exception
    {
        doTestProject( "project-030",
                       new String[] { "eartest-ejb-sample-one-1.0.jar", "eartest-ejb-sample-two-1.0-client.jar" } );
    }

    /**
     * Builds an EAR with a Jboss 4 configuration specifying the security domain and the unauthenticated-principal to
     * use.
     * @throws Exception in case of an error.
     */
    public void testProject031()
        throws Exception
    {
        doTestProject( "project-031",
                       new String[] { "eartest-ejb-sample-one-1.0.jar", "eartest-ejb-sample-two-1.0.jar" } );
    }

    /**
     * Builds an EAR with a Jboss 3.2 configuration specifying the jmx-name to use.
     * @throws Exception in case of an error.
     */
    public void testProject032()
        throws Exception
    {
        doTestProject( "project-032",
                       new String[] { "eartest-ejb-sample-one-1.0.jar", "eartest-ejb-sample-two-1.0.jar" } );
    }

    /**
     * Builds an EAR with a Jboss 4 configuration and Jboss specific modules.
     * @throws Exception in case of an error.
     */
    public void testProject033()
        throws Exception
    {
        doTestProject( "project-033", new String[] { "eartest-ejb-sample-one-1.0.jar", "eartest-ejb-sample-two-1.0.jar",
            "eartest-sar-sample-one-1.0.sar", "eartest-har-sample-one-1.0.har" } );
    }

    /**
     * Builds an EAR with custom security settings.
     * @throws Exception in case of an error.
     */
    public void testProject034()
        throws Exception
    {
        doTestProject( "project-034",
                       new String[] { "eartest-ejb-sample-one-1.0.jar", "eartest-ejb-sample-two-1.0.jar" } );
    }

    /**
     * Builds an EAR with a full filename mapping and make sure that custom locations are not overridden.
     * @throws Exception in case of an error.
     */
    public void testProject035()
        throws Exception
    {
        doTestProject( "project-035",
                       new String[] { "foo/eartest-ejb-sample-one-1.0.jar", "eartest-ejb-sample-two-1.0.jar",
                           "libs/eartest-jar-sample-one-1.0.jar", "libs/eartest-jar-sample-two-1.0.jar",
                           "sar-sample-one.sar" } );
    }

    /**
     * Builds an EAR with a full filename mapping and make sure that groupIds with dots are replaced by dashes in
     * filenames.
     * @throws Exception in case of an error.
     */
    public void testProject036()
        throws Exception
    {
        doTestProject( "project-036",
                       new String[] { "foo/eartest-ejb-sample-one-1.0.jar", "eartest-ejb-sample-two-1.0.jar",
                           "com.foo.bar-ejb-sample-one-1.0.jar", "com.foo.bar-ejb-sample-two-1.0.jar",
                           "libs/eartest-jar-sample-one-1.0.jar", "libs/eartest-jar-sample-two-1.0.jar",
                           "sar-sample-one.sar" } );
    }

    /**
     * Builds an EAR and make sure that ejb-client dependencies are detected and added in the generated application.xml
     * if includeInApplicationXml is set.
     * @throws Exception in case of an error.
     */
    public void testProject037()
        throws Exception
    {
        doTestProject( "project-037", new String[] { "eartest-ejb-sample-one-1.0.jar", "eartest-ejb-sample-two-1.0-client.jar" } );
    }

    /**
     * Builds an EAR and make sure that a non-classified dependency with mutiple candidates is detected when specifying
     * the mainArtifactId as classifier.
     * @throws Exception in case of an error.
     */
    public void testProject038()
        throws Exception
    {
        doTestProject( "project-038", new String[] { "eartest-ejb-sample-one-1.0-classified.jar", "eartest-ejb-sample-one-1.0.jar" },
                       new boolean[] { false, true } );
    }

    /**
     * Builds an EAR with a Jboss 4 configuration specifying specifying the loader repository to use.
     * @throws Exception in case of an error.
     */
    public void testProject039()
        throws Exception
    {
        doTestProject( "project-039", new String[] { "eartest-ejb-sample-one-1.0.jar", "eartest-ejb-sample-two-1.0.jar" } );
    }

    /**
     * Builds an EAR with deployment descriptor configuration for Java EE 5 and an alternative deployment descriptor.
     * @throws Exception in case of an error.
     */
    public void testProject040()
        throws Exception
    {
        doTestProject( "project-040", new String[] { "eartest-ejb-sample-one-1.0.jar" } );
    }

    /**
     * Builds an EAR with a Jboss 4.2 configuration specifying the module order to use.
     * @throws Exception in case of an error.
     */
    public void testProject041()
        throws Exception
    {
        doTestProject( "project-041", new String[] { "eartest-ejb-sample-one-1.0.jar", "eartest-ejb-sample-two-1.0.jar" } );
    }

    /**
     * Builds an EAR with a Jboss 4.2 configuration specifying a datasource to add.
     * @throws Exception in case of an error.
     */
    public void testProject042()
        throws Exception
    {
        doTestProject( "project-042", new String[] { "eartest-ejb-sample-one-1.0.jar", "eartest-ejb-sample-two-1.0.jar" } );
    }

    /**
     * Builds an EAR with a custom descriptor location (generatedDescriptorLocation setting).
     * @throws Exception in case of an error.
     */
    public void testProject043()
        throws Exception
    {
        final File baseDir = doTestProject( "project-043", new String[] { "eartest-ejb-sample-one-1.0.jar" } );
        final File expectedApplicationXml = new File( baseDir, "target/custom-descriptor-dir/application.xml" );
        assertTrue( "Application.xml file not found", expectedApplicationXml.exists() );
        assertFalse( "Application.xml file should not be empty", expectedApplicationXml.length() == 0 );
    }

    /**
     * Builds an EAR with a custom library-directory.
     * @throws Exception in case of an error.
     */
    public void testProject044()
        throws Exception
    {
        doTestProject( "project-044", new String[] { "eartest-ejb-sample-one-1.0.jar", "myLibs/eartest-jar-sample-one-1.0.jar" } );
    }

    /**
     * Builds an EAR and filter the content of the sources directory.
     * @throws Exception in case of an error.
     */
    public void testProject045()
        throws Exception
    {
        final File baseDir = doTestProject( "project-045", new String[] { "README.txt", "eartest-ejb-sample-one-1.0.jar" } );
        final File actualReadme = new File( getEarDirectory( baseDir, "project-045" ), "README.txt" );
        final String content = IOUtil.toString( ReaderFactory.newReader( actualReadme, "UTF-8" ) );
        assertTrue( "application name and version was not filtered properly", content.contains( "my-app 99.0" ) );
        assertTrue( "Escaping did not work properly", content.contains( "will not be filtered ${application.name}." ) );
    }

    /**
     * Builds an EAR and filter the content of the sources directory using a custom filter file.
     * @throws Exception in case of an error.
     */
    public void testProject046()
        throws Exception
    {
        final File baseDir = doTestProject( "project-046", new String[] { "README.txt", "eartest-ejb-sample-one-1.0.jar" } );
        final File actualReadme = new File( getEarDirectory( baseDir, "project-046" ), "README.txt" );
        final String content = IOUtil.toString( ReaderFactory.newReader( actualReadme, "UTF-8" ) );
        assertTrue( "application name and version was not filtered properly", content.contains( "my-app 99.0" ) );
        assertTrue( "application build was not filtered properly", content.contains( "(Build 2)" ) );
        assertTrue( "Unknown property should not have been filtered",
                    content.contains( "will not be filtered ${application.unknown}." ) );
    }

    /**
     * Builds an EAR and filter the content with a list of extensions.
     * @throws Exception in case of an error.
     */
    public void testProject047()
        throws Exception
    {
        final File baseDir = doTestProject( "project-047", new String[] { "README.txt", "eartest-ejb-sample-one-1.0.jar" } );
        final File actualReadme = new File( getEarDirectory( baseDir, "project-047" ), "README.txt" );
        final String content = IOUtil.toString( ReaderFactory.newReader( actualReadme, "UTF-8" ) );
        assertTrue( "application name and version should not have been filtered", !content.contains( "my-app 99.0" ) );
        assertTrue( "original properties not found", content.contains( "${application.name} ${project.version}" ) );
    }

    /**
     * Builds an EAR with a Jboss 5 configuration containing library directory.
     * @throws Exception in case of an error.
     */
    public void testProject048()
        throws Exception
    {
        doTestProject( "project-048", new String[] { "eartest-ejb-sample-one-1.0.jar", "eartest-ejb-sample-two-1.0.jar" } );
    }

    /**
     * Builds an EAR with a Jboss 4.2 configuration containing a library directory.
     * @throws Exception in case of an error.
     */
    public void testProject049()
        throws Exception
    {
        doTestProject( "project-049", new String[] { "eartest-ejb-sample-one-1.0.jar", "eartest-ejb-sample-two-1.0.jar" } );
    }

    /**
     * Builds an EAR with a Jboss 5 configuration containing a loader repository configuration definition.
     * @throws Exception in case of an error.
     */
    public void testProject050()
        throws Exception
    {
        doTestProject( "project-050", new String[] { "eartest-ejb-sample-one-1.0.jar", "eartest-ejb-sample-two-1.0.jar" } );
    }

    /**
     * Builds an EAR with a Jboss 5 configuration containing a loader repository class definition.
     * @throws Exception in case of an error.
     */
    public void testProject051()
        throws Exception
    {
        doTestProject( "project-051", new String[] { "eartest-ejb-sample-one-1.0.jar", "eartest-ejb-sample-two-1.0.jar" } );
    }

    /**
     * Builds an EAR with a Jboss 5 configuration containing a configuration parser class definition.
     * @throws Exception in case of an error.
     */
    public void testProject052()
        throws Exception
    {
        doTestProject( "project-052", new String[] { "eartest-ejb-sample-one-1.0.jar", "eartest-ejb-sample-two-1.0.jar" } );
    }

    /**
     * Builds an EAR with a Jboss 5 configuration containing only the loader repo configuration
     * @throws Exception in case of an error.
     */
    public void testProject053()
        throws Exception
    {
        doTestProject( "project-053", new String[] { "eartest-ejb-sample-one-1.0.jar", "eartest-ejb-sample-two-1.0.jar" } );
    }

    /**
     * Builds an EAR with deployment descriptor configuration for Java EE 5 and no application.xml
     * @throws Exception in case of an error.
     */
    public void testProject054()
        throws Exception
    {
        doTestProject( "project-054", new String[] { "eartest-ejb-sample-one-1.0.jar", "eartest-ejb-sample-two-1.0.jar" } );
    }

    /**
     * Builds an EAR with jar dependencies added in application.xml.
     * @throws Exception in case of an error.
     */
    public void testProject055()
        throws Exception
    {
        doTestProject( "project-055", new String[] { "eartest-jar-sample-one-1.0.jar", "eartest-jar-sample-two-1.0.jar",
            "eartest-jar-sample-three-with-deps-1.0.jar" } );
    }

    /**
     * Builds an EAR with deployment descriptor configuration for J2EE 1.4 and an alternative deployment descriptor.
     * @throws Exception in case of an error.
     */
    public void testProject056()
        throws Exception
    {
        doTestProject( "project-056", new String[] { "eartest-ejb-sample-one-1.0.jar" } );
    }

    /**
     * Builds an EAR with a complete JBoss 4.2 configuration and validate it matches the DTD (MEAR-104).
     * @throws Exception in case of an error.
     */
    public void testProject057()
        throws Exception
    {
        doTestProject( "project-057", new String[] { "eartest-ejb-sample-one-1.0.jar", "eartest-ejb-sample-two-1.0.jar" } );
    }

    /**
     * Builds an EAR with deployment descriptor configuration for Java EE 6.
     * @throws Exception in case of an error.
     */
    public void testProject058()
        throws Exception
    {
        doTestProject( "project-058", new String[] { "eartest-ejb-sample-one-1.0.jar" } );
    }

    /**
     * Builds an EAR with no display name entry at all.
     * @throws Exception in case of an error.
     */
    public void testProject059()
        throws Exception
    {
        doTestProject( "project-059", new String[] { "eartest-ejb-sample-one-1.0.jar" } );
    }

    /**
     * Builds an EAR with ejb-client packaged for J2EE 1.3 (MEAR-85)
     *
     * @throws Exception in case of an error.
     */
    public void testProject060()
        throws Exception
    {
        doTestProject( "project-060", new String[] { "eartest-ejb-sample-one-1.0.jar", "eartest-ejb-sample-two-1.0-client.jar" } );
    }

    /**
     * Builds an EAR with ejb-client packaged for J2EE 1.4 (MEAR-85)
     *
     * @throws Exception in case of an error.
     */
    public void testProject061()
        throws Exception
    {
        doTestProject( "project-061", new String[] { "eartest-ejb-sample-one-1.0.jar", "eartest-ejb-sample-two-1.0-client.jar" } );
    }

    /**
     * Builds an EAR with ejb-client packaged for JavaEE 5 (MEAR-85)
     *
     * @throws Exception in case of an error.
     */
    public void testProject062()
        throws Exception
    {
        doTestProject( "project-062", new String[] { "eartest-ejb-sample-one-1.0.jar", "lib/eartest-ejb-sample-two-1.0-client.jar" } );
    }

    /**
     * Builds an EAR with ejb-client packaged for JavaEE 6 (MEAR-85)
     *
     * @throws Exception in case of an error.
     */
    public void testProject063()
        throws Exception
    {
        doTestProject( "project-063", new String[] { "lib/eartest-ejb-sample-two-1.0-client.jar" } );
    }

    /**
     * Builds an EAR with ejb-client packaged for JavaEE 5 and still put it in the root (MEAR-85)
     *
     * @throws Exception in case of an error.
     */
    public void testProject064()
        throws Exception
    {
        doTestProject( "project-064", new String[] { "eartest-ejb-sample-one-1.0.jar", "eartest-ejb-sample-two-1.0-client.jar" } );
    }

    /**
     * Builds an EAR with a custom moduleId.
     * @throws Exception in case of an error.
     */
    public void testProject065()
        throws Exception
    {
        doTestProject( "project-065", new String[] { "eartest-ejb-sample-one-1.0.jar", "eartest-ejb-sample-two-1.0.jar" } );
    }

    /**
     * Builds an EAR with generateModuleId enabled.
     * @throws Exception in case of an error.
     */
    public void testProject066()
        throws Exception
    {
        doTestProject( "project-066", new String[] { "eartest-ejb-sample-one-1.0.jar", "eartest-ejb-sample-two-1.0.jar" } );
    }

    /**
     * Builds an EAR with generateModuleId enabled and a custom module.
     * @throws Exception in case of an error.
     */
    public void testProject067()
        throws Exception
    {
        doTestProject( "project-067", new String[] { "eartest-ejb-sample-one-1.0.jar", "eartest-ejb-sample-two-1.0.jar" } );
    }

    /**
     * Builds an EAR with the no-version file name mapping.
     * @throws Exception in case of an error.
     */
    public void testProject068()
        throws Exception
    {
        doTestProject( "project-068", new String[] { "eartest-ejb-sample-one.jar", "eartest-ejb-sample-two.jar" } );
    }

    /**
     * Builds an EAR with a custom library-directory and JavaEE 6.
     * @throws Exception in case of an error.
     */
    public void testProject069()
        throws Exception
    {
        doTestProject( "project-069", new String[] { "eartest-ejb-sample-one-1.0.jar", "myLibs/eartest-jar-sample-one-1.0.jar" } );
    }

    /**
     * Builds an EAR with application-name and initialize-in-order tags.
     * @throws Exception in case of an error.
     */
    public void testProject070()
        throws Exception
    {
        doTestProject( "project-070", new String[] { "eartest-ejb-sample-one-1.0.jar", "eartest-jar-sample-one-1.0.jar" } );
    }

    /**
     * Builds an EAR with application-name and initialize-in-order tags for unsupported version.
     * @throws Exception in case of an error.
     */
    public void testProject071()
        throws Exception
    {
        doTestProject( "project-071", new String[] { "eartest-ejb-sample-one-1.0.jar", "eartest-jar-sample-one-1.0.jar" } );
    }

    /**
     * Builds an EAR with an application client module (app-client).
     * @throws Exception in case of an error.
     */
    public void testProject072()
        throws Exception
    {
        doTestProject( "project-072", new String[] { "eartest-ejb-sample-one-1.0.jar", "eartest-app-client-sample-one-1.0.jar" } );
    }

    /**
     * Builds an EAR with an application client module (app-client) and a default bundle directory for _java_ modules.
     * @throws Exception in case of an error.
     */
    public void testProject073()
        throws Exception
    {
        doTestProject( "project-073", new String[] { "eartest-ejb-sample-one-1.0.jar", "eartest-app-client-sample-one-1.0.jar",
            "APP-INF/lib/eartest-jar-sample-one-1.0.jar", "APP-INF/lib/eartest-jar-sample-two-1.0.jar" } );
    }

    /**
     * Builds an EAR with custom env entries settings and J2EE 1.3. Not supported by the specification so this should be
     * ignored.
     * @throws Exception in case of an error.
     */
    public void testProject074()
        throws Exception
    {
        doTestProject( "project-074", new String[] { "eartest-ejb-sample-one-1.0.jar", "eartest-ejb-sample-two-1.0.jar" } );
    }

    /**
     * Builds an EAR with custom env entries settings and J2EE 1.4. Not supported by the specification so this should be
     * ignored.
     * @throws Exception in case of an error.
     */
    public void testProject075()
        throws Exception
    {
        doTestProject( "project-075", new String[] { "eartest-ejb-sample-one-1.0.jar", "eartest-ejb-sample-two-1.0.jar" } );
    }

    /**
     * Builds an EAR with custom env entries settings and JavaEE 5. Not supported by the specification so this should be
     * ignored.
     * @throws Exception in case of an error.
     */
    public void testProject076()
        throws Exception
    {
        doTestProject( "project-076", new String[] { "eartest-ejb-sample-one-1.0.jar", "eartest-ejb-sample-two-1.0.jar" } );
    }

    /**
     * Builds an EAR with custom env entries settings and JavaEE 6.
     * @throws Exception in case of an error.
     */
    public void testProject077()
        throws Exception
    {
        doTestProject( "project-077", new String[] { "eartest-ejb-sample-one-1.0.jar", "eartest-ejb-sample-two-1.0.jar" } );
    }

    /**
     * Builds an EAR with the no version for ejb file name mapping.
     * @throws Exception in case of an error.
     */
    public void testProject078()
        throws Exception
    {
        doTestProject( "project-078",
                       new String[] { "ejb-sample-one.jar", "war-sample-one.war", "jar-sample-two.jar" } );
    }

    /**
     * Builds an EAR with the 'default' library directory mode. Uses the value of the defaultLibBundleDir.
     * @throws Exception in case of an error.
     */
    public void testProject079()
        throws Exception
    {
        doTestProject( "project-079", new String[] { "eartest-ejb-sample-one-1.0.jar", "myLibs/eartest-jar-sample-one-1.0.jar" } );
    }

    /**
     * Builds an EAR with the 'empty' library directory mode. Generate an empty library-directory element.
     * @throws Exception in case of an error.
     */
    public void testProject080()
        throws Exception
    {
        doTestProject( "project-080", new String[] { "eartest-ejb-sample-one-1.0.jar", "myLibs/eartest-jar-sample-one-1.0.jar" } );
    }

    /**
     * Builds an EAR with the 'none' library directory mode. Does not generate an library-directory element.
     * @throws Exception in case of an error.
     */
    public void testProject081()
        throws Exception
    {
        doTestProject( "project-081", new String[] { "eartest-ejb-sample-one-1.0.jar", "myLibs/eartest-jar-sample-one-1.0.jar" } );
    }

    /**
     * Builds an EAR with deployment descriptor configuration for JavaEE 7.
     * @throws Exception in case of an error.
     */
    public void testProject082()
        throws Exception
    {
        doTestProject( "project-082", new String[] { "eartest-ejb-sample-one-1.0.jar" } );
    }

    /**
     * Builds an EAR with a library directory and custom env entries. The library-directory element must come first
     * (MEAR-158).
     * @throws Exception in case of an error.
     */
    public void testProject083()
        throws Exception
    {
        doTestProject( "project-083", new String[] { "eartest-ejb-sample-one-1.0.jar", "eartest-ejb-sample-two-1.0.jar" } );
    }

    /**
     * Support of an application id (MEAR-174).
     * @throws Exception in case of an error.
     */
    public void testProject084()
        throws Exception
    {
        doTestProject( "project-084", new String[] { "eartest-ejb-sample-one-1.0.jar" } );
    }

    /**
     * Builds an EAR with custom ejbRef entries settings and JavaEE 6.
     * @throws Exception in case of an error.
     */
    public void testProject085()
        throws Exception
    {
        doTestProject( "project-085", new String[] { "eartest-ejb-sample-one-1.0.jar", "eartest-ejb-sample-two-1.0.jar" } );
    }

    /**
     * Builds an EAR with custom ejbRef entries plus lookup-name entry.
     * @throws Exception in case of an error.
     */
    public void testProject086()
        throws Exception
    {
        doTestProject( "project-086", new String[] { "eartest-ejb-sample-one-1.0.jar", "eartest-ejb-sample-two-1.0.jar" } );
    }

    /**
     * Builds an EAR with resource-ref entries.
     * @throws Exception in case of an error.
     */
    public void testProject087()
        throws Exception
    {
        doTestProject( "project-087", new String[] { "eartest-ejb-sample-one-1.0.jar", "eartest-ejb-sample-two-1.0.jar" } );
    }
}
