/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements. See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache license, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the license for the specific language governing permissions and
 * limitations under the license.
 */

package org.apache.logging.log4j.util;

import org.junit.jupiter.api.Test;

import java.util.Arrays;
import java.util.Collections;
import java.util.Iterator;

import static org.junit.jupiter.api.Assertions.*;

public class StringsTest {

    @Test
    public void testIsEmpty() {
        assertTrue(Strings.isEmpty(null));
        assertTrue(Strings.isEmpty(""));
        assertFalse(Strings.isEmpty(" "));
        assertFalse(Strings.isEmpty("a"));
    }

    @Test
    public void testIsBlank() {
        assertTrue(Strings.isBlank(null));
        assertTrue(Strings.isBlank(""));
        assertTrue(Strings.isBlank(" "));
        assertTrue(Strings.isBlank("\n"));
        assertTrue(Strings.isBlank("\r"));
        assertTrue(Strings.isBlank("\t"));
        assertFalse(Strings.isEmpty("a"));
    }

    /**
     * A sanity test to make sure a typo does not mess up {@link Strings#EMPTY}.
     */
    @Test
    public void testEMPTY() {
        assertEquals("", Strings.EMPTY);
        assertEquals(0, Strings.EMPTY.length());
    }

    @Test
    public void testJoin() {
        assertNull(Strings.join((Iterable<?>) null, '.'));
        assertNull(Strings.join((Iterator<?>) null, '.'));
        assertEquals("", Strings.join((Collections.emptyList()), '.'));

        assertEquals("a", Strings.join(Collections.singletonList("a"), '.'));
        assertEquals("a.b", Strings.join(Arrays.asList("a", "b"), '.'));
        assertEquals("a.b.c", Strings.join(Arrays.asList("a", "b", "c"), '.'));

        assertEquals("", Strings.join(Collections.singletonList((String) null), ':'));
        assertEquals(":", Strings.join(Arrays.asList(null, null), ':'));
        assertEquals("a:", Strings.join(Arrays.asList("a", null), ':'));
        assertEquals(":b", Strings.join(Arrays.asList(null, "b"), ':'));
    }

    @Test
    public void testQuote() {
        assertEquals("'Q'", Strings.quote("Q"));
    }

}
