/**
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package org.apache.lens.ml.algo.spark.dt;

import java.util.Map;

import org.apache.lens.ml.algo.api.AlgoParam;
import org.apache.lens.ml.algo.api.Algorithm;
import org.apache.lens.ml.algo.spark.BaseSparkAlgo;
import org.apache.lens.ml.algo.spark.BaseSparkClassificationModel;
import org.apache.lens.server.api.error.LensException;

import org.apache.spark.mllib.regression.LabeledPoint;
import org.apache.spark.mllib.tree.DecisionTree$;
import org.apache.spark.mllib.tree.configuration.Algo$;
import org.apache.spark.mllib.tree.impurity.Entropy$;
import org.apache.spark.mllib.tree.impurity.Gini$;
import org.apache.spark.mllib.tree.impurity.Impurity;
import org.apache.spark.mllib.tree.impurity.Variance$;
import org.apache.spark.mllib.tree.model.DecisionTreeModel;
import org.apache.spark.rdd.RDD;

import scala.Enumeration;

/**
 * The Class DecisionTreeAlgo.
 */
@Algorithm(name = "spark_decision_tree", description = "Spark Decision Tree classifier algo")
public class DecisionTreeAlgo extends BaseSparkAlgo {

  /** The algo. */
  @AlgoParam(name = "algo", help = "Decision tree algorithm. Allowed values are 'classification' and 'regression'")
  private Enumeration.Value algo;

  /** The decision tree impurity. */
  @AlgoParam(name = "impurity", help = "Impurity measure used by the decision tree. "
    + "Allowed values are 'gini', 'entropy' and 'variance'")
  private Impurity decisionTreeImpurity;

  /** The max depth. */
  @AlgoParam(name = "maxDepth", help = "Max depth of the decision tree. Integer values expected.",
    defaultValue = "100")
  private int maxDepth;

  /**
   * Instantiates a new decision tree algo.
   *
   * @param name        the name
   * @param description the description
   */
  public DecisionTreeAlgo(String name, String description) {
    super(name, description);
  }

  /*
   * (non-Javadoc)
   *
   * @see org.apache.lens.ml.spark.algos.BaseSparkAlgo#parseAlgoParams(java.util.Map)
   */
  @Override
  public void parseAlgoParams(Map<String, String> params) {
    String dtreeAlgoName = params.get("algo");
    if ("classification".equalsIgnoreCase(dtreeAlgoName)) {
      algo = Algo$.MODULE$.Classification();
    } else if ("regression".equalsIgnoreCase(dtreeAlgoName)) {
      algo = Algo$.MODULE$.Regression();
    }

    String impurity = params.get("impurity");
    if ("gini".equals(impurity)) {
      decisionTreeImpurity = Gini$.MODULE$;
    } else if ("entropy".equals(impurity)) {
      decisionTreeImpurity = Entropy$.MODULE$;
    } else if ("variance".equals(impurity)) {
      decisionTreeImpurity = Variance$.MODULE$;
    }

    maxDepth = getParamValue("maxDepth", 100);
  }

  /*
   * (non-Javadoc)
   *
   * @see org.apache.lens.ml.spark.algos.BaseSparkAlgo#trainInternal(java.lang.String, org.apache.spark.rdd.RDD)
   */
  @Override
  protected BaseSparkClassificationModel trainInternal(String modelId, RDD<LabeledPoint> trainingRDD)
    throws LensException {
    DecisionTreeModel model = DecisionTree$.MODULE$.train(trainingRDD, algo, decisionTreeImpurity, maxDepth);
    return new DecisionTreeClassificationModel(modelId, new SparkDecisionTreeModel(model));
  }
}
